/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.view.presentation

import androidx.annotation.NonNull
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleObserver
import androidx.lifecycle.OnLifecycleEvent
import universum.studios.android.arkhitekton.ArkhitektonLogging
import universum.studios.android.arkhitekton.view.View
import universum.studios.android.arkhitekton.view.ViewModel
import java.util.concurrent.atomic.AtomicBoolean

/**
 * A [Presenter] base implementation which is recommended to be inherited by concrete presenter
 * implementations in order to take advantage of already implemented stubs of [Presenter] interface
 * and also to be 'resistant' against future changes in that interface.
 *
 * Inheritance hierarchies should provide their specific builder which implements [BasePresenter.BaseBuilder]
 * for convenience.
 *
 * @param V  Type of the view that may be attached to the presenter a then used by the presenter for
 * example to perform navigation actions.
 * @param VM Type of the view model of which state may the presenter update according to responses
 * for interaction events.
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @constructor Creates a new instance of BasePresenter with the given `builder's` configuration.
 * @param builder The builder with configuration for the new presenter.
 */
@Suppress("MemberVisibilityCanBePrivate")
abstract class BasePresenter<V : View<VM>, VM : ViewModel> protected constructor(@NonNull builder: BaseBuilder<*, V, VM>) : Presenter<V> {

    /*
	 * Companion ===================================================================================
	 */

    /**
     */
    companion object {

        /**
         * Log TAG.
         */
        // internal const val TAG = "BasePresenter"
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /**
     * View model instance associated with this presenter.
     */
    private val viewModel: VM = builder.viewModel

    /**
     * View instance attached to this presenter (if any).
     *
     * @see [attachView]
     * @see [detachView]
     */
    private var view: V? = null

    /**
     * Registry with lifecycle actions registered via [registerViewLifecycleAction].
     */
    @Suppress("LeakingThis")
    private val lifecycleActionRegistry by lazy { LifecycleActionRegistry(this) }

    /**
     * Boolean flag indicating whether this presenter instance is already destroyed or not.
     *
     * @see [onDestroyed]
     */
    private val destroyed = AtomicBoolean(false)

    /*
     * Initialization ==============================================================================
     */

    /*
     * Functions ===================================================================================
     */

    /**
     * Returns the name of this presenter.
     *
     * @return This presenter's name.
     */
    @NonNull fun name(): String = javaClass.simpleName

    /**
     * Returns the view model associated with this presenter.
     *
     * @return View model which this presenter uses to persist view's state.
     */
    protected fun getViewModel(): VM = viewModel

    /*
     */
    override fun attachView(view: V) {
        check(this.view == null) { "Already has view attached!" }
        this.view = view
        if (this is LifecycleObserver) {
            view.lifecycle.addObserver(lifecycleActionRegistry)
        }
        onViewAttached(view)
    }

    /**
     * Invoked whenever the specified [view] has been just attached to this presenter.
     *
     * At this point, the view is already attached to this presenter, that is, [isViewAttached]
     * returns `true`.
     *
     * @param view The attached view.
     *
     * @see attachView
     * @see onViewDetached
     */
    protected open fun onViewAttached(view: V) {
        // Inheritance hierarchies may for example register here as lifecycle observers on the attached view.
    }

    /**
     * Checks whether this presenter has view attached at this time or not.
     *
     * @return `True` if there is view attached, `false` otherwise.
     */
    protected fun isViewAttached(): Boolean = view != null

    /**
     * Asserts that this presenter has view attached at this time. If not [IllegalStateException] is thrown.
     */
    protected fun assertViewAttached() {
        this.view ?: throw IllegalStateException("No view attached!")
    }

    /**
     * Returns the current lifecycle state of the view attached to this presenter.
     *
     * @return Current lifecycle state.
     *
     * @see View.getLifecycle
     * @see Lifecycle.getCurrentState
     */
    protected fun getViewLifecycleCurrentState(): Lifecycle.State = getView().lifecycle.currentState

    /**
     * Performs the specified [action] immediately if the following conditions are met:
     * - view is **attached**,
     * - if the action has specified **state condition** and the current lifecycle state **is at least**
     *   that state
     *
     * otherwise the action is registered for lifecycle event corresponding with that state condition
     * (if any) via [registerViewLifecycleAction].
     *
     * @param action The desired action to be performed or registered until the desired lifecycle
     * event occurs.
     * @return `True` if action was performed/run immediately, `false` if it was registered for later.
     */
    protected fun performViewLifecycleAction(action: LifecycleAction): Boolean {
        return if (isViewAttached() && getViewLifecycleCurrentState().isAtLeast(action.stateCondition)) {
            action.run()
            true
        } else {
            registerViewLifecycleAction(
                    LifecycleActionRegistry.resolveEventForState(action.stateCondition),
                    action
            )
            false
        }
    }

    /**
     * Registers the specified lifecycle [action] to be run when lifecycle callback is received for
     * the specified lifecycle [event].
     *
     * @param event The lifecycle event for which to register the desired action. If this event occurs,
     * the action will be run.
     * @param action The desired lifecycle action to run.
     * @return Subscription associated with the registered action.
     */
    protected fun registerViewLifecycleAction(event: Lifecycle.Event, action: LifecycleAction): LifecycleActionSubscription {
        return lifecycleActionRegistry.registerActionForEvent(event, action)
    }

    /**
     * Returns the view that is currently attached to this presenter.
     *
     * @return View attached.
     * @throws IllegalStateException If there is no view attached to this presenter at this time.
     *
     * @see isViewAttached
     * @see attachView
     * @see detachView
     */
    protected fun getView(): V {
        assertViewAttached()
        return view!!
    }

    /*
     */
    override fun detachView(view: V) {
        if (this.view == null) {
            return
        }
        require(!(this.view !== view)) { "Cannot detach current view. The view passed to detachView(...) is of different reference." }
        this.view = null
        if (this is LifecycleObserver) {
            view.lifecycle.removeObserver(lifecycleActionRegistry)
        }
        onViewDetached(view)
    }

    /**
     * Invoked whenever the specified [view] has been just detached from this presenter.
     *
     * At this point, the view is no longer attached to this presenter, that is, [isViewAttached]
     * returns `false`.
     *
     * @param view The detached view.
     *
     * @see detachView
     * @see onViewAttached
     */
    protected open fun onViewDetached(view: V) {
        // Inheritance hierarchies should for example unregister here as lifecycle observers on the detached view.
    }

    /*
     */
    override fun destroy() {
        if (destroyed.compareAndSet(false, true)) {
            this.lifecycleActionRegistry.clearAllActions()
            onDestroyed()
        }
    }

    /**
     * Invoked when this presenter instance is being destroyed due to call to [destroy].
     *
     * At this point are all lifecycle actions previously registered via [registerViewLifecycleAction]
     * cleared.
     */
    protected open fun onDestroyed() {
        // Inheritance hierarchies may for example clear here resources that will be no longer used.
    }

    /**
     * Checks whether this presenter instance is destroyed.
     *
     * @return `True` if this presenter has been already destroyed, `false` otherwise.
     *
     * @see destroy
     */
    protected fun isDestroyed(): Boolean = destroyed.get()

    /*
     */
    @NonNull override fun toString(): String = "${name()}(viewAttached: ${view != null}, destroyed: $destroyed)"

    /*
     * Inner classes ===============================================================================
     */

    /**
     * Base implementation of [Presenter.Builder] which should be used by [BasePresenter] inheritance
     * hierarchies.
     *
     * @param B Type of the builder used as return type for builder's chain-able methods.
     * @param V Type of the view of which instance may be associated with new presenter.
     * @param VM Type of the view model of which instance may be associated with new presenter.
     *
     * @author Martin Albedinsky
     * @since 1.0
     *
     * @constructor Creates a new instance of BaseBuilder with empty state.
     */
    abstract class BaseBuilder<B : BaseBuilder<B, V, VM>, V : View<VM>, VM : ViewModel> protected constructor() : Presenter.Builder<Presenter<V>> {

        /**
         * Obtains reference to this builder instance which is used for purpose of methods chaining.
         *
         * @return This builder instance.
         */
        protected abstract val self: B

        /**
         * See [BaseBuilder.viewModel].
         */
        lateinit var viewModel: VM

        /**
         * Specifies a view model to be associated with the new presenter.
         *
         * @param viewModel The desired view model for the presenter.
         * @return This builder to allow methods chaining.
         */
        fun viewModel(viewModel: VM): B {
            this.viewModel = viewModel
            return self
        }
    }

    /**
     * A [BaseBuilder] simple implementation which may be used in cases when a concrete implementation
     * of [BasePresenter] does not require any additional dependencies except its view model.
     * Such presenters may be instantiated simply by passing instance of this builder created via
     * [SimpleBuilder.create] to such presenters's constructor.
     *
     * @param V Type of the view of which instance may be associated with new presenter.
     * @param VM Type of the view model of which instance may be associated with new presenter.
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    class SimpleBuilder<V : View<VM>, VM : ViewModel> private constructor() : BaseBuilder<SimpleBuilder<V, VM>, V, VM>() {

        /**
         */
        companion object {

            /**
             * Creates a new SimpleBuilder with the specified [viewModel].
             *
             * @param viewModel The desired view model for the new presenter.
             * @param V Type of the view of which instance may be associated with the new presenter.
             * @param VM Type of the view model of which instance may be associated with the new presenter.
             * @return Builder instance ready to be passed to the presenter's constructor.
             */
            @JvmStatic fun <V : View<VM>, VM : ViewModel> create(viewModel: VM) = SimpleBuilder<V, VM>().apply {this.viewModel = viewModel }
        }

        /*
         */
        override val self = this

        /*
         */
        override fun build(): Presenter<V> = throw UnsupportedOperationException("Pass instance of SimpleBuilder to the appropriate constructor!")
    }

    /**
     * Runnable action that may be registered for a specific [Lifecycle.Event] via [BasePresenter.registerViewLifecycleAction].
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    abstract class LifecycleAction : Runnable {

        /**
         * State conditioning execution of this action.
         */
        internal var stateCondition = Lifecycle.State.INITIALIZED

        /**
         * Specifies a state that should condition execution of this action.
         *
         * This action will be executed/run only if lifecycle's state is at least the specified one.
         *
         * @param state The desired state to be used as condition.
         * @return This action to allow methods chaining.
         */
        fun withStateCondition(state: Lifecycle.State): LifecycleAction {
            this.stateCondition = state
            return this
        }
    }

    /**
     * Subscription that may be used to cancel [LifecycleAction] registered via [BasePresenter.registerViewLifecycleAction].
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    interface LifecycleActionSubscription {

        /**
         * Cancels this lifecycle action subscription if it is not canceled yet.
         *
         * When canceled, the associated lifecycle action is removed from registered actions so it
         * may no longer be run when the corresponding lifecycle event occurs.
         */
        fun cancel()
    }

    /**
     * [LifecycleActionSubscription] simple implementation.
     *
     * @constructor Creates a new instance of LifecycleActionSubscriptionImpl with the specified parameters.
     * @param actions List of actions where is the specified [action] registered.
     * @param action The desired action for which is the new subscription being created.
     */
    private class LifecycleActionSubscriptionImpl(
            /**
             * List of actions where is the given [action] registered.
             */
            private val actions: MutableList<LifecycleAction>,
            /**
             * The registered action for which is this subscription created. When this subscription
             * is canceled via [cancel][LifecycleActionSubscriptionImpl.cancel] this action will be
             * removed from the registered [actions] so it may no longer be run.
             */
            private val action: LifecycleAction) : LifecycleActionSubscription {

        /**
         * Boolean flag indicating whether this subscription is already canceled or not.
         */
        private val canceled = AtomicBoolean()

        /*
         */
        override fun cancel() {
            if (canceled.compareAndSet(false, true)) {
                this.actions.remove(action)
            }
        }
    }

    /**
     * Registry for [LifecycleActions][LifecycleAction] registered for a specific [Lifecycle.Event].
     * This registry also acts as [LifecycleObserver] in order to receive lifecycle event related
     * callbacks so the actions may be run according to their associated lifecycle event.
     */
    @Suppress("unused")
    internal class LifecycleActionRegistry(private val parent: BasePresenter<*, *>) : LifecycleObserver {

        /**
         */
        companion object {

            /**
             * Resolves proper lifecycle event for the specified lifecycle [state].
             *
             * @param state The desired state for which to resolve its associated event.
             * @return Resolved lifecycle event.
             */
            fun resolveEventForState(state: Lifecycle.State): Lifecycle.Event {
                return when (state) {
                    Lifecycle.State.INITIALIZED -> Lifecycle.Event.ON_ANY
                    Lifecycle.State.CREATED -> Lifecycle.Event.ON_CREATE
                    Lifecycle.State.STARTED -> Lifecycle.Event.ON_START
                    Lifecycle.State.RESUMED -> Lifecycle.Event.ON_RESUME
                    Lifecycle.State.DESTROYED -> Lifecycle.Event.ON_DESTROY
                    else -> Lifecycle.Event.ON_ANY
                }
            }
        }

        /**
         * List of registered actions for [ON_DESTROY][Lifecycle.Event.ON_CREATE] event.
         */
        private val createActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * List of registered actions for [ON_DESTROY][Lifecycle.Event.ON_START] event.
         */
        private val startActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * List of registered actions for [ON_DESTROY][Lifecycle.Event.ON_RESUME] event.
         */
        private val resumeActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * List of registered actions for [ON_DESTROY][Lifecycle.Event.ON_PAUSE] event.
         */
        private val pauseActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * List of registered actions for [ON_DESTROY][Lifecycle.Event.ON_STOP] event.
         */
        private val stopActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * List of registered actions for [ON_DESTROY][Lifecycle.Event.ON_DESTROY] event.
         */
        private val destroyActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * Registers the given [action] for the specified [event]. The registered
         * action will be run when this registry receives callback for the associated lifecycle event
         * from the lifecycle pipeline.
         *
         * @param event The event for which to register the desired action.
         * @param action The desired action to be run when the specified lifecycle event occurs.
         * @return Subscription for the registered action which may be canceled if desired.
         * @throws IllegalArgumentException If the event is [ON_ANY][Lifecycle.Event.ON_ANY].
         */
        fun registerActionForEvent(event: Lifecycle.Event, action: LifecycleAction): LifecycleActionSubscription {
            val actions = when (event) {
                Lifecycle.Event.ON_CREATE -> this.createActions.apply { add(action) }
                Lifecycle.Event.ON_START -> this.startActions.apply { add(action) }
                Lifecycle.Event.ON_RESUME -> this.resumeActions.apply { add(action) }
                Lifecycle.Event.ON_PAUSE -> this.pauseActions.apply { add(action) }
                Lifecycle.Event.ON_STOP -> this.stopActions.apply { add(action) }
                Lifecycle.Event.ON_DESTROY -> this.destroyActions.apply { add(action) }
                Lifecycle.Event.ON_ANY -> throw IllegalArgumentException("Cannot register lifecycle action for event($event)!")
            }
            ArkhitektonLogging.d(parent.name(), "Registered lifecycle action for event($event).")
            return LifecycleActionSubscriptionImpl(actions, action)
        }

        /**
         * Invoked by the lifecycle pipeline whenever view becomes **created**.
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_CREATE)
        fun onViewCreated() = runActionsForViewState("CREATED", createActions)

        /**
         * Invoked by the lifecycle pipeline whenever view becomes **started**.
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_START)
        fun onViewStarted() = runActionsForViewState("STARTED", startActions)

        /**
         * Invoked by the lifecycle pipeline whenever view becomes **resumed**.
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_RESUME)
        fun onViewResumed() = runActionsForViewState("RESUMED", resumeActions)

        /**
         * Invoked by the lifecycle pipeline whenever view becomes **paused**.
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_PAUSE)
        fun onViewPaused() = runActionsForViewState("PAUSED", pauseActions)

        /**
         * Invoked by the lifecycle pipeline whenever view becomes **stopped**.
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
        fun onViewStop() = runActionsForViewState("STOPPED", stopActions)

        /**
         * Invoked by the lifecycle pipeline whenever view becomes **destroyed**.
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
        fun onViewDestroyed() = runActionsForViewState("DESTROYED", destroyActions)

        /**
         * Runs all specified [actions]. When run, the list of actions will be cleared.
         *
         * @param viewState Name of the current view state for which the actions will be run.
         * @param actions List of actions to be run.
         */
        private fun runActionsForViewState(viewState: String, actions: MutableList<LifecycleAction>) {
            ArkhitektonLogging.d(parent.name(), createLogMessageForLifecycleActions(viewState, actions.size))
            if (actions.isNotEmpty()) {
                actions.forEach { it.run() }
                actions.clear()
            }
        }

        /**
         * Creates a log message informing about whether there are some registered actions to be run
         * or not for the specified parameters.
         *
         * @param viewState Name of the current view state.
         * @param actionsCount Count of actions to be run.
         * @return Log message ready to be printed.
         */
        private fun createLogMessageForLifecycleActions(viewState: String, actionsCount: Int): String {
            return "View $viewState. ${if (actionsCount == 0) "No registered actions to run." else "Running registered actions in count($actionsCount)."}"
        }

        /**
         * Clears all registered actions.
         */
        fun clearAllActions() {
            this.createActions.clear()
            this.startActions.clear()
            this.resumeActions.clear()
            this.pauseActions.clear()
            this.stopActions.clear()
            this.destroyActions.clear()
        }
    }
}