/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.control

import androidx.annotation.CallSuper
import androidx.annotation.NonNull
import androidx.annotation.VisibleForTesting
import io.reactivex.Scheduler
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.disposables.Disposable
import universum.studios.android.arkhitekton.data.DataSchedulers
import universum.studios.android.arkhitekton.interaction.InteractionSchedulers
import universum.studios.android.arkhitekton.interaction.Interactor
import universum.studios.android.arkhitekton.view.presentation.PresentationSchedulers
import universum.studios.android.arkhitekton.view.presentation.Presenter

/**
 * A [BaseController] implementation which provides a convenient api supporting **reactive approach**
 * using *Reactive Extensions* in order to implement controlling logic for a desired controller.
 *
 * Reactive controller implementation assumes usage of [Scheduler] for **interaction** and **presentation**
 * logic respectively. These schedulers may be specified via
 * [BaseBuilder.interactionScheduler][ReactiveController.BaseBuilder.interactionScheduler] and
 * [BaseBuilder.presentationScheduler][ReactiveController.BaseBuilder.presentationScheduler] and
 * later accessed directly within the controller implementation.
 *
 * @param I Type of the interactor to which may the controller pass interaction requests to be processed.
 * @param P Type of the presenter to which may the controller dispatch appropriate responses for
 * processed requests and/or other interaction events.
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @constructor Creates a new instance of ReactiveController with the given `builder's` configuration.
 * @param builder The builder with configuration for the new controller.
 */
abstract class ReactiveController<out I : Interactor, out P : Presenter<*>> protected constructor(builder: BaseBuilder<*, I, P>) : BaseController<I, P>(builder) {

    /*
	 * Companion ===================================================================================
	 */

    /**
     */
    companion object {

        /**
         * Log TAG.
         */
        // internal const val TAG = "ReactiveController"
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /**
     * Scheduler which should be used by this controller for loading of data for its associated presenter.
     */
    @NonNull protected val dataScheduler: Scheduler = builder.dataScheduler

    /**
     * Scheduler which should be used by this controller for processing of interaction requests.
     */
    @NonNull protected val interactionScheduler: Scheduler = builder.interactionScheduler

    /**
     * Scheduler which should by used by this controller for presentation of received responses as
     * results of interaction requests.
     */
    @NonNull protected val presentationScheduler: Scheduler = builder.presentationScheduler

    /**
     * Subscriptions that have been registered via [registerSubscription]. May be disposed,
     * each respectively, via [unregisterSubscription] or all via [clearSubscriptions].
     */
    private val subscriptions by lazy { CompositeDisposable() }

    /*
     * Initialization ==============================================================================
     */

    /*
     * Functions ===================================================================================
     */

    /**
     * Checks whether this controller has some subscriptions registered or not.
     *
     * @return `True` if there are some subscriptions registered, `false` otherwise.
     *
     * @see registerSubscription
     * @see unregisterSubscription
     */
    @VisibleForTesting fun hasSubscriptions(): Boolean = subscriptions.size() > 0

    /**
     * Registers a subscription for observation with possible long duration which starts right now.
     *
     * All registered subscriptions which are not manually unregistered are automatically unregistered
     * (disposed) when [onDeactivated] is invoked for this controller.
     *
     * @param subscription The desired subscription which may be later disposed via [unregisterSubscription].
     *
     * @see clearSubscriptions
     */
    protected fun registerSubscription(@NonNull subscription: Disposable): Boolean = subscriptions.add(subscription)

    /**
     * Un-registers the subscription previously registered via [registerSubscription].
     *
     * If the given [subscription] has not been registered, this method does nothing.
     *
     * @param subscription The desired subscription to be removed and also disposed.
     *
     * @see hasSubscriptions
     * @see clearSubscriptions
     */
    protected fun unregisterSubscription(@NonNull subscription: Disposable): Boolean = subscriptions.remove(subscription)

    /**
     * Clears all subscriptions registered by this controller. This will also dispose those subscriptions.
     *
     * @see hasSubscriptions
     */
    protected fun clearSubscriptions() = subscriptions.clear()

    /*
     */
    @CallSuper override fun onDeactivated() {
        super.onDeactivated()
        clearSubscriptions()
    }

    /*
     * Inner classes ===============================================================================
     */

    /**
     * Extended base implementation of [BaseController.BaseBuilder] which should be used by
     * implementations of [ReactiveController].
     *
     * @param B Type of the builder used as return type for builder's chain-able methods.
     * @param I Type of the interactor of which instance may be associated with new controller.
     * @param P Type of the presenter of which instance may be associated with new controller.
     *
     * @author Martin Albedinsky
     * @since 1.0
     *
     * @constructor Creates a new instance of BaseBuilder with default [interaction scheduler][InteractionSchedulers.primaryScheduler]
     * and default [presentation scheduler][PresentationSchedulers.primaryScheduler].
     */
    abstract class BaseBuilder<B : BaseBuilder<B, I, P>, I : Interactor, P : Presenter<*>> protected constructor() : BaseController.BaseBuilder<B, I, P>() {

        /**
         * See [ReactiveController.dataScheduler].
         */
        var dataScheduler: Scheduler = DataSchedulers.primaryScheduler()

        /**
         * See [ReactiveController.interactionScheduler].
         */
        var interactionScheduler: Scheduler = InteractionSchedulers.primaryScheduler()

        /**
         * See [ReactiveController.presentationScheduler].
         */
        var presentationScheduler: Scheduler = PresentationSchedulers.primaryScheduler()

        /**
         * Specifies a scheduler which should be used by the new controller for loading of data
         * for its associated presenter.
         *
         * Default value: **[DataSchedulers.primaryScheduler]**
         *
         * @param scheduler The desired scheduler to be used.
         * @return This builder to allow methods chaining.
         */
        fun dataScheduler(@NonNull scheduler: Scheduler): B {
            this.dataScheduler = scheduler
            return self
        }

        /**
         * Specifies a scheduler which should be used by the new controller for processing of
         * interaction requests.
         *
         * Default value: **[InteractionSchedulers.primaryScheduler]**
         *
         * @param scheduler The desired scheduler to be used.
         * @return This builder to allow methods chaining.
         */
        fun interactionScheduler(@NonNull scheduler: Scheduler): B {
            this.interactionScheduler = scheduler
            return self
        }

        /**
         * Specifies a scheduler which should by used by the new controller for presentation of
         * received responses as results of interaction requests.
         *
         * Default value: **[PresentationSchedulers.primaryScheduler]**
         *
         * @param scheduler The desired scheduler to be used.
         * @return This builder to allow methods chaining.
         */
        fun presentationScheduler(@NonNull scheduler: Scheduler): B {
            this.presentationScheduler = scheduler
            return self
        }
    }
}