/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.interaction

import androidx.annotation.NonNull
import universum.studios.android.arkhitekton.util.Error
import universum.studios.android.arkhitekton.util.Value

/**
 * Basic interface describing result which may be carried by [Response].
 *
 * @param T Type of the result value.
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @see Response.getValue
 */
interface Result<out T> {

    /*
	 * Companion ===================================================================================
	 */

    /**
     * Contract for [Result] element.
     */
    companion object Contract {

        /**
         * Empty instance with empty value and error.
         */
        @NonNull internal val EMPTY: Result<Value> = object : Result<Value> {

            /*
             */
            override fun isSuccess() = false

            /*
             */
            override fun getValue() = Value.EMPTY

            /*
             */
            override fun isFailure() = false

            /*
             */
            override fun getError() = Error.none()
        }

        /**
         * Returns an instance of empty result which has no data.
         *
         * @return Empty result ready to be attached to a desired response.
         */
        @Suppress("UNCHECKED_CAST")
        @NonNull fun <T> empty(): Result<T> = EMPTY as Result<T>
    }

    /*
     * Interface ===================================================================================
     */

    /*
     * Functions ===================================================================================
     */

    /**
     * Checks whether this result is a success or not.
     *
     * @return `True` if this result represents a success of which result value may be obtained via
     * [getValue], `false` if it is actually a failure.
     *
     * @see isFailure
     */
    fun isSuccess(): Boolean

    /**
     * Returns the value of this successful result.
     *
     * If this result is a failure, the returned value should be a simple `EMPTY` value.
     *
     * @return Value associated with this result.
     *
     * @see isSuccess
     */
    fun getValue(): T

    /**
     * Checks whether this result is a failure or not.
     *
     * @return `True` if this result represents a failure of which error may be obtained via [getError],
     * `false` if it is actually a success.
     *
     * @see isSuccess
     */
    fun isFailure(): Boolean

    /**
     * Returns the error due to which value of this result could not be produced.
     *
     * If this result is a successful one, the returned error should be a simple `NONE` error.
     *
     * @return Error associated with this result.
     *
     * @see isFailure
     */
    fun getError(): Error
}