/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.util

import android.support.annotation.NonNull

/**
 * Basic error which may be identified by its [code][code] and described via [description][description]
 * along with its [cause][cause].
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
@Suppress("MemberVisibilityCanBePrivate")
interface Error {

    /**
     * Contract for [Error] element.
     */
    companion object Contract {

        /**
         * Instance which may be used to indicate a 'none' error.
         */
        @NonNull internal val NONE = object : Error {

            /*
             */
            override fun code() = "NONE"

            /*
             */
            override fun description() = Description.EMPTY

            /*
             */
            override fun cause() = Cause.NONE

            /*
             */
            override fun toString() = code()
        }

        /**
         * Instance which may be used to indicate an 'unknown' error.
         */
        @NonNull internal val UNKNOWN = object : Error {

            /*
             */
            override fun code() = "UNKNOWN"

            /*
             */
            override fun description() = Description.EMPTY

            /*
             */
            override fun cause() = Cause.NONE

            /*
             */
            override fun toString() = code()
        }

        /**
         * Returns an instance of Error with 'NONE' code and [EMPTY][Description.EMPTY] description
         * and [NONE][Cause.NONE] cause.
         *
         * May be used as empty object to indicate that no error has occurred.
         *
         * @return Error ready to be associated with a desired failure.
         */
        @NonNull fun none(): Error = NONE

        /**
         * Returns an instance of Error with 'UNKNOWN' code and [EMPTY][Description.EMPTY] description
         * and [NONE][Cause.NONE] cause.
         *
         * May be used to indicate that an unknown error has occurred.
         *
         * @return Error ready to be associated with a desired failure.
         */
        @NonNull fun unknown(): Error = UNKNOWN

        /**
         * Convenience method for casting a throwable into [Error].
         *
         * It is safe to call this method only if the specified `throwable` is actually an Error.
         *
         * @param throwable The desired throwable to cast into error.
         * @return The given throwable as Error or an exception is thrown.
         * @throws AssertionError If the given throwable is not of [Error] type.
         */
        @NonNull fun asError(@NonNull throwable: Throwable): Error {
            return throwable as? Error ?: throw AssertionError("Specified throwable($throwable) is not an Error!")
        }

        /**
         * Unwraps an error from the specified `throwable`.
         *
         * If is safe to call this method only if the specified `throwable` is actually a [RuntimeException]
         * which has [Error] specified as its cause.
         *
         * @param throwable The desired throwable from which to unwrap the desired error.
         * @return Cause of the given throwable as Error or an exception is thrown.
         * @throws AssertionError If the given throwable is not a [RuntimeException], or it has no
         * cause specified or its cause is not of [Error] type.
         */
        @NonNull fun unwrap(@NonNull throwable: Throwable): Error {
            val cause = (throwable as? RuntimeException)?.cause
            cause ?: throw AssertionError("Specified throwable($throwable) is not a RuntimeException or it has no cause!")
            return asError(cause)
        }
    }

    /**
     * Returns the code of this error.
     *
     * @return This error's code.
     * @see description
     * @see cause
     */
    @NonNull fun code(): String

    /**
     * Returns the description specified for this error.
     *
     * @return This error's description.
     */
    @NonNull fun description(): Description

    /**
     * Returns the cause associated with this error.
     *
     * @return This error's cause.
     */
    @NonNull fun cause(): Throwable
}