/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.view.presentation

import android.support.annotation.NonNull
import android.support.annotation.VisibleForTesting
import io.reactivex.Scheduler
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.disposables.Disposable
import universum.studios.android.arkhitekton.data.DataSchedulers
import universum.studios.android.arkhitekton.view.View
import universum.studios.android.arkhitekton.view.ViewModel

/**
 * Class that todo:
 *
 * @param V  Type of the view todo:
 * @param VM Type of the view model todo:
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @constructor Creates a new instance of ReactivePresenter with the given `builder's` configuration.
 * @param builder The builder with configuration for the new presenter.
 */
abstract class ReactivePresenter<V : View<VM>, VM : ViewModel> protected constructor(builder: BaseBuilder<*, V, VM>) : BasePresenter<V, VM>(builder) {

    /*
	 * Companion ===================================================================================
	 */

    /**
     * todo:
     */
    companion object {

        /**
         * Log TAG.
         */
        // const internal val TAG = "ReactivePresenter"
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
	 * Members =====================================================================================
	 */

    /**
     * Scheduler which should be used by this presenter for presentation logic.
     */
    @NonNull protected val scheduler: Scheduler = builder.scheduler

    /**
     * Scheduler which should by used by this presenter for observation of data sources of which data
     * this presenter presents.
     */
    @NonNull protected val dataScheduler: Scheduler = builder.dataScheduler

    /**
     * Subscriptions that have been registered via [registerSubscription]. May be disposed,
     * each respectively, via [unregisterSubscription] or all via [clearSubscriptions].
     */
    private val subscriptions by lazy { CompositeDisposable() }

    /*
     * Initialization ==============================================================================
     */

    /*
	 * Functions ===================================================================================
	 */

    /**
     * Checks whether this presenter has some subscriptions registered or not.
     *
     * @return `True` if there are some subscriptions registered, `false` otherwise.
     *
     * @see registerSubscription
     * @see unregisterSubscription
     */
    @VisibleForTesting fun hasSubscriptions() = subscriptions.size() > 0

    /**
     * Registers a subscription for observation with possible long duration which starts right now.
     *
     * All registered subscriptions which are not manually unregistered are automatically cleared
     * (disposed) when [onDestroyed] is invoked for this presenter.
     *
     * @param subscription The desired subscription which may be later disposed via [unregisterSubscription].
     *
     * @see clearSubscriptions
     */
    protected fun registerSubscription(@NonNull subscription: Disposable) = subscriptions.add(subscription)

    /**
     * Un-registers the subscription previously registered via [registerSubscription].
     *
     * If the given `subscription` has not been registered, this method does nothing.
     *
     * @param subscription The desired subscription to be removed and also disposed.
     *
     * @see hasSubscriptions
     * @see clearSubscriptions
     */
    protected fun unregisterSubscription(@NonNull subscription: Disposable) = subscriptions.remove(subscription)

    /**
     * Clears all subscriptions registered by this presenter. This will also dispose those subscriptions.
     *
     * @see hasSubscriptions
     */
    protected fun clearSubscriptions() = subscriptions.clear()

    /*
     */
    override fun onDestroyed() {
        super.onDestroyed()
        clearSubscriptions()
    }

    /*
	 * Inner classes ===============================================================================
	 */

    /**
     * Extended base implementation of [BasePresenter.BaseBuilder] which should be used by
     * implementations of [ReactivePresenter].
     *
     * @param B Type of the builder used as return type for builder's chain-able methods.
     * @param V  Type of the view todo:
     * @param VM Type of the view model todo:
     *
     * @author Martin Albedinsky
     * @since 1.0
     *
     * @constructor Creates a new instance of BaseBuilder with default [presentation scheduler][PresentationSchedulers.primaryScheduler]
     * and default [data scheduler][DataSchedulers.primaryScheduler].
     */
    abstract class BaseBuilder<B : BaseBuilder<B, V, VM>, V : View<VM>, VM : ViewModel> protected constructor() : BasePresenter.BaseBuilder<B, V, VM>() {

        /**
         * See [ReactivePresenter.scheduler].
         */
        var scheduler: Scheduler = PresentationSchedulers.primaryScheduler()

        /**
         * See [ReactivePresenter.dataScheduler].
         */
        var dataScheduler: Scheduler = DataSchedulers.primaryScheduler()

        /**
         * Specifies a scheduler which should be used by the new presenter for presentation logic.
         *
         * Default value: **[PresentationSchedulers.primaryScheduler]**
         *
         * @param scheduler The desired scheduler to be used.
         * @return This builder to allow methods chaining.
         */
        fun scheduler(@NonNull scheduler: Scheduler): B {
            this.scheduler = scheduler
            return self
        }

        /**
         * Specifies a scheduler which should by used by the new presenter for observation of data
         * sources of which data the presenter will present.
         *
         * Default value: **[DataSchedulers.primaryScheduler]**
         *
         * @param scheduler The desired scheduler to be used.
         * @return This builder to allow methods chaining.
         */
        fun dataScheduler(@NonNull scheduler: Scheduler): B {
            this.dataScheduler = scheduler
            return self
        }
    }
}