/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.interaction

import android.support.annotation.NonNull
import universum.studios.android.arkhitekton.interaction.ResponseImpl.Builder
import universum.studios.android.arkhitekton.interaction.Result.Empty
import universum.studios.android.arkhitekton.util.Failure

/**
 * Factory class which may be used to create [Response] instances.
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @see Responses.createSuccess
 * @see Responses.createFailure
 */
class Responses private constructor() {

    /**
     */
    companion object {

        /**
         * Convenience method which creates instance of [Response] with [EMPTY][Result.empty] result.
         *
         * @param request  Request for which is the new response being created. Default is [EMPTY][Request.empty].
         * @return Success response ready to be dispatched.
         *
         * @see createSuccess
         */
        @JvmStatic @JvmOverloads @NonNull fun createEmptySuccess(@NonNull request: Request = Request.empty()): Response<Empty> {
            return createSuccess(request, Result.empty())
        }

        /**
         * Creates a new instance of successful [Response] with the specified `result`.
         *
         * @param request Request for which is the new response being created.
         * @param result Result of the processed request.
         * @param Result Type of the response result.
         * @return Use case response ready to be dispatched.
         *
         * @see createEmptySuccess
         * @see createFailure
         */
        @JvmStatic @NonNull fun <Result> createSuccess(@NonNull request: Request, @NonNull result: Result): Response<Result> {
            return Builder<Result>(request).result(result).build()
        }

        /**
         * Convenience method which creates instance of [Response] with [UNKNOWN][Failure.unknown] failure.
         *
         * @param request Request for which is the new response being created. Default is [EMPTY][Request.empty].
         * @param Result Type of the response result.
         * @return Failed response ready to be dispatched.
         *
         * @see createFailure
         */
        @JvmStatic @JvmOverloads @NonNull fun <Result> createUnknownFailure(@NonNull request: Request = Request.empty()): Response<Result> {
            return createFailure(request, Failure.unknown())
        }

        /**
         * Creates a new instance of [Response] with the specified `failure`.
         *
         * @param request Request for which is the new response being created.
         * @param failure Failure due to which the request has failed.
         * @param Result Type of the response result.
         * @return Use case response ready to be dispatched.
         *
         * @see createUnknownFailure
         * @see createSuccess
         */
        @JvmStatic @NonNull fun <Result> createFailure(@NonNull request: Request, @NonNull failure: Failure): Response<Result> {
            return Builder<Result>(request).failure(failure).build()
        }
    }

    /**
     */
    init {
        // Not allowed to be instantiated publicly.
        throw UnsupportedOperationException()
    }
}