/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.interaction

import android.support.annotation.NonNull
import universum.studios.android.arkhitekton.interaction.Result.Empty
import universum.studios.android.arkhitekton.util.Failure

/**
 * Interface describing a response which may be returned as response to execution of a specific [Request].
 * Each response is associated with its request which may be obtained via [getRequest]. If response
 * represents a [success][isSuccess], result of executed request may be obtained via [getResult],
 * otherwise if response represents a [failure][isFailure], the failure may be obtained via [getFailure].
 *
 * @param Result Type of the result that this response can provide via [getResult] in case of success.
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @see Request
 */
interface Response<out Result> {

    /*
     * Companion ===================================================================================
     */

    /**
     * Contract for [Response] element.
     */
    companion object Contract {

        /**
         * Empty instance of successful response.
         */
        private val SUCCESS: Response<Empty> = object : Response<Empty> {

            /*
             */
            @Suppress("UNCHECKED_CAST")
            override fun <R : Request> getRequest(): R = Request.empty() as R

            /*
             */
            override fun isSuccess(): Boolean = true

            /*
             */
            override fun getResult(): Empty = Result.empty()

            /*
             */
            override fun isFailure(): Boolean = false

            /*
             */
            override fun getFailure(): Failure = Failure.none()
        }

        /**
         * Empty instance of failed response.
         */
        private val FAILURE: Response<Empty> = object : Response<Empty> {

            /*
             */
            @Suppress("UNCHECKED_CAST")
            override fun <R : Request> getRequest(): R = Request.empty() as R

            /*
             */
            override fun isSuccess(): Boolean = false

            /*
             */
            override fun getResult(): Empty = Result.empty()

            /*
             */
            override fun isFailure(): Boolean = true

            /*
             */
            override fun getFailure(): Failure = Failure.unknown()
        }

        /**
         * Returns an empty instance of successful response with
         * [EMPTY][universum.studios.android.arkhitekton.interaction.Result.empty] result.
         *
         * @return Empty response ready to be dispatched.
         */
        @NonNull fun success(): Response<Empty> = SUCCESS

        /**
         * Returns an empty instance of failed response with
         * [EMPTY][universum.studios.android.arkhitekton.interaction.Result.empty] result.
         *
         * @return Empty response ready to be dispatched.
         */
        @NonNull fun failure(): Response<Empty> = FAILURE
    }

    /*
	 * Interface ===================================================================================
	 */

    /**
     * Basic interface for factory which may be used to create instances of [Response].
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    interface Factory {

        /**
         * Creates a new instance of successful response with the specified `result`.
         *
         * @param request  Request for which is the new response being created.
         * @param result   Result of the processed request.
         * @param Result Type of the response result.
         * @return Response ready to be dispatched.
         *
         * @see createFailure
         */
        @NonNull fun <Result> createSuccess(@NonNull request: Request, @NonNull result: Result): Response<Result>

        /**
         * Creates a new instance of response which represents a failure.
         *
         * @param request  Request for which is the new response being created.
         * @param failure  Failure due to which the request has failed.
         * @param Result Type of the response result.
         * @return Response ready to be dispatched.
         *
         * @see createSuccess
         */
        @NonNull fun <Result> createFailure(@NonNull request: Request, @NonNull failure: Failure): Response<Result>
    }

    /*
     * Functions ===================================================================================
     */

    /**
     * Returns the request this response is associated with.
     *
     * @param R Type of the expected request used to cast the request of this response to.
     * @return Request this response was created for.
     */
    @NonNull fun <R : Request> getRequest(): R

    /**
     * Checks whether this response is a success or not.
     *
     * @return `True` if this response represents a success of which result may be obtained via
     * [getResult], `false` if it is actually a failure.
     *
     * @see isFailure
     */
    fun isSuccess(): Boolean

    /**
     * Returns the result of the successful processing of the associated request.
     *
     * If this response is a failure, the returned result should be a simple `EMPTY` result.
     *
     * @return Result associated with this response.
     *
     * @see isSuccess
     */
    @NonNull fun getResult(): Result

    /**
     * Checks whether this response is a failure or not.
     *
     * @return `True` if this response represents a failure which may be obtained via [getFailure],
     * `false` if it is actually a success.
     *
     * @see isSuccess
     */
    fun isFailure(): Boolean

    /**
     * Returns the failure due to which the associated request has failed.
     *
     * If this response is a successful one, the returned failure should be a simple `NONE` failure.
     *
     * @return Failure associated with this response.
     *
     * @see isFailure
     */
    @NonNull fun getFailure(): Failure
}