/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.util

import android.support.annotation.NonNull
import android.support.annotation.Nullable

/**
 * Provides group of utility methods which may be used for checking of argument values, states and
 * also return values.
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
class Preconditions private constructor() {

    /**
     */
    companion object {

        /**
         * Checks whether the given argument `value` is **not null**.
         *
         * **Requirements:**
         *
         * *value != null*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @throws IllegalArgumentException In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkArgument(@Nullable value: Any?, @NonNull message: String = "Expected not null argument!") {
            value ?: throw IllegalArgumentException(message)
        }

        /**
         * Checks whether the given argument `value` is **positive**.
         *
         * **Requirements:**
         *
         * *value > 0*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @throws IllegalArgumentException In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkArgumentPositive(value: Long, @NonNull message: String = "Expected positive argument!") {
            if (value <= 0) throw IllegalArgumentException(message)
        }

        /**
         * Checks whether the given argument `value` is **non positive**.
         *
         * **Requirements:**
         *
         * *value <= 0*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @throws IllegalArgumentException In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkArgumentNonPositive(value: Long, @NonNull message: String = "Expected non positive argument!") {
            if (value > 0) throw IllegalArgumentException(message)
        }

        /**
         * Checks whether the given argument `value` is **negative**.
         *
         * **Requirements:**
         *
         * *value < 0*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @throws IllegalArgumentException In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkArgumentNegative(value: Long, @NonNull message: String = "Expected negative argument!") {
            if (value >= 0) throw IllegalArgumentException(message)
        }

        /**
         * Checks whether the given argument `value` is **non negative**.
         *
         * **Requirements:**
         *
         * *value >= 0*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @throws IllegalArgumentException In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkArgumentNonNegative(value: Long, @NonNull message: String = "Expected non negative argument!") {
            if (value < 0) throw IllegalArgumentException(message)
        }

        /**
         * Checks whether the given argument `value` is **true**.
         *
         * **Requirements:**
         *
         * *value == true*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @throws IllegalArgumentException In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkTrue(@Nullable value: Boolean, @NonNull message: String = "Expected true!"): Boolean {
            return if (value) value else throw AssertionError(message)
        }

        /**
         * Checks whether the given argument `value` is **false** and returns that value if it meets
         * the requirements.
         *
         * **Requirements:**
         *
         * *value == false*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @return The given value in case of successful check.
         * @throws AssertionError In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkFalse(@Nullable value: Boolean, @NonNull message: String = "Expected false!"): Boolean {
            return if (!value) value else throw AssertionError(message)
        }

        /**
         * Checks whether the given `value` is **not null** and returns that value if it meets the
         * requirements.
         *
         * **Requirements:**
         *
         * *value != null*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @param T The type of value to be checked.
         * @return The given value in case of successful check.
         * @throws AssertionError In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun <T> checkNotNull(@Nullable value: T?, @NonNull message: String = "Expected not null!"): T {
            return value ?: throw AssertionError(message)
        }

        /**
         * Checks whether the given `value` is **positive** and returns that value if it meets the
         * requirements.
         *
         * **Requirements:**
         *
         * *value > 0*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @return The given value in case of successful check.
         * @throws AssertionError In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkPositive(value: Long, @NonNull message: String = "Expected positive!"): Long {
            return if (value > 0) value else throw AssertionError(message)
        }

        /**
         * Checks whether the given `value` is **non positive** and returns that value if it meets the
         * requirements.
         *
         * **Requirements:**
         *
         * *value <= 0*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @return The given value in case of successful check.
         * @throws AssertionError In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkNonPositive(value: Long, @NonNull message: String = "Expected non positive!"): Long {
            return if (value <= 0) value else throw AssertionError(message)
        }

        /**
         * Checks whether the given `value` is **negative** and returns that value if it meets the
         * requirements.
         *
         * **Requirements:**
         *
         * *value < 0*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @return The given value in case of successful check.
         * @throws AssertionError In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkNegative(value: Long, @NonNull message: String = "Expected negative!"): Long {
            return if (value < 0) value else throw AssertionError(message)
        }

        /**
         * Checks whether the given `value` is **non negative** and returns that value if it meets the
         * requirements.
         *
         * **Requirements:**
         *
         * *value >= 0*
         *
         * @param value The desired value to be checked.
         * @param message The desired message for exception thrown in case of failed check.
         * @return The given value in case of successful check.
         * @throws AssertionError In case when the checked value does not meet the requirements.
         */
        @JvmStatic @JvmOverloads fun checkNonNegative(value: Long, @NonNull message: String = "Expected non negative!"): Long {
            return if (value >= 0) value else throw AssertionError(message)
        }
    }

    /**
     * Factory class which may be used to create standard messages for exceptions thrown in cases
     * when desired preconditions are not met.
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    class Messages private constructor() {

        /**
         */
        companion object {

            /**
             * Creates a description message informing that argument with the specified `argumentName`
             * is not provided or provided as null.
             *
             * @param argumentName Name of the argument that is null or not provided.
             * @return Message ready to be used.
             */
            @JvmStatic @NonNull fun nullArgument(@NonNull argumentName: String) = "No or null > $argumentName < specified!"
        }

        /**
         */
        init {
            // Not allowed to be instantiated publicly.
            throw UnsupportedOperationException()
        }
    }

    /**
     */
    init {
        // Not allowed to be instantiated publicly.
        throw UnsupportedOperationException()
    }
}