/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.view.presentation

import android.arch.lifecycle.Lifecycle
import android.arch.lifecycle.Lifecycle.State.CREATED
import android.arch.lifecycle.Lifecycle.State.DESTROYED
import android.arch.lifecycle.Lifecycle.State.INITIALIZED
import android.arch.lifecycle.Lifecycle.State.RESUMED
import android.arch.lifecycle.Lifecycle.State.STARTED
import android.arch.lifecycle.LifecycleObserver
import android.arch.lifecycle.OnLifecycleEvent
import android.support.annotation.NonNull
import universum.studios.android.arkhitekton.ArkhitektonLogging
import universum.studios.android.arkhitekton.util.Preconditions
import universum.studios.android.arkhitekton.view.View
import universum.studios.android.arkhitekton.view.ViewModel
import java.util.concurrent.atomic.AtomicBoolean

/**
 * A [Presenter] base implementation ... todo:
 *
 * @param V  Type of the view todo:
 * @param VM Type of the view model todo:
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @constructor Creates a new instance of BasePresenter with the given `builder's` configuration.
 * @param builder The builder with configuration for the new presenter.
 */
abstract class BasePresenter<V : View<VM>, VM : ViewModel> protected constructor(@NonNull builder: BaseBuilder<*, V, VM>) : Presenter<V> {

    /*
	 * Companion ===================================================================================
	 */

    /**
     */
    companion object {

        /**
         * Log TAG.
         */
        internal const val TAG = "BasePresenter"
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /**
     * todo:
     */
    private val viewModel: VM = Preconditions.checkNotNull(builder.viewModel, Preconditions.Messages.nullArgument("viewModel"))

    /**
     * todo:
     */
    private var view: V? = null

    /**
     * todo:
     */
    @Suppress("LeakingThis")
    private val viewLifecycleActionRegistry by lazy { LifecycleActionRegistry(this) }

    /**
     * todo:
     */
    private val destroyed = AtomicBoolean(false)

    /*
     * Initialization ==============================================================================
     */

    /*
     * Methods =====================================================================================
     */

    /**
     * Returns the name of this presenter.
     *
     * @return This presenter's name.
     */
    @NonNull fun name(): String = javaClass.simpleName

    /**
     * Returns the view model associated with this presenter.
     *
     * @return View model which this presenter uses to persist view's state.
     */
    protected fun getViewModel() = viewModel

    /**
     */
    override fun attachView(view: V) {
        if (this.view != null) {
            throw IllegalStateException("Already has view attached!")
        }
        this.view = view
        if (this is LifecycleObserver) {
            view.lifecycle.addObserver(viewLifecycleActionRegistry)
        }
        onViewAttached(view)
    }

    /**
     * todo:
     *
     * @param view
     */
    protected open fun onViewAttached(view: V) {}

    /**
     * todo:
     *
     * @return
     */
    protected fun isViewAttached() = view != null

    /**
     * todo:
     */
    private fun assertViewAttached() {
        this.view ?: throw IllegalStateException("No view attached!")
    }

    /**
     * todo:
     *
     * @return
     */
    protected fun getViewLifecycleCurrentState() = getView().lifecycle.currentState

    /**
     * todo:
     *
     * @param lifecycleAction
     */
    protected fun performViewLifecycleAction(lifecycleAction: LifecycleAction): Boolean {
        if (isViewAttached() && getViewLifecycleCurrentState().isAtLeast(lifecycleAction.stateCondition)) {
            lifecycleAction.run()
            return true;
        } else {
            registerViewLifecycleAction(
                    LifecycleActionRegistry.resolveEventForState(lifecycleAction.stateCondition),
                    lifecycleAction
            )
            return false;
        }
    }

    /**
     * todo:
     *
     * @param lifecycleEvent
     * @param action
     */
    protected fun registerViewLifecycleAction(lifecycleEvent: Lifecycle.Event, lifecycleAction: LifecycleAction) = viewLifecycleActionRegistry.registerActionForEvent(lifecycleEvent, lifecycleAction)

    /**
     * todo:
     *
     * @return
     * @throws IllegalStateException todo:
     */
    protected fun getView(): V {
        assertViewAttached()
        return view!!
    }

    /**
     */
    override fun detachView(view: V) {
        if (this.view == null) {
            return
        }
        if (this.view !== view) {
            throw IllegalArgumentException("Cannot detach current view. The view passed to detachView(...) is of different reference.")
        }
        this.view = null
        if (this is LifecycleObserver) {
            view.lifecycle.removeObserver(viewLifecycleActionRegistry)
        }
        onViewDetached(view)
    }

    /**
     * todo:
     *
     * @param view
     */
    protected open fun onViewDetached(view: V) {}

    /**
     */
    override fun destroy() {
        if (destroyed.compareAndSet(false, true)) {
            this.viewLifecycleActionRegistry.clearAllActions()
            onDestroyed()
        }
    }

    /**
     * todo:
     */
    protected open fun onDestroyed() {
        // todo:
    }

    /*
     */
    @NonNull override fun toString() = "${name()}{viewAttached: ${view != null}, destroyed: $destroyed}"

    /*
     * Inner classes ===============================================================================
     */

    /**
     * Base implementation of [Presenter.Builder] which should be used by [BasePresenter] inheritance
     * hierarchies.
     *
     * @param B Type of the builder used as return type for builder's chain-able methods.
     * @param V Type of the view of which instance may be associated with new presenter.
     * @param VM Type of the view model of which instance may be associated with new presenter.
     *
     * @author Martin Albedinsky
     * @since 1.0
     *
     * @constructor Creates a new instance of BaseBuilder with empty state.
     */
    abstract class BaseBuilder<B : BaseBuilder<B, V, VM>, V : View<VM>, VM : ViewModel> protected constructor() : Presenter.Builder<Presenter<V>> {

        /**
         * Obtains reference to this builder instance which is used for purpose of methods chaining.
         *
         * @return This builder instance.
         */
        protected abstract val self: B

        /**
         * See [BaseBuilder.viewModel].
         */
        lateinit var viewModel: VM

        /**
         * Specifies a view model to be associated with the new presenter.
         *
         * @param viewModel The desired view model for the presenter.
         * @return This builder to allow methods chaining.
         */
        fun viewModel(viewModel: VM): B {
            this.viewModel = viewModel
            return self
        }
    }

    /**
     * A [BaseBuilder] simple implementation which may be used in cases when a concrete implementation
     * of [BasePresenter] does not require any additional dependencies except its view model.
     * Such presenters may be instantiated simply by passing instance of this builder created via
     * [SimpleBuilder.create] to such presenters's constructor.
     *
     * @param V Type of the view of which instance may be associated with new presenter.
     * @param VM Type of the view model of which instance may be associated with new presenter.
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    class SimpleBuilder<V : View<VM>, VM : ViewModel> private constructor() : BaseBuilder<SimpleBuilder<V, VM>, V, VM>() {

        /**
         */
        companion object {

            /**
             * Creates a new SimpleBuilder with the specified `viewModel`.
             *
             * @param viewModel The desired view model for the new presenter.
             * @param V Type of the view of which instance may be associated with the new presenter.
             * @param VM Type of the view model of which instance may be associated with the new presenter.
             * @return Builder instance ready to be passed to the presenter's constructor.
             */
            @JvmStatic fun <V : View<VM>, VM : ViewModel> create(viewModel: VM) = SimpleBuilder<V, VM>().apply {this.viewModel = viewModel }
        }

        /*
         */
        override val self = this

        /*
         */
        override fun build(): Presenter<V> = throw UnsupportedOperationException("Pass instance of SimpleBuilder to the appropriate constructor!")
    }

    /**
     * todo:
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    abstract class LifecycleAction : Runnable {

        /**
         */
        companion object {

            /**
             * todo:
             */
            @JvmStatic
            val DEFAULT_PRIORITY = 0
        }

        /**
         * todo:
         */
        internal var priority = DEFAULT_PRIORITY

        /**
         * todo:
         */
        internal var stateCondition = Lifecycle.State.INITIALIZED

        /**
         * todo:
         */
        fun withPriority(priority: Int): LifecycleAction {
            this.priority = priority
            return this
        }

        /**
         * todo:
         */
        fun withStateCondition(lifecycleState: Lifecycle.State): LifecycleAction {
            this.stateCondition = lifecycleState
            return this
        }
    }

    /**
     * todo:
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    interface LifecycleActionSubscription {

        /**
         * todo:
         */
        fun cancel()
    }

    /**
     *
     */
    private class LifecycleActionSubscriptionImpl : LifecycleActionSubscription {

        /**
         */
        override fun cancel() {
            // todo:
        }
    }

    /**
     * todo:
     */
    internal class LifecycleActionRegistry(internal val parent: BasePresenter<*, *>) : LifecycleObserver {

        /**
         */
        companion object {

            /**
             * todo:
             *
             * @param state
             * @return
             */
            fun resolveEventForState(state: Lifecycle.State): Lifecycle.Event {
                return when (state) {
                    INITIALIZED -> Lifecycle.Event.ON_ANY
                    CREATED -> Lifecycle.Event.ON_CREATE
                    STARTED -> Lifecycle.Event.ON_START
                    RESUMED -> Lifecycle.Event.ON_RESUME
                    DESTROYED -> Lifecycle.Event.ON_DESTROY
                    else -> Lifecycle.Event.ON_ANY
                }
            }
        }

        /**
         * todo:
         */
        private val onCreateActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onStartActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onResumeActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onPauseActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onStopActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onDestroyActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         *
         * @throws IllegalArgumentException
         */
        fun registerActionForEvent(lifecycleEvent: Lifecycle.Event, lifecycleAction: LifecycleAction): LifecycleActionSubscription {
            when (lifecycleEvent) {
                Lifecycle.Event.ON_CREATE -> this.onCreateActions.add(lifecycleAction)
                Lifecycle.Event.ON_START -> this.onStartActions.add(lifecycleAction)
                Lifecycle.Event.ON_RESUME -> this.onResumeActions.add(lifecycleAction)
                Lifecycle.Event.ON_PAUSE -> this.onPauseActions.add(lifecycleAction)
                Lifecycle.Event.ON_STOP -> this.onStopActions.add(lifecycleAction)
                Lifecycle.Event.ON_DESTROY -> this.onDestroyActions.add(lifecycleAction)
                Lifecycle.Event.ON_ANY -> throw IllegalArgumentException("Cannot register lifecycle action for event($lifecycleEvent)!")
            }
            ArkhitektonLogging.d(parent.name(), "Registered lifecycle action for event($lifecycleEvent).")
            // todo: associate subscriptionn with action ...
            return LifecycleActionSubscriptionImpl()
        }

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_CREATE)
        fun onViewCreated() = runActionsForViewState("CREATED", onCreateActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_START)
        fun onViewStarted() = runActionsForViewState("STARTED", onStartActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_RESUME)
        fun onViewResumed() = runActionsForViewState("RESUMED", onResumeActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_PAUSE)
        fun onViewPaused() = runActionsForViewState("PAUSED", onPauseActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
        fun onViewStop() = runActionsForViewState("STOPPED", onStopActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
        fun onViewDestroyed() = runActionsForViewState("DESTROYED", onDestroyActions)

        /**
         * todo:
         */
        private fun runActionsForViewState(viewState: String, actions: MutableList<LifecycleAction>) {
            ArkhitektonLogging.d(parent.name(), createLogMessageForLifecycleActions(viewState, actions.size))
            if (!actions.isEmpty()) {
                actions.forEach { it.run() }
                actions.clear()
            }
        }

        /**
         * todo:
         *
         * @param viewState
         * @param actionsCount
         * @return
         */
        private fun createLogMessageForLifecycleActions(viewState: String, actionsCount: Int): String {
            var message = "View $viewState."
            message += if (actionsCount == 0) {
                " No registered actions to run."
            } else {
                " Running registered actions in count($actionsCount)."
            }
            return message
        }

        /**
         * todo:
         */
        fun clearAllActions() {
            onCreateActions.clear()
            onStartActions.clear()
            onResumeActions.clear()
            onPauseActions.clear()
            onStopActions.clear()
            onDestroyActions.clear()
        }
    }
}