/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.interaction.usecase

import universum.studios.android.arkhitekton.util.Failure
import universum.studios.android.arkhitekton.util.InstanceBuilder

/**
 * An implementation of [UseCaseResponse].
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @constructor Creates a new instance of UseCaseResponseImpl with the given `builder's` configuration.
 * @param builder The builder with configuration for the new response.
 */
internal class UseCaseResponseImpl<out Result> internal constructor(builder: Builder<Result>) : UseCaseResponse<Result> {

    /*
     * Companion ===================================================================================
     */

    /**
     */
    companion object {

        /**
         * Log TAG.
         */
        internal const val TAG = "UseCaseResponse"
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /**
     * Request associated with this response.
     */
    private val request = builder.request

    /**
     * Result which was produced by successful execution of [.request] by the associated use case.
     *
     *
     * If this response is a failure, this should be [UseCaseResult.Contract.empty].
     */
    private val result = builder.result

    /**
     * Failure which occurred during execution of the associated use case.
     *
     *
     * If this response is not a failure this should be [Failure.Contract.none].
     */
    private val failure = builder.failure

    /*
     * Initialization ==============================================================================
     */

    /*
     * Methods =====================================================================================
     */

    /*
     */
    @Suppress("UNCHECKED_CAST")
    override fun <R : UseCaseRequest> getRequest(): R = request as R

    /*
     */
    override fun isSuccess() = !isFailure()

    /*
     */
    override fun getResult(): Result = result

    /*
     */
    override fun isFailure() = Failure.none() != failure

    /*
     */
    override fun getFailure(): Failure = failure

    /*
     */
    override fun toString(): String {
        val builder = StringBuilder(64)
        builder.append(TAG)
        builder.append("{request: ")
        builder.append(request)
        builder.append(", isSuccess: ")
        builder.append(isSuccess())
        builder.append(", result: ")
        builder.append(result)
        builder.append(", isFailure: ")
        builder.append(isFailure())
        builder.append(", failure: ")
        builder.append(failure)
        return builder.append("}").toString()
    }

    /*
     * Inner classes ===============================================================================
     */

    /**
     * A builder which may be used to create instances of [UseCaseResponseImpl].
     *
     * @param Result Type of the result the response will carry.
     *
     * @author Martin Albedinsky
     *
     * @constructor Creates a new instance of BuilderImpl with given `request` the new response
     * will be created with.
     * @param request The request the new response to be associated with.
     */
    internal class Builder<Result> internal constructor(
            /**
             * See [UseCaseResponseImpl.request].
             */
            val request: UseCaseRequest) : InstanceBuilder<UseCaseResponseImpl<Result>> {

        /**
         * See [UseCaseResponseImpl.result].
         */
        @Suppress("UNCHECKED_CAST")
        internal var result = UseCaseResult.empty() as Result

        /**
         * See [UseCaseResponseImpl.failure].
         */
        internal var failure = Failure.none()

        /**
         * Specifies a result for the new response should represent a success response.
         *
         * @param result The desired result for the new response.
         * @return This builder to allow methods chaining.
         */
        fun result(result: Result): Builder<Result> = apply { this.result = result }

        /**
         * Specifies a failure for the new response which should represent a failure response.
         *
         * @param failure The desired failure for the new response.
         * @return This builder to allow methods chaining.
         */
        fun failure(failure: Failure): Builder<Result> = apply { this.failure = failure }

        /*
         */
        override fun build() = UseCaseResponseImpl(this)
    }
}