/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.util;

import android.support.annotation.NonNull;

/**
 * Basic implementation of {@link Failure} todo:
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
public class BasicFailure extends Exception implements Failure {

	/**
	 * Log TAG.
	 */
	@SuppressWarnings("unused")
	private static final String TAG = "BasicFailure";

	/**
	 * todo:
	 */
	public static BasicFailure UNKNOWN = create("UNKNOWN");

	/**
	 * Error identifying and describing this failure.
	 */
	private final Error error;

	/**
	 * Creates a new instance of BasicFailure with the specified <var>error</var> and <var>cause</var>.
	 *
	 * @param error Error describing the occurred failure.
	 * @param cause The cause due to which the failure occurred.
	 */
	protected BasicFailure(@NonNull final Error error, @NonNull final Throwable cause) {
		super(cause);
		this.error = error;
	}

	/**
	 * Creates a new instance of BasicFailure with an error that will have the specified
	 * <var>errorCode</var> and {@code NONE} cause.
	 *
	 * @param errorCode Code for the desired error for the new failure.
	 * @return Failure ready to be associated with a desired response.
	 */
	@NonNull public static BasicFailure create(@NonNull final String errorCode) {
		return create(Failure.Error.create(errorCode));
	}

	/**
	 * Creates a new instance of BasicFailure with the specified <var>error</var> and {@code NONE}
	 * cause.
	 *
	 * @param error Desired error for the new failure.
	 * @return Failure ready to be associated with a desired response.
	 * @see #create(String)
	 */
	@NonNull public static BasicFailure create(@NonNull final Failure.Error error) {
		return create(error, Failure.Cause.none());
	}

	/**
	 * Creates a new instance of BasicFailure with the specified <var>error</var> and <var>cause</var>.
	 *
	 * @param error Desired error for the new failure.
	 * @param cause Desired cause for the new failure.
	 * @return Failure ready to be associated with a desired response.
	 * @see #create(Error)
	 * @see #create(String)
	 */
	@NonNull public static BasicFailure create(@NonNull final Failure.Error error, @NonNull final Throwable cause) {
		return new BasicFailure(error, cause);
	}

	/**
	 */
	@Override @NonNull public Error getError() { return error; }

	/**
	 */
	@Override @NonNull public Exception toException() { return this; }

	/**
	 */
	@Override public String toString() {
		return getClass().getSimpleName() + "{error: " + error + ", cause: " + getCause() + "}";
	}
}