/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.control

import android.support.annotation.CallSuper
import android.support.annotation.NonNull
import io.reactivex.Scheduler
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import universum.studios.android.arkhitekton.interaction.Interactor
import universum.studios.android.arkhitekton.presentation.Presenter

/**
 * todo: description
 *
 * @param I Type of the interactor to which should the controller pass interaction requests to be processed.
 * @param P Type of the presenter to which should the controller dispatch appropriate responses for&nbsp;
 * processed requests and/or other interaction events.
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
abstract class ReactiveController<out I : Interactor, out P : Presenter<*>> protected constructor(builder: BaseBuilder<*, I, P>) : BaseController<I, P>(builder) {

    /*
	 * Companion ===================================================================================
	 */

    /**
     */
    companion object {

        /**
         * Log TAG.
         */
        internal const val TAG = "ReactiveController"

        /**
         * Global scheduler by default used for interaction.
         *
         * @see Schedulers.computation
         */
        @NonNull var interactionScheduler: Scheduler = Schedulers.computation()

        /**
         * Global scheduler by default used for presentation.
         *
         * @see AndroidSchedulers.mainThread
         */
        @NonNull var presentationScheduler: Scheduler = AndroidSchedulers.mainThread()
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /**
     * A scheduler which should be used by this controller for processing of interaction requests.
     */
    @NonNull protected val interactionScheduler = builder.interactionScheduler

    /**
     * Scheduler which should by used by this controller for presentation of a received responses as
     * results of interaction requests.
     */
    @NonNull protected val presentationScheduler = builder.presentationScheduler

    /**
     * Subscriptions that has been added via [registerSubscription]. May be deleted,
     * each respectively, via [unregisterSubscription]
     */
    private val subscriptions by lazy { CompositeDisposable() }

    /*
     * Constructors ================================================================================
     */

    /*
     * Methods =====================================================================================
     */

    /**
     * Checks whether this controler has some subscriptions registered or not.
     *
     * @return `True` if there are some subscriptions registered, `false` otherwise.
     *
     * @see registerSubscription
     * @see unregisterSubscription
     */
    fun hasSubscriptions() = subscriptions.size() > 0

    /**
     * Registers a subscription todo: ...
     *
     * @param subscription The desired subscription which may be later disposed via [unregisterSubscription].
     *
     * @see hasSubscriptions
     * @see unregisterAllSubscriptions
     */
    protected fun registerSubscription(@NonNull subscription: Disposable) = subscriptions.add(subscription)

    /**
     * Un-registers the subscription previously registered via [registerSubscription].
     *
     * If the given `subscription` has not been registered, this method does nothing.
     *
     * @param subscription The desired subscription to be removed and also disposed.
     */
    protected fun unregisterSubscription(@NonNull subscription: Disposable) = subscriptions.remove(subscription)

    /**
     * Un-registers all subscriptions registered by this controller. This will also dispose all such
     * subscriptions.
     *
     * @see unregisterSubscription
     */
    protected fun unregisterAllSubscriptions() = subscriptions.clear()

    /*
     */
    @CallSuper override fun onDeactivated() {
        super.onDeactivated()
        unregisterAllSubscriptions()
    }

    /*
     * Inner classes ===============================================================================
     */

    /**
     * Extended base implementation of [BaseController.BaseBuilder] which should be used by
     * implementations of [ReactiveController].
     *
     * @param B Type of the builder used as return type for builder's chain-able methods.
     * @param I Type of the interactor of which instance may be associated with new controller.
     * @param P Type of the presenter of which instance may be associated with new controller.
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    abstract class BaseBuilder<B : BaseBuilder<B, I, P>, I : Interactor, P : Presenter<*>>
    protected constructor(@NonNull interactor: I, @NonNull presenter: P) : BaseController.BaseBuilder<B, I, P>(interactor, presenter) {

        /**
         * See [ReactiveController.interactionScheduler].
         */
        internal var interactionScheduler = ReactiveController.interactionScheduler

        /**
         * See [ReactiveController.presentationScheduler].
         */
        internal var presentationScheduler = ReactiveController.presentationScheduler

        /**
         * Specifies a scheduler which should be used by the new interactor for processing of an
         * interaction requests.
         *
         * Default value: **[ReactiveController.Companion.interactionScheduler]**
         *
         * @param scheduler The desired scheduler.
         * @return This builder to allow methods chaining.
         */
        fun interactionScheduler(@NonNull scheduler: Scheduler): B {
            this.interactionScheduler = scheduler
            return self
        }

        /**
         * Specifies a scheduler which should by used by the new interactor for presentation of a
         * received responses as results of interaction requests.
         *
         *
         * Default value: **[ReactiveController.Companion.presentationScheduler]**
         *
         * @param scheduler The desired scheduler.
         * @return This builder to allow methods chaining.
         */
        fun presentationScheduler(@NonNull scheduler: Scheduler): B {
            this.presentationScheduler = scheduler
            return self
        }
    }
}