/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.util

import android.support.annotation.NonNull

/**
 * Defines an interface for describing a failure which may occur due to some [Cause][getCause] and
 * may be identified by a specific [Error][getError].
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
interface Failure {

    /*
	 * Companion ===================================================================================
	 */

    /**
     * Contract for [Failure] element.
     */
    companion object Contract {

        /**
         * Instance which may be used to indicate a *none* failure.
         */
        internal val NONE: Failure = object : Failure {

            /*
             */
            @NonNull override fun getError() = Error.NONE

            /*
             */
            @NonNull override fun getCause() = Cause.NONE

            /*
             */
            @NonNull override fun toException() = Exception(toString())

            /*
             */
            override fun toString() = "${Failure::class.java.simpleName}{NONE}"
        }

        /**
         * Instance which may be used to indicate an *unknown* failure.
         */
        @NonNull internal val UNKNOWN: Failure = object : Failure {

            /*
             */
            @NonNull override fun getError() = Error.UNKNOWN

            /*
             */
            @NonNull override fun getCause() = Cause.NONE

            /*
             */
            @NonNull override fun toException() = Exception(toString())

            /*
             */
            override fun toString() = "${Failure::class.java.simpleName}{UNKNOWN}"
        }

        /**
         * Returns an instance of [Failure] which may be used as *null Object* like element.
         *
         * @return Failure ready to be used.
         */
        @NonNull fun none() = NONE

        /**
         * Returns an instance of [Failure] which may be used to inform that an unknown failure has
         * occurred and there are no specific information known.
         *
         * @return Failure ready to be associated with a response.
         */
        @NonNull fun unknown() = UNKNOWN

        /**
         * Convenience method for casting a throwable into [Failure] if the throwable is of such type.
         *
         * @param throwable Throwable to cast into failure.
         * @return The given throwable as failure or an exception is thrown.
         * @throws IllegalArgumentException If the given throwable is not of [Failure] type.
         */
        @NonNull fun asFailure(@NonNull throwable: Throwable): Failure {
            return throwable as? Failure ?: throw AssertionError("Specified throwable($throwable) is not a Failure!")
        }
    }

    /*
     * Methods =====================================================================================
     */

    /**
     * Returns the error identifying this failure.
     *
     * @return This failure's error.
     * @see getCause
     */
    @NonNull fun getError(): Error

    /**
     * Returns the cause describing why this failure occurred.
     *
     * @return This failure's cause.
     * @see getError
     */
    @NonNull fun getCause(): Throwable

    /**
     * Maps (transforms) this failure into Exception.
     *
     * @return This failure as Exception ready to be propagated further.
     */
    @NonNull fun toException(): Exception

    /*
	 * Interface ===================================================================================
	 */

    /**
     * Basic error which may be identified by its [code][getCode] and described via [description][getDescription].
     *
     * @author Martin Albedinsky
     * @since 1.0
     *
     * @constructor Creates a new instance of Error with the given `code` and `description`.
     * @param code The desired code for the new error.
     * @param description The desired description for the new error.
     */
    @Suppress("MemberVisibilityCanBePrivate")
    open class Error protected constructor(
            /**
             * Code identifying this error.
             */
            @NonNull internal val code: String,
            /**
             * Description describing this error.
             */
            @NonNull internal val description: Description) {

        /**
         * Contract for [Failure.Error] element.
         */
        companion object Contract {

            /**
             * Instance which may be used to indicate a 'none' error.
             */
            @NonNull internal val NONE = Error("NONE", Description.EMPTY)

            /**
             * Instance which may be used to indicate an 'unknown' error.
             */
            @NonNull internal val UNKNOWN = Error("UNKNOWN", Description.EMPTY)

            /**
             * Returns an instance of error with 'NONE' code and [EMPTY][Description.empty] description.
             *
             * @return Error ready to be associated with a desired failure.
             */
            @JvmStatic @NonNull fun none() = NONE

            /**
             * Returns an instance of error with 'UNKNOWN' code and [EMPTY][Description.empty] description.
             *
             * @return Error ready to be associated with a desired failure.
             */
            @JvmStatic @NonNull fun unknown() = UNKNOWN

            /**
             * Creates a new Error with the specified `code` and `description`.
             *
             * @param code The desired code for the new error.
             * @param description The desired description for the new error. Default is [UNKNOWN][Error.unknown].
             * @return Error ready to be associated with desired failure.
             */
            @JvmStatic @JvmOverloads @NonNull fun create(
                    @NonNull code: String,
                    @NonNull description: Description = Description.EMPTY
            ) = Error(code, description)
        }

        /**
         * Returns the code of this error.
         *
         * @return This error's code.
         * @see getDescription
         */
        @NonNull fun getCode() = code

        /**
         * Returns the description specified for this error.
         *
         * @return This error's description.
         */
        @NonNull fun getDescription() = description

        /*
         */
        @NonNull override fun toString() = "${javaClass.simpleName}{code: $code, description: $description}"
    }

    /**
     * Base implementation of [Throwable] which may be used as base for concrete cause implementations.
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    abstract class Cause : Throwable() {

        /**
         * Contract for [Failure.Cause] element.
         */
        companion object Contract {

            /**
             * Instance of [Throwable] indicating no particular cause.
             */
            @NonNull internal var NONE: Throwable = object : Cause() {

                /*
                 */
                @NonNull override fun toString() = "${Cause::class.java.simpleName}{NONE}"
            }

            /**
             * Returns an instance of [Throwable] which may be used to indicate that a [Failure] has
             * no particular cause.
             *
             * @return Cause ready to be associated with a desired failure.
             */
            @JvmStatic @NonNull fun none() = NONE
        }
    }
}