/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.interaction.usecase

import android.support.annotation.NonNull
import universum.studios.android.arkhitekton.util.Failure

/**
 * Factory class which may be used to create [UseCaseResponse] instances.
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @see UseCaseResponses.createSuccess
 * @see UseCaseResponses.createFailure
 */
@Suppress("MemberVisibilityCanBePrivate")
class UseCaseResponses private constructor() {

    /**
     */
    companion object {

        /**
         * Convenience method which creates instance of [UseCaseResponse] with [EMPTY][UseCaseRequest.empty]
         * request and [EMPTY][UseCaseResult.empty] result.
         *
         * @return Success response ready to be dispatched.
         *
         * @see createSuccess
         */
        @JvmStatic @NonNull fun createEmptySuccess() = createSuccess(UseCaseRequest.empty(), UseCaseResult.empty())

        /**
         * Creates a new instance of successful use case response with the specified `result`.
         *
         * @param request  Request for which is the response being created.
         * @param result   Result of the processed request.
         * @param Result Type of the response result.
         * @return Use case response ready to be dispatched.
         *
         * @see createEmptySuccess
         * @see createFailure
         */
        @JvmStatic @NonNull fun <Result> createSuccess(@NonNull request: UseCaseRequest, @NonNull result: Result): UseCaseResponse<Result> =
                UseCaseResponseImpl.BuilderImpl<Result>(request).result(result).build()

        /**
         * Convenience method which creates instance of [UseCaseResponse] with [EMPTY][UseCaseRequest.empty]
         * request and [UNKNOWN][Failure.unknown] failure.
         *
         * @return Failed response ready to be dispatched.
         *
         * @see createFailure
         */
        @JvmStatic @NonNull fun createUnknownFailure() = createFailure<UseCaseResult.Empty>(UseCaseRequest.empty(), Failure.unknown())

        /**
         * Creates a new instance of use case response which represents a failure.
         *
         * @param request  Request for which is the response being created.
         * @param failure  Failure due to which processing of the request has failed.
         * @param Result Type of the response result.
         * @return Use case response ready to be dispatched.
         *
         * @see createUnknownFailure
         * @see createSuccess
         */
        @JvmStatic @NonNull fun <Result> createFailure(@NonNull request: UseCaseRequest, @NonNull failure: Failure): UseCaseResponse<Result> =
                UseCaseResponseImpl.BuilderImpl<Result>(request).failure(failure).build()
    }

    /**
     */
    init {
        // Not allowed to be instantiated publicly.
        throw UnsupportedOperationException()
    }
}