/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.interaction.usecase

import android.support.annotation.NonNull
import universum.studios.android.arkhitekton.util.Failure

/**
 * Interface describing a response which is returned by [UseCase] as result of its execution via
 * [UseCase.execute].
 *
 * @param Result Type of the result that this response can provide as result of processed request via [getResult].
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @see UseCaseRequest
 */
interface UseCaseResponse<out Result> {

    /*
     * Companion ===================================================================================
     */

    /**
     * Contract for [UseCaseResponse] element.
     */
    companion object Contract {

        /**
         * Convenience method which creates instance of [UseCaseResponse] with [EMPTY][UseCaseRequest.empty]
         * request and [EMPTY][UseCaseResult.empty] result.
         *
         * @return Success response ready to be dispatched.
         *
         * @see createUnknownFailure
         * @see factory
         */
        // fixme: remove in the next release ...
        @Deprecated("use UseCaseReponses.createEmptySuccess(...)")
        @NonNull fun createEmptySuccess() = UseCaseResponses.createEmptySuccess()

        /**
         * Convenience method which creates instance of [UseCaseResponse] with [EMPTY][UseCaseRequest.empty]
         * request and [UNKNOWN][Failure.unknown] failure.
         *
         * @return Failed response ready to be dispatched.
         *
         * @see createEmptySuccess
         * @see factory
         */
        // fixme: remove in the next release ...
        @Deprecated("use UseCaseReponses.createUnknownFailure(...)")
        @NonNull fun createUnknownFailure() = UseCaseResponses.createUnknownFailure()

        /**
         * Returns the factory instance which may be used to create instances of [UseCaseResponse].
         *
         * @return Response factory ready to be used.
         *
         * @see createEmptySuccess
         * @see createUnknownFailure
         */
        // fixme: remove in the next release ...
        @Deprecated("use UseCaseReponses.createSuccess(...) or UseCaseResponses.createFailure(...)")
        @NonNull fun factory(): Factory = UseCaseResponseImpl.FACTORY
    }

    /*
	 * Interface ===================================================================================
	 */

    /**
     * Basic interface for factory which may be used to create instances of [UseCaseResponse].
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    interface Factory {

        /**
         * Creates a new instance of successful use case response with the specified `result`.
         *
         * @param request  Request for which is the response being created.
         * @param result   Result of the processed request.
         * @param Result Type of the response result.
         * @return Use case response ready to be dispatched.
         *
         * @see createFailure
         */
        @NonNull fun <Result> createSuccess(@NonNull request: UseCaseRequest, @NonNull result: Result): UseCaseResponse<Result>

        /**
         * Creates a new instance of use case response which represents a failure.
         *
         * @param request  Request for which is the response being created.
         * @param failure  Failure due to which processing of the request has failed.
         * @param Result Type of the response result.
         * @return Use case response ready to be dispatched.
         *
         * @see createSuccess
         */
        @NonNull fun <Result> createFailure(@NonNull request: UseCaseRequest, @NonNull failure: Failure): UseCaseResponse<Result>
    }

    /*
     * Methods =====================================================================================
     */

    /**
     * Returns the request this response is associated with.
     *
     * @param R Type of the expected request used to cast the request of this response to.
     * @return Request due to of which processing has been this response created.
     */
    @NonNull fun <R : UseCaseRequest> getRequest(): R

    /**
     * Checks whether this response is a success or not.
     *
     * @return `True` if this response represents a success of which result may be obtained via&nbsp;
     * [getResult], `false` if it is actually a failure.
     *
     * @see isFailure
     */
    fun isSuccess(): Boolean

    /**
     * Returns the result of successful processing of the associated request.
     *
     * If this response is a failure, the returned result should be a simple `EMPTY` result.
     *
     * @return Result associated with this response.
     *
     * @see isSuccess
     */
    @NonNull fun getResult(): Result

    /**
     * Checks whether this response is a failure or not.
     *
     * @return `True` if this response represents a failure which may be obtained via [getFailure],&nbsp;
     * `false` if it is actually a success.
     *
     * @see isSuccess
     */
    fun isFailure(): Boolean

    /**
     * Returns the failure due to which processing of the associated request has failed.
     *
     * If this response is a successful one, the returned failure should be a simple `NONE` failure.
     *
     * @return Failure associated with this response.
     *
     * @see isFailure
     */
    @NonNull fun getFailure(): Failure
}