/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton

import android.support.annotation.NonNull
import android.support.annotation.Nullable
import android.support.annotation.VisibleForTesting
import android.util.Log

import universum.studios.android.logging.Logger
import universum.studios.android.logging.SimpleLogger

/**
 * Utility class used by the **Arkhitekton** library for logging purpose.
 *
 * Custom [Logger] may be specified via [setLogger] which may be used to control logging outputs of
 * the library.
 *
 * Default logger used by this class has specified [Log.ASSERT] log level which means the the library
 * by default does not print out any logs.
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
object ArkhitektonLogging {

    /**
     * Default logger used by the library for logging purpose.
     */
    @VisibleForTesting internal val LOGGER = SimpleLogger(Log.ASSERT)

    /**
     * Logger to which is this logging utility class delegating all log related requests.
     */
    @JvmStatic @NonNull var logger: Logger = LOGGER

    /**
     * Delegates to [Logger.d].
     */
    @JvmStatic @JvmOverloads fun d(@NonNull tag: String, @NonNull msg: String, @Nullable error: Throwable? = null) = logger.d(tag, msg, error)

    /**
     * Delegates to [Logger.v].
     */
    @JvmStatic @JvmOverloads fun v(@NonNull tag: String, @NonNull msg: String, @Nullable error: Throwable? = null) = logger.v(tag, msg, error)

    /**
     * Delegates to [Logger.i].
     */
    @JvmStatic @JvmOverloads fun i(@NonNull tag: String, @NonNull msg: String, @Nullable error: Throwable? = null) = logger.i(tag, msg, error)

    /**
     * Delegates to [Logger.w].
     */
    @JvmStatic @JvmOverloads fun w(@NonNull tag: String, @NonNull msg: String = "", @Nullable error: Throwable? = null) = logger.w(tag, msg, error)

    /**
     * Delegates to [Logger.e].
     */
    @JvmStatic @JvmOverloads fun e(@NonNull tag: String, @NonNull msg: String = "", @Nullable error: Throwable? = null) = logger.e(tag, msg, error)

    /**
     * Delegates to [Logger.wtf].
     */
    @JvmStatic @JvmOverloads fun wtf(@NonNull tag: String, @NonNull msg: String = "", @Nullable error: Throwable? = null) = logger.wtf(tag, msg, error)
}