/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.util

import android.support.annotation.NonNull

/**
 * todo:
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @constructor Creates a new instance of Description with the given `builder's` configuration.
 * @param builder The builder with configuration for the new description.
 */
class Description internal constructor(@NonNull builder: Builder) : Cloneable {

    /**
     * Contract for [Description] element.
     */
    companion object Contract {

        /**
         * Empty description instance.
         */
        @NonNull internal val EMPTY = Builder().apply { empty = true }.build()

        /**
         * Returns an instance of [Description] without any descriptive properties.
         *
         * @return Description ready to be associated with a desired element.
         */
        @JvmStatic @NonNull fun empty() = EMPTY

        /**
         * Returns a new instance of description [Builder].
         *
         * @return Builder ready to be used to build instance of a desired description.
         */
        @JvmStatic @NonNull fun builder() = Builder()
    }

    /**
     * Boolean flag indicating whether this is an empty instance of description or not.
     */
    private val empty = builder.empty

    /**
     * Title text specified for this description.
     */
    private val title = builder.title

    /**
     * Summary text specified for this description.
     */
    private val summary = builder.summary

    /**
     * todo:
     *
     * @return This description's title.
     */
    @NonNull fun getTitle() = title

    /**
     * todo:
     *
     * @return This description's summary.
     */
    @NonNull fun getSummary() = summary

    /*
     */
    override fun toString(): String {
        return when(empty) {
            true -> "{}"
            else -> "{title: '$title', summary: '$summary'}"
        }
    }

    /*
     */
    override public fun clone() = super.clone() as Description

    /**
     * A builder which may be used to build instances of [Description].
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    class Builder internal constructor() : InstanceBuilder<Description> {

        /**
         * See [Description.empty].
         */
        internal var empty = false

        /**
         * See [Description.getTitle].
         */
        var title = ""

        /**
         * See [Description.getSummary].
         */
        var summary = ""

        /**
         * Specifies a [title][Description.getTitle] text for description.
         *
         * @param title The desired title text for new description.
         * @return This builder to allow methods chaining.
         * @see summary
         */
        fun title(title: String) = apply { this.title = title }

        /**
         * Specifies a [summary][Description.getSummary] text for description.
         *
         * @param summary The desired summary text for new description.
         * @return This builder to allow methods chaining.
         * @see title
         */
        fun summary(summary: String) = apply { this.summary = summary }

        /*
         */
        @NonNull override fun build() = Description(this)
    }
}