/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.presentation

import android.arch.lifecycle.Lifecycle
import android.arch.lifecycle.Lifecycle.State.CREATED
import android.arch.lifecycle.Lifecycle.State.DESTROYED
import android.arch.lifecycle.Lifecycle.State.INITIALIZED
import android.arch.lifecycle.Lifecycle.State.RESUMED
import android.arch.lifecycle.Lifecycle.State.STARTED
import android.arch.lifecycle.LifecycleObserver
import android.arch.lifecycle.OnLifecycleEvent
import universum.studios.android.arkhitekton.ArkhitektonLogging
import universum.studios.android.arkhitekton.view.View
import universum.studios.android.arkhitekton.view.ViewModel
import java.util.concurrent.atomic.AtomicBoolean

/**
 * A [Presenter] base implementation ... todo:
 *
 * @param V  Type of the view todo:
 * @param VM Type of the view model todo:
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @constructor Creates a new instance of BaseViewPresenter with the given `viewModel`.
 * @param viewModel The view model to be used by the presenter to persist view's state.
 */
abstract class BasePresenter<V : View<VM>, VM : ViewModel> protected constructor(private val viewModel: VM) : Presenter<V> {

    /*
	 * Companion ===================================================================================
	 */

    /**
     */
    companion object {

        /**
         * Log TAG.
         */
        internal const val TAG = "BasePresenter"

        /**
         * todo:
         */
        const val FEATURE_OBSERVE_VIEW_LIFECYCLE = 0x00000001

        /**
         * todo:
         *
         * @author Martin Albedinsky
         * @since 1.0
         */
        /*@IntDef(flag = true, value = {
            FEATURE_OBSERVE_VIEW_LIFECYCLE
        })*/
        @Target(AnnotationTarget.VALUE_PARAMETER, AnnotationTarget.FUNCTION)
        @Retention(AnnotationRetention.SOURCE)
        @MustBeDocumented
        annotation class Features
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /**
     * todo:
     */
    private var features = 0

    /**
     * todo:
     */
    private var view: V? = null

    /**
     * todo:
     */
    private val viewLifecycleActionRegistry = LifecycleActionRegistry(this)

    /**
     * todo:
     */
    private val destroyed = AtomicBoolean(false)

    /*
     * Initialization ==============================================================================
     */

    /*
     * Methods =====================================================================================
     */

    /**
     * todo:
     *
     * @param
     * @return
     */
    protected fun requestFeature(@Features feature: Int): Int {
        if (feature or FEATURE_OBSERVE_VIEW_LIFECYCLE == FEATURE_OBSERVE_VIEW_LIFECYCLE) {
            if (this !is LifecycleObserver) throw UnsupportedOperationException("${javaClass.name} cannot observe view's lifecycle because it does not implement LifecycleObserver interface!")
        }
        return features or feature
    }

    /**
     * todo:
     *
     * @param
     * @return
     */
    protected fun removeFeature(@Features feature: Int) = features and feature.inv()

    /**
     * todo:
     *
     * @param
     * @return
     */
    protected fun hasFeature(@Features feature: Int) = features and feature == feature

    /**
     * Returns the name of this presenter.
     *
     * By default, this is a value produced by [toString].
     *
     * @return This presenter's name.
     */
    fun getName() = toString()

    /**
     */
    override fun toString(): String = javaClass.simpleName

    /**
     * Returns the view model associated with this presenter.
     *
     * @return View model which this presenter uses to persist view's state.
     */
    protected fun getViewModel() = viewModel

    /**
     */
    override fun attachView(view: V) {
        if (this.view != null) {
            throw IllegalStateException("Already has view attached!")
        }
        this.view = view
        if (this is LifecycleObserver) {
            view.lifecycle.addObserver(viewLifecycleActionRegistry)
        }
        onViewAttached(view)
    }

    /**
     * todo:
     *
     * @param view
     */
    protected open fun onViewAttached(view: V) {
        if (hasFeature(FEATURE_OBSERVE_VIEW_LIFECYCLE)) {
            view.lifecycle.addObserver(this as LifecycleObserver)
        }
    }

    /**
     * todo:
     *
     * @return
     */
    protected fun isViewAttached() = view != null

    /**
     * todo:
     */
    private fun assertViewAttached() {
        view ?: throw IllegalStateException("No view attached!")
    }

    /**
     * todo:
     *
     * @return
     */
    protected fun getViewLifecycleCurrentState() = getView().lifecycle.currentState

    /**
     * todo:
     *
     * @param lifecycleAction
     */
    protected fun performViewLifecycleAction(lifecycleAction: LifecycleAction): Boolean {
        if (isViewAttached() && getViewLifecycleCurrentState().isAtLeast(lifecycleAction.stateCondition)) {
            lifecycleAction.run()
            return true;
        } else {
            registerViewLifecycleAction(
                    LifecycleActionRegistry.resolveEventForState(lifecycleAction.stateCondition),
                    lifecycleAction
            )
            return false;
        }
    }

    /**
     * todo:
     *
     * @param lifecycleEvent
     * @param action
     */
    protected fun registerViewLifecycleAction(lifecycleEvent: Lifecycle.Event, lifecycleAction: LifecycleAction)
            = viewLifecycleActionRegistry.registerActionForEvent(lifecycleEvent, lifecycleAction)

    /**
     * todo:
     *
     * @return
     * @throws IllegalStateException todo:
     */
    protected fun getView(): V {
        assertViewAttached()
        return view!!
    }

    /**
     */
    override fun detachView(view: V) {
        if (this.view == null) {
            return
        }
        if (this.view !== view) {
            throw IllegalArgumentException("Cannot detach view that is of different reference.")
        }
        this.view = null
        if (this is LifecycleObserver) {
            view.lifecycle.removeObserver(viewLifecycleActionRegistry)
        }
        onViewDetached(view)
    }

    /**
     * todo:
     *
     * @param view
     */
    protected open fun onViewDetached(view: V) {
        if (hasFeature(FEATURE_OBSERVE_VIEW_LIFECYCLE)) {
            view.lifecycle.removeObserver(this as LifecycleObserver)
        }
    }

    /**
     */
    override fun destroy() {
        if (destroyed.get()) {
            return
        }
        viewLifecycleActionRegistry.clearAllActions()
        onDestroy()
        destroyed.set(true)
    }

    /**
     * todo:
     */
    protected fun onDestroy() {
        // todo:
    }

    /*
     * Inner classes ===============================================================================
     */

    /**
     * todo:
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    abstract class LifecycleAction : Runnable {

        /**
         */
        companion object {

            /**
             * todo:
             */
            @JvmStatic
            val DEFAULT_PRIORITY = 0
        }

        /**
         * todo:
         */
        internal var priority = DEFAULT_PRIORITY

        /**
         * todo:
         */
        internal var stateCondition = Lifecycle.State.INITIALIZED

        /**
         * todo:
         */
        fun withPriority(priority: Int): LifecycleAction {
            this.priority = priority
            return this
        }

        /**
         * todo:
         */
        fun withStateCondition(lifecycleState: Lifecycle.State): LifecycleAction {
            this.stateCondition = lifecycleState
            return this
        }
    }

    /**
     * todo:
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    interface LifecycleActionSubscription {

        /**
         * todo:
         */
        fun cancel()
    }

    /**
     *
     */
    private class LifecycleActionSubscriptionImpl : LifecycleActionSubscription {

        /**
         */
        override fun cancel() {
            // todo:
        }
    }

    /**
     * todo:
     */
    internal class LifecycleActionRegistry(internal val parent: BasePresenter<*, *>) : LifecycleObserver {

        /**
         */
        companion object {

            /**
             * todo:
             *
             * @param state
             * @return
             */
            fun resolveEventForState(state: Lifecycle.State): Lifecycle.Event {
                return when (state) {
                    INITIALIZED -> Lifecycle.Event.ON_ANY
                    CREATED -> Lifecycle.Event.ON_CREATE
                    STARTED -> Lifecycle.Event.ON_START
                    RESUMED -> Lifecycle.Event.ON_RESUME
                    DESTROYED -> Lifecycle.Event.ON_DESTROY
                    else -> Lifecycle.Event.ON_ANY
                }
            }
        }

        /**
         * todo:
         */
        private val onCreateActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onStartActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onResumeActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onPauseActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onStopActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         */
        private val onDestroyActions: MutableList<LifecycleAction> by lazy { mutableListOf<LifecycleAction>() }

        /**
         * todo:
         *
         * @throws IllegalArgumentException
         */
        fun registerActionForEvent(lifecycleEvent: Lifecycle.Event, lifecycleAction: LifecycleAction): LifecycleActionSubscription {
            when (lifecycleEvent) {
                Lifecycle.Event.ON_CREATE -> this.onCreateActions.add(lifecycleAction)
                Lifecycle.Event.ON_START -> this.onStartActions.add(lifecycleAction)
                Lifecycle.Event.ON_RESUME -> this.onResumeActions.add(lifecycleAction)
                Lifecycle.Event.ON_PAUSE -> this.onPauseActions.add(lifecycleAction)
                Lifecycle.Event.ON_STOP -> this.onStopActions.add(lifecycleAction)
                Lifecycle.Event.ON_DESTROY -> this.onDestroyActions.add(lifecycleAction)
                Lifecycle.Event.ON_ANY -> throw IllegalArgumentException("Cannot register lifecycle action for event($lifecycleEvent)!")
            }
            ArkhitektonLogging.d(parent.getName(), "Registered lifecycle action for event($lifecycleEvent).")
            // todo: associate subscriptionn with action ...
            return LifecycleActionSubscriptionImpl()
        }

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_CREATE)
        fun onViewCreated() = runActionsForViewState("CREATED", onCreateActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_START)
        fun onViewStarted() = runActionsForViewState("STARTED", onStartActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_RESUME)
        fun onViewResumed() = runActionsForViewState("RESUMED", onResumeActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_PAUSE)
        fun onViewPaused() = runActionsForViewState("PAUSED", onPauseActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
        fun onViewStop() = runActionsForViewState("STOPPED", onStopActions)

        /**
         * todo:
         */
        @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
        fun onViewDestroyed() = runActionsForViewState("DESTROYED", onDestroyActions)

        /**
         * todo:
         */
        private fun runActionsForViewState(viewState: String, actions: MutableList<LifecycleAction>) {
            ArkhitektonLogging.d(parent.getName(), createLogMessageForLifecycleActions(viewState, actions.size))
            if (!actions.isEmpty()) {
                actions.forEach { it.run() }
                actions.clear()
            }
        }

        /**
         * todo:
         *
         * @param viewState
         * @param actionsCount
         * @return
         */
        private fun createLogMessageForLifecycleActions(viewState: String, actionsCount: Int): String {
            var message = "View $viewState."
            if (actionsCount == 0) {
                message += " No registered actions to run."
            } else {
                message += " Running registered actions in count($actionsCount)."
            }
            return message
        }

        /**
         * todo:
         */
        fun clearAllActions() {
            onCreateActions.clear()
            onStartActions.clear()
            onResumeActions.clear()
            onPauseActions.clear()
            onStopActions.clear()
            onDestroyActions.clear()
        }
    }
}