/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.interaction.usecase;

import android.support.annotation.NonNull;

import universum.studios.android.arkhitekton.util.Failure;
import universum.studios.android.arkhitekton.util.InstanceBuilder;

/**
 * An implementation of {@link UseCaseResponse}.
 *
 * @author Martin Albedinsky
 */
final class UseCaseResponseImpl<Result> implements UseCaseResponse<Result> {

	/**
	 * Log TAG.
	 */
	@SuppressWarnings("unused")
	private static final String TAG = "UseCaseResponseImpl";

	/**
	 * Factory which may be used to create instances of {@link UseCaseResponse} implementation.
	 */
	static final Factory FACTORY = new FactoryImpl();

	/**
	 * Request associated with this response.
	 */
	private final UseCaseRequest request;

	/**
	 * Result which was produced by successful execution of {@link #request} by the associated use case.
	 * <p>
	 * If this response is a failure, this should be {@link UseCaseResult.Contract#empty()}.
	 */
	private final Result result;

	/**
	 * Failure which occurred during execution of the associated use case.
	 * <p>
	 * If this response is not a failure this should be {@link Failure.Contract#none()}.
	 */
	private final Failure failure;

	/**
	 * Creates a new instance of UseCaseResponseImpl with configuration provided by the specified
	 * <var>builder</var>
	 *
	 * @param builder Builder with
	 */
	UseCaseResponseImpl(@NonNull BuilderImpl<Result> builder) {
		this.request = builder.request;
		this.result = builder.result;
		this.failure = builder.failure;
	}

	/**
	 */
	@NonNull
	@Override
	@SuppressWarnings("unchecked")
	public final <R extends UseCaseRequest> R getRequest() {
		return (R) request;
	}

	/**
	 */
	@Override
	public final boolean isSuccess() {
		return !isFailure();
	}

	/**
	 */
	@NonNull
	@Override
	public Result getResult() {
		return result;
	}

	/**
	 */
	@Override
	public final boolean isFailure() {
		return !Failure.Contract.none().equals(failure);
	}

	/**
	 */
	@NonNull
	@Override
	public final Failure getFailure() {
		return failure;
	}

	/**
	 */
	@Override
	@SuppressWarnings("StringBufferReplaceableByString")
	public String toString() {
		final StringBuilder builder = new StringBuilder(64);
		builder.append(UseCaseResponse.class.getSimpleName());
		builder.append("{request: ");
		builder.append(request);
		builder.append(", isSuccess: ");
		builder.append(isSuccess());
		builder.append(", result: ");
		builder.append(result);
		builder.append(", isFailure: ");
		builder.append(isFailure());
		builder.append(", failure: ");
		builder.append(failure);
		return builder.append("}").toString();
	}

	/**
	 * A builder which may be used to create instances of {@link UseCaseResponseImpl}.
	 *
	 * @param <Result> Type of the result the response will carry.
	 */
	static final class BuilderImpl<Result> implements InstanceBuilder<UseCaseResponseImpl<Result>> {

		/**
		 * See {@link UseCaseResponseImpl#request}.
		 */
		final UseCaseRequest request;

		/**
		 * See {@link UseCaseResponseImpl#result}.
		 */
		@SuppressWarnings("unchecked")
		Result result = (Result) UseCaseResult.Contract.empty();

		/**
		 * See {@link UseCaseResponseImpl#failure}.
		 */
		Failure failure = Failure.Contract.none();

		/**
		 * Creates a new instance of BuilderImpl with given <var>request</var> the new response
		 * will be created with.
		 *
		 * @param request The request the new response to be associated with.
		 */
		BuilderImpl(@NonNull UseCaseRequest request) {
			this.request = request;
		}

		/**
		 * Specifies a result to be the new response created with as a successful response.
		 *
		 * @param result The desired result for the new response.
		 * @return This builder to allow methods chaining.
		 */
		BuilderImpl<Result> result(@NonNull Result result) {
			this.result = result;
			return this;
		}

		/**
		 * Specifies a failure to be the new response created with as a failure response.
		 *
		 * @param failure The desired failure for the new response.
		 * @return This builder to allow methods chaining.
		 */
		BuilderImpl<Result> failure(@NonNull Failure failure) {
			this.failure = failure;
			return this;
		}

		/**
		 */
		@NonNull
		@Override
		public UseCaseResponseImpl<Result> build() {
			return new UseCaseResponseImpl<>(this);
		}
	}

	/**
	 * Implementation of {@link Factory} which is used to create <b>success</b> or
	 * <b>failure</b> instances of {@link UseCaseResponseImpl}.
	 */
	private static final class FactoryImpl implements Factory {

		/**
		 */
		@NonNull
		@Override
		public <Result> UseCaseResponse<Result> createSuccess(@NonNull UseCaseRequest request, @NonNull Result result) {
			return new BuilderImpl<Result>(request).result(result).build();
		}

		/**
		 */
		@NonNull
		@Override
		public <Result> UseCaseResponse<Result> createFailure(@NonNull UseCaseRequest request, @NonNull Failure failure) {
			return new BuilderImpl<Result>(request).failure(failure).build();
		}
	}
}
