/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.interaction

import android.support.annotation.NonNull
import universum.studios.android.arkhitekton.util.InstanceBuilder

/**
 * Interface for architectural elements that provide interaction facade for set of [UseCases][UseCase].
 *
 * It is recommended that each interactor implementation provides its specific builder that may be
 * used to build instances of that interactor type and that such builder implement [Interactor.Builder]
 * for convenience.
 *
 * **Note that it is also recommended to not directly implement this interface as its signature may
 * change over time, but rather to inherit from its [BaseInteractor] implementation.**
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
interface Interactor {

    /*
	 * Companion ===================================================================================
	 */

    /**
     * Contract for [Interactor] element.
     */
    companion object Contract {

        /**
         * Dummy instance of interactor element.
         */
        @NonNull internal val DUMMY: Interactor = object : Interactor {}

        /**
         * Returns a dummy instance of [Interactor] which may be used in cases where an instance of
         * concrete Interactor implementation is not needed.
         *
         * @return Dummy interactor ready to be used.
         */
        @NonNull fun dummy() = DUMMY
    }

    /*
     * Methods =====================================================================================
     */

    /*
     * Inner classes ===============================================================================
     */

    /**
     * Interface for builders which may be used to build instances of [Interactor] implementations.
     *
     * @param T Type of the interactor of which instance this builder can build.
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    interface Builder<out T : Interactor> : InstanceBuilder<T>
}