/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton.control

import android.support.annotation.CallSuper
import android.support.annotation.NonNull
import universum.studios.android.arkhitekton.ArkhitektonLogging
import universum.studios.android.arkhitekton.interaction.Interactor
import universum.studios.android.arkhitekton.presentation.Presenter
import java.util.concurrent.atomic.AtomicBoolean

/**
 * A [Controller] base implementation which is recommended to be inherited by concrete controller
 * implementations in order to take advantage of already implemented stubs of [Controller] interface
 * and also to be 'resistent' against future changes in that interface.
 *
 * Inheritance hierarchies should provide their specific builder which implements [BaseController.BaseBuilder]
 * for convenience.
 *
 * @param I Type of the interactor to which should the controller pass interaction requests to be processed.
 * @param P Type of the presenter to which should the controller dispatch appropriate responses for&nbsp;
 * processed requests and/or other interaction events.
 *
 * @author Martin Albedinsky
 * @since 1.0
 *
 * @constructor Creates a new instance of BaseController with the given `builder's` configuration.
 * @param builder The builder with configuration for the new controller.
 */
abstract class BaseController<out I : Interactor, out P : Presenter<*>> protected constructor(@NonNull builder: BaseBuilder<*, I, P>) : Controller<I, P> {

    /*
	 * Companion ===================================================================================
	 */

    /**
     */
    companion object {

        /**
         * Log TAG.
         */
        internal const val TAG = "BaseController"
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /**
     * Interactor instance associated with this controller. Controller should use this interactor
     * to process interaction requests and deliver responses for those requests to the [presenter].
     */
    private val interactor = builder.interactor

    /**
     * Presenter instance associated with this controller. Controller should dispatch to this
     * presenter any appropriate responses for requests processed by the [interactor] so the presenter
     * may present their results by means that are relevant to it.
     */
    private val presenter = builder.presenter

    /**
     * Boolean flag indicating whether this controller is active or not. If controller is active, it
     * should handle all interaction events that are dispatched to it and respond to them accordingly.
     */
    private val active = AtomicBoolean(false)

    /**
     * Boolean flag indicating whether this controller is enabled or not. If controller is temporarily
     * disabled, it should not handle any interaction events until it is enabled again.
     */
    private val enabled = AtomicBoolean(true)

    /**
     * Boolean flag indicating whether this controller is already destroyed or not. `True` if
     * [deactivate] has been already called for this controller instance.
     */
    private val destroyed = AtomicBoolean(false)

    /*
     * Initialization ==============================================================================
     */

    /*
     * Methods =====================================================================================
     */

    /*
     */
    override fun toString() = "${getName()}{active: ${active.get()}, enabled: ${enabled.get()}}"

    /**
     * Returns the name of this controller.
     *
     * @return This controller's name.
     */
    @NonNull fun getName(): String = javaClass.simpleName

    /*
     */
    override fun activate() {
        if (destroyed.get()) {
            throw IllegalStateException("Deactivated controller cannot be activated again!")
        }
        if (active.get()) {
            return
        }
        this.active.set(true)
        onActivated()
        ArkhitektonLogging.d(getName(), "Activated.")
    }

    /**
     * Invoked whenever this controller instance is activated due to call to [activate] when it was
     * in inactive state.
     *
     * From now on, this controller is considered active and should handle all interaction events
     * and respond to them accordingly until it is deactivated via [deactivate] and [onDeactivated]
     * is invoked.
     *
     * **Also note that controller may be disabled via [setEnabled].
     */
    protected open fun onActivated() {
        // Inheritance hierarchies may start here subscriptions if appropriate.
    }

    /*
     */
    override fun isActive() = active.get()

    /**
     * Asserts that this controller is activated. If not an exception is thrown.
     *
     * @throws IllegalStateException If this controller instance is not activated.
     * @see activate
     */
    protected fun assertActivated() {
        if (!active.get()) throw IllegalStateException("Not activated!")
    }

    /*
     */
    override fun deactivate() {
        if (destroyed.get()) {
            return
        }
        if (active.get()) {
            this.active.set(false)
            onDeactivated()
            ArkhitektonLogging.d(getName(), "Deactivated.")
        }
        destroyed.set(true)
    }

    /**
     * Invoked whenever this controller instance is deactivated due to call to [deactivate] when it
     * was in active state.
     *
     * From now on, this controller is considered inactive and should not handle any interaction
     * events further as its associated interaction component is about to be destroyed (permanently).
     *
     * This implementation destroys the attached presenter via [Presenter.destroy].
     */
    @CallSuper protected open fun onDeactivated() {
        // Inheritance hierarchies should cancel here all subscriptions started in onActivated().
        presenter.destroy()
    }

    /*
     */
    override fun setEnabled(enabled: Boolean) {
        if (this.enabled.get() != enabled) {
            this.enabled.set(enabled)
            onEnabledChanged(enabled)
        }
    }

    /**
     * Invoked whenever enabled state of this controller changes as result of call to [setEnabled].
     *
     * @param enabled The current enabled state of this controller instance.
     */
    protected open fun onEnabledChanged(enabled: Boolean) {
        // May be implemented by the inheritance hierarchies.
    }

    /*
     */
    override fun isEnabled() = enabled.get()

    /*
     */
    override fun getInteractor() = interactor

    /*
     */
    override fun getPresenter() = presenter

    /*
     * Inner classes ===============================================================================
     */

    /**
     * Base implementation of [Controller.Builder] which should be used by [BaseController] inheritance
     * hierarchies.
     *
     * @param B Type of the builder used as return type for builder's chain-able methods.
     * @param I Type of the interactor of which instance may be associated with new controller.
     * @param P Type of the presenter of which instance may be associated with new controller.
     *
     * @author Martin Albedinsky
     * @since 1.0
     */
    abstract class BaseBuilder<B : BaseBuilder<B, I, P>, I : Interactor, P : Presenter<*>>
    protected constructor(
            /**
             * See [BaseController.interactor].
             */
            @NonNull internal val interactor: I,
            /**
             * See [BaseController.presenter].
             */
            @NonNull internal val presenter: P) : Controller.Builder<Controller<I, P>> {

        /**
         * Obtains reference to this builder instance which is used for purpose of methods chaining.
         *
         * @return This builder instance.
         */
        protected abstract val self: B

        /**
         * Applies the given change to this builder.
         *
         * @param change The change to be applied.
         * @return This builder to allow methods chaining.
         */
        protected fun applyChange(@NonNull change: () -> Unit): B { change(); return self }
    }
}