/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.arkhitekton

import android.support.annotation.NonNull
import android.util.Log

import universum.studios.android.logging.Logger
import universum.studios.android.logging.SimpleLogger

/**
 * Utility class used by the **Arkhitekton** library for logging purpose.
 *
 * Custom [Logger] may be specified via [setLogger] which may be used to control logging outputs of
 * the library.
 *
 * Default logger used by this class has specified [Log.ASSERT] log level which means the the library
 * by default does not print out any logs.
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
class ArkhitektonLogging private constructor() {

    /**
     */
    companion object {

        /**
         * Default logger used by the library for logging purpose.
         */
        private val LOGGER = SimpleLogger(Log.ASSERT)

        /**
         * Logger to which is this logging utility class delegating all log related requests.
         */
        @JvmStatic var logger: Logger? = LOGGER
            set(logger) {
                field = logger ?: LOGGER
            }

        /**
         * Delegates to [Logger.d].
         */
        fun d(@NonNull tag: String, @NonNull msg: String) {
            logger!!.d(tag, msg)
        }

        /**
         * Delegates to [Logger.d].
         */
        fun d(@NonNull tag: String, @NonNull msg: String, @NonNull tr: Throwable?) {
            logger!!.d(tag, msg, tr)
        }

        /**
         * Delegates to [Logger.v].
         */
        fun v(@NonNull tag: String, @NonNull msg: String) {
            logger!!.d(tag, msg)
        }

        /**
         * Delegates to [Logger.v].
         */
        fun v(@NonNull tag: String, @NonNull msg: String, @NonNull tr: Throwable?) {
            logger!!.v(tag, msg, tr)
        }

        /**
         * Delegates to [Logger.i].
         */
        fun i(@NonNull tag: String, @NonNull msg: String) {
            logger!!.i(tag, msg)
        }

        /**
         * Delegates to [Logger.i].
         */
        fun i(@NonNull tag: String, @NonNull msg: String, @NonNull tr: Throwable?) {
            logger!!.i(tag, msg, tr)
        }

        /**
         * Delegates to [Logger.w].
         */
        fun w(@NonNull tag: String, @NonNull msg: String) {
            logger!!.w(tag, msg)
        }

        /**
         * Delegates to [Logger.w].
         */
        fun w(@NonNull tag: String, @NonNull tr: Throwable?) {
            logger!!.w(tag, tr)
        }

        /**
         * Delegates to [Logger.w].
         */
        fun w(@NonNull tag: String, @NonNull msg: String, @NonNull tr: Throwable?) {
            logger!!.w(tag, msg, tr)
        }

        /**
         * Delegates to [Logger.e].
         */
        fun e(@NonNull tag: String, @NonNull msg: String) {
            logger!!.e(tag, msg)
        }

        /**
         * Delegates to [Logger.e].
         */
        fun e(@NonNull tag: String, @NonNull msg: String, @NonNull tr: Throwable?) {
            logger!!.e(tag, msg, tr)
        }

        /**
         * Delegates to [Logger.wtf].
         */
        fun wtf(@NonNull tag: String, @NonNull msg: String) {
            logger!!.wtf(tag, msg)
        }

        /**
         * Delegates to [Logger.wtf].
         */
        fun wtf(@NonNull tag: String, @NonNull tr: Throwable?) {
            logger!!.wtf(tag, tr)
        }

        /**
         * Delegates to [Logger.wtf].
         */
        fun wtf(@NonNull tag: String, @NonNull msg: String, @NonNull tr: Throwable?) {
            logger!!.wtf(tag, msg, tr)
        }
    }

    /**
     */
    init {
        // Not allowed to be instantiated publicly.
        throw UnsupportedOperationException()
    }
}