/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.analytics.screen

import android.support.annotation.NonNull
import universum.studios.android.analytics.event.BaseAnalyticsEvent

/**
 * Analytics event which may be used to identify that a specific **screen** has been **viewed** by a
 * user.
 *
 * Instances of this event may be created via [ScreenViewEvent.Builder] and desired attribute values
 * may be supplied via predefined builder's chainable methods.
 *
 * @author Martin Albedinsky
 *
 * @constructor Creates a new instance of ScreenViewEvent with the given `builder's` configuration.
 * @param builder The builder with configuration for the new event.
 * @see ScreenViewEvent.builder
 */
class ScreenViewEvent internal constructor(@NonNull builder: Builder) : BaseAnalyticsEvent(builder) {

    /*
     * Companion ===================================================================================
     */

    /**
     */
    companion object {

        /**
         * Type uniquely identifying [ScreenViewEvent].
         */
        internal const val TYPE = "Screen View"

        /**
         * Returns a new instance of screen view event [Builder].
         *
         * @return Builder ready to be used to build instance of a desired event.
         */
        @JvmStatic @NonNull fun builder() = Builder()
    }

    /**
     * Defines general attributes for which may be supplied theirs corresponding values via [ScreenViewEvent].
     *
     * @author Martin Albedinsky
     */
    class Attribute private constructor() {

        /**
         */
        companion object {

            /**
             * Attribute for supplying name of the screen that has been viewed.
             */
            const val NAME = "Name"
        }

        /**
         */
        init {
            // Not allowed to be instantiated publicly.
            throw UnsupportedOperationException()
        }
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /*
     * Initialization ==============================================================================
     */

    /*
     * Methods =====================================================================================
     */

    /*
     * Inner classes ===============================================================================
     */

    /**
     * A builder which may be used to build instances of [ScreenViewEvent].
     *
     * @author Martin Albedinsky
     *
     * @constructor Creates a new instance of Builder with [TYPE] used as `eventType`.
     */
    class Builder internal constructor() : BaseBuilder<Builder, ScreenViewEvent>(TYPE) {

        /*
         */
        override val self = this

        /**
         * Specifies a name of the screen that has been viewed.
         *
         * This value is **required** and it is associated with [NAME][Attribute.NAME] attribute key.
         *
         * @return This builder to allow methods chaining.
         */
        fun screenName(@NonNull name: String) = putAttribute(Attribute.NAME, name)

        /*
         */
        @NonNull override fun build(): ScreenViewEvent {
            assertHasAttributeValues(Attribute.NAME)
            return ScreenViewEvent(this)
        }
    }
}