/*
 * =================================================================================================
 *                             Copyright (C) 2018 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.analytics

import android.app.Activity
import android.app.Application
import android.app.Application.ActivityLifecycleCallbacks
import android.os.Bundle
import android.support.annotation.NonNull

import universum.studios.android.analytics.ScreenViewAnalytics.Companion.install
import universum.studios.android.analytics.event.ScreenViewEvent

/**
 * Analytics for reporting of screen related events. May be installed for a desired [Application]
 * via [install].
 *
 * **Reported events:**
 * - [ScreenViewEvent]
 *
 * @author Martin Albedinsky
 */
class ScreenViewAnalytics private constructor() {

    // todo: allow to build this analytics via builder where we could specify set of destinations
    // todo: where should be ScreenViewEvents delivered ...

    /*
     * Companion ===================================================================================
     */

    /**
     */
    companion object {

        /**
         * Name of the destination where will be all screen view events logged.
         */
        internal const val DESTINATION = FabricDestination.NAME

        /**
         * Installs ScreenViewAnalytics for the specified `application`.
         *
         * @param application The desired application for which to install screen view analytics.
         * @param analytics The desired analytics facade instance that should be used for events logging.
         */
        @JvmStatic fun install(@NonNull application: Application, @NonNull analytics: AnalyticsFacade) {
            application.registerActivityLifecycleCallbacks(ActivityLifecycleListener(analytics))
        }
    }

    /*
	 * Interface ===================================================================================
	 */

    /*
     * Members =====================================================================================
     */

    /*
     * Initialization ==============================================================================
     */

    /*
     * Methods =====================================================================================
     */

    /*
     * Inner classes ===============================================================================
     */

    /**
     * A listener which listens for activity lifecycle events, particularly *onStarted(...)* event,
     * and logs [ScreenViewEvent] for the appropriate activity along with its **analytic name**.
     *
     * @see AnalyticsScreen.getAnalyticName
     */
    private class ActivityLifecycleListener(
            /**
             * The desired analytics facade instance that should be used for events logging.
             */
            private val analytics: AnalyticsFacade) : ActivityLifecycleCallbacks {

        /*
         */
        override fun onActivityPaused(activity: Activity) {}

        /*
         */
        override fun onActivityResumed(activity: Activity) {}

        /*
         */
        override fun onActivityStarted(activity: Activity) {
            if (activity is AnalyticsScreen) {
                analytics.logEvent(ScreenViewEvent.builder().addDestination(DESTINATION).screenName(activity.getAnalyticName()).build())
            }
        }

        /*
         */
        override fun onActivityDestroyed(activity: Activity) {}

        /*
         */
        override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle?) {}

        /*
         */
        override fun onActivityStopped(activity: Activity) {}

        /*
         */
        override fun onActivityCreated(activity: Activity?, savedInstanceState: Bundle?) {}
    }
}