(ns truckerpath.clj-config.core
  (:require [clojure.edn :as edn]
            [clojure.tools.logging :as log])
  (:refer-clojure :exclude [reset! get get-in]))

(def ^:private app-config (atom nil))

(defn read-edn
  "Reads file and parse it assuming it's EDN format."
  [filename]
  (log/info "Reading EDN file" filename)
  (edn/read-string (slurp filename)))

(defn load!
  "Reads EDN file and store it in app-config."
  [filename]
  (log/info "Loading config from" filename)
  (let [config (read-edn filename)]
    (clojure.core/reset! app-config config)))

(defn reset!
  "Resets app-config to a given one for testing and more complex operations
   (like merging two configs)."
  [config]
  (log/info "Reseting config")
  (clojure.core/reset! app-config config))

(defn get-in
  "Returns the value in a nested config. Return nil if the key is not present."
  [[& keys]]
  (if @app-config
    (let [missing-value :##unknown-missing-value##
          value (clojure.core/get-in @app-config keys missing-value)]
      (if (not= value missing-value)
        value
        (log/error "Unable to get value for" keys "- config doesn't have such key")))
    (log/error "Unable to get value for" keys "- config isn't initialized yet.")))

(defn get
  "Returns the value in config. Return nil if the key is not present."
  [key]
  (get-in [key]))

(defn config
  "Returns whole config."
  []
  @app-config)
