package threads.webrtc;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.webrtc.IceCandidate;
import org.webrtc.SessionDescription;

import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkNotNull;


public class RTCClient implements RTCSession.Listener {

    private final SignalingEvents events;
    private final PID user;


    public RTCClient(@NonNull PID user, @NonNull SignalingEvents events) {
        checkNotNull(user);
        checkNotNull(events);
        this.events = events;
        this.user = user;

        RTCSession.getInstance().setListener(this);
    }

    @Override
    public void busy(@NonNull PID pid) {
        checkNotNull(pid);
        events.onChannelBusy();
    }

    @Override
    public void accept(@NonNull PID pid, @Nullable String[] ices) {
        checkNotNull(pid);
        events.onChannelAccepted(ices);
    }

    @Override
    public void reject(@NonNull PID pid) {
        checkNotNull(pid);
        events.onChannelReject();
    }

    @Override
    public void offer(@NonNull PID pid, @NonNull String sdp) {
        checkNotNull(pid);
        checkNotNull(sdp);

        events.onChannelOffer(new SessionDescription(
                SessionDescription.Type.OFFER, sdp));
    }

    @Override
    public void answer(@NonNull PID pid,
                       @NonNull String sdp,
                       @NonNull String type) {
        checkNotNull(pid);
        checkNotNull(sdp);
        checkNotNull(type);
        events.onRemoteDescription(new SessionDescription(
                SessionDescription.Type.fromCanonicalForm(type), sdp));
    }

    @Override
    public void candidate(@NonNull PID pid,
                          @NonNull String sdp,
                          @NonNull String mid,
                          @NonNull String index) {
        checkNotNull(pid);
        checkNotNull(sdp);
        checkNotNull(mid);
        checkNotNull(index);
        events.onRemoteIceCandidate(new IceCandidate(
                mid,
                Integer.valueOf(index),
                sdp));
    }

    @Override
    public void candidate_remove(@NonNull PID pid,
                                 @NonNull String sdp,
                                 @NonNull String mid,
                                 @NonNull String index) {
        checkNotNull(pid);
        checkNotNull(sdp);
        checkNotNull(mid);
        checkNotNull(index);
        events.onRemoteIceCandidateRemoved(new IceCandidate(
                mid,
                Integer.valueOf(index),
                sdp));
    }

    @Override
    public void close(@NonNull PID pid) {
        checkNotNull(pid);
        events.onChannelClose();
    }

    @Override
    public void timeout(@NonNull PID pid) {
        checkNotNull(pid);
        events.onChannelTimeout();
    }


    public void sendOfferSdp(@NonNull Context context, @NonNull final SessionDescription sdp) {
        checkNotNull(context);
        checkNotNull(sdp);
        RTCSession.getInstance().emitSessionOffer(context, user, sdp);
    }


    public void sendAnswerSdp(@NonNull Context context, @NonNull final SessionDescription sdp) {
        checkNotNull(context);
        checkNotNull(sdp);
        RTCSession.getInstance().emitSessionAnswer(context, user, sdp);
    }


    public void sendLocalIceCandidate(@NonNull Context context, @NonNull final IceCandidate candidate) {
        checkNotNull(context);
        checkNotNull(candidate);
        RTCSession.getInstance().emitIceCandidate(context, user, candidate);
    }

    public void sendLocalIceCandidateRemovals(@NonNull Context context,
                                              @Nullable final IceCandidate[] candidates) {
        checkNotNull(context);
        if (candidates != null) {
            RTCSession.getInstance().emitIceCandidatesRemove(context,
                    user, candidates);
        }
    }


    /**
     * Callback interface for messages delivered on signaling channel.
     *
     * <p>Methods are guaranteed to be invoked on the UI thread of |activity|.
     */
    public interface SignalingEvents {


        /**
         * Callback fired once the room's signaling parameters
         * SignalingParameters are extracted.
         */
        void onChannelOffer(final SessionDescription sdp);

        /**
         * Callback fired once remote SDP is received.
         */
        void onRemoteDescription(final SessionDescription sdp);

        /**
         * Callback fired once remote Ice candidate is received.
         */
        void onRemoteIceCandidate(final IceCandidate candidate);

        /**
         * Callback fired once remote Ice candidate removals are received.
         */
        void onRemoteIceCandidateRemoved(final IceCandidate candidate);

        /**
         * Callback fired once channel is closed.
         */
        void onChannelClose();

        /**
         * Callback fired once channel is busy.
         */
        void onChannelBusy();

        void onChannelTimeout();

        void onChannelReject();

        void onChannelAccepted(@Nullable String[] ices);
    }


}
