package threads.webrtc;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;
import androidx.fragment.app.DialogFragment;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import threads.core.Preferences;
import threads.core.Singleton;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkNotNull;

public class RTCCallDialogFragment extends DialogFragment {

    public static final String TAG = RTCCallDialogFragment.class.getSimpleName();
    private static final String AES = "AES";
    private static final String CALLEE = "CALLEE";
    private static final String NAME = "NAME";
    private static final String VIDEO = "VIDEO";
    private final AtomicBoolean dismissed = new AtomicBoolean(false);
    private RTCCallDialogFragment.ActionListener mListener;
    private Context mContext;
    private RTCSoundPool soundPoolManager;

    static RTCCallDialogFragment newInstance(@NonNull String pid,
                                             @NonNull String name,
                                             @NonNull String aes,
                                             boolean video) {
        checkNotNull(pid);
        checkNotNull(aes);
        checkNotNull(name);
        Bundle bundle = new Bundle();
        bundle.putString(AES, aes);
        bundle.putString(CALLEE, pid);
        bundle.putString(NAME, name);
        bundle.putBoolean(VIDEO, video);

        RTCCallDialogFragment fragment = new RTCCallDialogFragment();
        fragment.setArguments(bundle);
        return fragment;
    }

    @Override
    public void onDetach() {
        super.onDetach();
        mContext = null;
        dismissed.set(true);
        soundPoolManager.release();
    }

    @Override
    public void onAttach(@NonNull Context context) {
        super.onAttach(context);
        mContext = context;
        try {
            mListener = (RTCCallDialogFragment.ActionListener) getActivity();
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
        soundPoolManager = RTCSoundPool.create(mContext, R.raw.outgoing, true);
    }


    @Override
    @NonNull
    public Dialog onCreateDialog(Bundle savedInstanceState) {

        Activity activity = getActivity();
        checkNotNull(activity);


        Bundle args = getArguments();
        checkNotNull(args);
        final String aesKey = args.getString(AES);
        checkNotNull(aesKey);
        final String pid = args.getString(CALLEE);
        checkNotNull(pid);
        final String name = args.getString(NAME);
        checkNotNull(name);
        AlertDialog.Builder builder = new AlertDialog.Builder(mContext);

        boolean video = args.getBoolean(VIDEO);
        if (video) {
            builder.setIcon(R.drawable.text_video_call);
        } else {
            builder.setIcon(R.drawable.text_phone);
        }
        builder.setTitle(getString(R.string.outgoing_call));
        builder.setPositiveButton(getString(R.string.abort), (dialog, which) -> {

            soundPoolManager.stop();
            mListener.abortCall(pid);
            dialog.dismiss();

        });

        builder.setMessage(getString(R.string.calling, name));

        AlertDialog dialog = builder.create();
        dialog.setCancelable(false);
        dialog.setCanceledOnTouchOutside(false);
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE);


        // hide the status bar and enter full screen mode
        checkNotNull(dialog.getWindow());
        dialog.getWindow().setFlags(
                WindowManager.LayoutParams.FLAG_FULLSCREEN,
                WindowManager.LayoutParams.FLAG_FULLSCREEN
        );

        // dismiss the keyguard
        dialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD);
        dialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED);
        dialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);


        // Set the content to appear under the system bars so that the content
        // doesn't resize when the system bars hide and show.
        dialog.getWindow().getDecorView().getRootView().setSystemUiVisibility(
                View.SYSTEM_UI_FLAG_IMMERSIVE
                        // Set the content to appear under the system bars so that the
                        // content doesn't resize when the system bars hide and show.
                        | View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                        | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                        | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                        // Hide the nav bar and status bar
                        | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                        | View.SYSTEM_UI_FLAG_FULLSCREEN);

        // immersive hack 1: set the dialog to not focusable (makes navigation ignore us adding the window)
        dialog.getWindow().addFlags(WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE);


        dialog.setOnShowListener((dialogInterface) -> {

            Button positive = dialog.getButton(AlertDialog.BUTTON_POSITIVE);

            positive.setCompoundDrawablesRelativeWithIntrinsicBounds(
                    R.drawable.phone_reject, 0, 0, 0);
            positive.setCompoundDrawablePadding(16);

        });
        dialog.setOnDismissListener((arg) -> dismissed.set(true));


        final int timeout = Preferences.getConnectionTimeout(mContext);
        final long start = System.currentTimeMillis();
        try {
            final IPFS ipfs = Singleton.getInstance(mContext).getIpfs();
            final PID host = Preferences.getPID(mContext);

            checkNotNull(host);
            if (ipfs != null) {
                ExecutorService executor = Executors.newSingleThreadExecutor();
                executor.submit(() -> {
                    try {
                        PID user = PID.create(pid);
                        boolean value = RTCSession.connectUserTimeout(mContext, user, aesKey);

                        if (value) {
                            if (video) {
                                RTCSession.getInstance().emitSessionVideoCall(mContext, host, user);
                            } else {
                                RTCSession.getInstance().emitSessionCall(mContext, host, user);
                            }

                        } else {
                            if (!dismissed.get()) {
                                mListener.failCall(pid);
                            }
                        }


                        long rest = System.currentTimeMillis() - (start - timeout);
                        if (rest > 0) {
                            try {
                                java.lang.Thread.sleep(rest);
                                if (!dismissed.get()) {
                                    mListener.timeoutCall(pid);
                                }
                            } catch (Throwable e) {
                                // ignore exception
                            }
                        }

                    } catch (Throwable e) {
                        // ignore exception
                    } finally {
                        soundPoolManager.stop();
                        dialog.dismiss();
                    }
                });
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


        soundPoolManager.play();

        return dialog;
    }

    @Override
    public void onDismiss(@NonNull DialogInterface dialog) {
        super.onDismiss(dialog);
        dismissed.set(true);
    }


    public interface ActionListener {

        void abortCall(@NonNull String pid);

        void timeoutCall(@NonNull String pid);

        void failCall(@NonNull String pid);

    }
}
