package threads.share;

import android.content.Context;
import android.util.Log;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.DiffUtil;
import androidx.recyclerview.widget.RecyclerView;

import com.bumptech.glide.Glide;

import java.util.ArrayList;
import java.util.List;

import threads.core.Preferences;
import threads.core.Singleton;
import threads.core.api.IPeer;
import threads.ipfs.IPFS;

public class PeersViewAdapter extends RecyclerView.Adapter<PeersViewAdapter.ViewHolder> {

    private static final String TAG = PeersViewAdapter.class.getSimpleName();
    private final List<IPeer> peers = new ArrayList<>();
    private final Context context;
    private final UsersViewAdapterListener listener;
    private final int timeout;

    public PeersViewAdapter(@NonNull Context context,
                            @NonNull PeersViewAdapter.UsersViewAdapterListener listener) {

        timeout = Preferences.getConnectionTimeout(context);
        this.context = context;
        this.listener = listener;
    }

    private static int getThemeTextColor(final Context context) {
        final TypedValue value = new TypedValue();
        context.getTheme().resolveAttribute(android.R.attr.textColorSecondary, value, true);
        return value.data;
    }

    @Override
    public int getItemViewType(int position) {
        return 0;
    }

    @Override
    @NonNull
    public PeersViewAdapter.UserViewHolder onCreateViewHolder(@NonNull ViewGroup parent,
                                                              int viewType) {

        View v;
        switch (viewType) {
            case 0:
                v = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.users, parent, false);
                return new PeersViewAdapter.UserViewHolder(v);
        }
        throw new RuntimeException("View type not supported.");
    }


    @Override
    public void onBindViewHolder(@NonNull ViewHolder holder, int position) {

        final IPeer peer = peers.get(position);

        if (holder instanceof PeersViewAdapter.UserViewHolder) {
            PeersViewAdapter.UserViewHolder userViewHolder = (PeersViewAdapter.UserViewHolder) holder;

            try {

                if (listener.generalActionSupport(peer)) {
                    userViewHolder.user_action.setVisibility(View.VISIBLE);
                    userViewHolder.user_action.setOnClickListener((v) ->
                            listener.invokeGeneralAction(peer)
                    );
                } else {
                    userViewHolder.user_action.setVisibility(View.GONE);
                }


                if (peer.isBlocked()) {
                    userViewHolder.progress_bar.setVisibility(View.GONE);
                    userViewHolder.user_alias.setCompoundDrawablesRelativeWithIntrinsicBounds(
                            R.drawable.blocked, 0, 0, 0);
                } else if (peer.isDialing()) {
                    userViewHolder.progress_bar.setVisibility(View.VISIBLE);
                    userViewHolder.user_alias.setCompoundDrawablesRelativeWithIntrinsicBounds(
                            R.drawable.text_lan_pending, 0, 0, 0);
                } else if (peer.isConnected()) {
                    userViewHolder.progress_bar.setVisibility(View.GONE);
                    userViewHolder.user_alias.setCompoundDrawablesRelativeWithIntrinsicBounds(
                            R.drawable.green_bubble, 0, 0, 0);
                } else {
                    userViewHolder.progress_bar.setVisibility(View.GONE);
                    userViewHolder.user_alias.setCompoundDrawablesRelativeWithIntrinsicBounds(
                            R.drawable.record, 0, 0, 0);
                }


                if (peer.getImage() != null) {
                    userViewHolder.user_image.setVisibility(View.VISIBLE);
                    IPFS ipfs = Singleton.getInstance(context).getIpfs();
                    IPFSData data = IPFSData.create(ipfs, peer.getImage(), timeout);
                    Glide.with(context).load(data).into(userViewHolder.user_image);
                } else {
                    userViewHolder.user_image.setVisibility(View.GONE);
                }

                userViewHolder.user_alias.setText(peer.getAlias());

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }

    }


    @Override
    public int getItemCount() {
        return peers.size();
    }


    public void updateData(@NonNull List<IPeer> peers) {

        final PeerDiffCallback diffCallback = new PeerDiffCallback(this.peers, peers);
        final DiffUtil.DiffResult diffResult = DiffUtil.calculateDiff(diffCallback);

        this.peers.clear();
        this.peers.addAll(peers);
        diffResult.dispatchUpdatesTo(this);
    }


    public interface UsersViewAdapterListener {

        void invokeGeneralAction(@NonNull IPeer peer);

        boolean generalActionSupport(@NonNull IPeer peer);


    }

    static class ViewHolder extends RecyclerView.ViewHolder {
        // each data item is just a string in this case
        final View view;

        ViewHolder(View v) {
            super(v);
            view = v;

        }
    }


    static class UserViewHolder extends ViewHolder {
        // each data item is just a string in this case
        final TextView user_alias;
        final ImageView user_action;
        final ImageView user_image;
        final ProgressBar progress_bar;

        UserViewHolder(View v) {
            super(v);
            user_image = v.findViewById(R.id.user_image);
            user_alias = v.findViewById(R.id.user_alias);
            user_action = v.findViewById(R.id.user_action);
            progress_bar = v.findViewById(R.id.progress_bar);

        }
    }
}
