package threads.share;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.media.AudioAttributes;
import android.media.AudioManager;
import android.net.Uri;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.util.Preconditions;

import com.google.gson.Gson;

import org.webrtc.IceCandidate;
import org.webrtc.SessionDescription;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import threads.core.IdentityService;
import threads.core.Preferences;
import threads.core.Singleton;
import threads.core.THREADS;
import threads.core.api.Addresses;
import threads.core.api.Content;
import threads.core.api.User;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class RTCSession {
    public static final String RTC_CHANNEL_ID = "RTC_CHANNEL_ID";
    public static final int RTC_CALL_ID = 100;
    private static final int PUBSUB_TIMEOUT = 50;
    private static final String TAG = RTCSession.class.getSimpleName();
    private static final Gson gson = new Gson();
    private static RTCSession INSTANCE = new RTCSession();
    private final AtomicBoolean busy = new AtomicBoolean(false);

    @Nullable
    private Listener listener = null;


    private RTCSession() {
    }

    public static void createRTCChannel(@NonNull Context context) {
        checkNotNull(context);
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
            try {
                CharSequence name = context.getString(R.string.rtc_channel_name);
                String description = context.getString(R.string.rtc_channel_description);
                int importance = NotificationManager.IMPORTANCE_HIGH;
                NotificationChannel mChannel = new NotificationChannel(RTC_CHANNEL_ID, name, importance);
                mChannel.setDescription(description);

                mChannel.setSound(
                        Uri.parse("android.resource://" + context.getPackageName() + "/" + R.raw.incoming),
                        new AudioAttributes.Builder().setContentType(
                                AudioAttributes.CONTENT_TYPE_SONIFICATION)
                                .setLegacyStreamType(AudioManager.STREAM_VOICE_CALL)
                                .setUsage(AudioAttributes.USAGE_VOICE_COMMUNICATION).build());

                NotificationManager notificationManager = (NotificationManager) context.getSystemService(
                        Context.NOTIFICATION_SERVICE);
                if (notificationManager != null) {
                    notificationManager.createNotificationChannel(mChannel);
                }

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }
        }
    }

    public static RTCSession getInstance() {
        return INSTANCE;
    }

    public static void handleContent(@NonNull Context context,
                                     @NonNull String aesKey,
                                     @NonNull PID senderPid,
                                     @NonNull Content content) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkNotNull(senderPid);
        checkNotNull(content);
        try {
            THREADS threadsAPI = Singleton.getInstance(context).getThreads();
            String est = content.get(Content.EST);
            RTCSession.Event type = RTCSession.Event.valueOf(est);
            switch (type) {
                case SESSION_VIDEO_CALL: {

                    if (RTCSession.getInstance().isBusy()) {
                        RTCSession.getInstance().emitSessionBusy(context, senderPid);
                    } else {

                        User user = threadsAPI.getUserByPID(senderPid);
                        String name = senderPid.getPid();
                        if (user != null) {
                            name = user.getAlias();
                        }

                        String adds = content.get(Content.ICES);
                        String[] ices = null;
                        if (adds != null) {
                            Addresses addresses = Addresses.toAddresses(adds);
                            ices = RTCSession.getInstance().turnUris(addresses);
                        }

                        RTCVideoCallActivity.createVideoCallNotification(context, aesKey,
                                senderPid.getPid(), name, ices);
                    }
                    break;
                }
                case SESSION_CALL: {

                    if (RTCSession.getInstance().isBusy()) {
                        RTCSession.getInstance().emitSessionBusy(context, senderPid);
                    } else {

                        User user = threadsAPI.getUserByPID(senderPid);
                        String name = senderPid.getPid();
                        if (user != null) {
                            name = user.getAlias();
                        }

                        String adds = content.get(Content.ICES);
                        String[] ices = null;
                        if (adds != null) {
                            Addresses addresses = Addresses.toAddresses(adds);
                            ices = RTCSession.getInstance().turnUris(addresses);
                        }

                        RTCCallActivity.createCallNotification(context, aesKey,
                                senderPid.getPid(), name, ices);
                    }
                    break;
                }
                case SESSION_TIMEOUT: {
                    RTCSession.getInstance().timeout(senderPid);
                    NotificationManager notificationManager = (NotificationManager)
                            context.getSystemService(Context.NOTIFICATION_SERVICE);
                    if (notificationManager != null) {
                        notificationManager.cancel(RTC_CALL_ID);
                    }
                    break;
                }
                case SESSION_BUSY: {
                    RTCSession.getInstance().busy(senderPid);
                    NotificationManager notificationManager = (NotificationManager)
                            context.getSystemService(Context.NOTIFICATION_SERVICE);
                    if (notificationManager != null) {
                        notificationManager.cancel(RTC_CALL_ID);
                    }
                    break;
                }
                case SESSION_ACCEPT: {
                    String[] ices = null;
                    String adds = content.get(Content.ICES);
                    if (adds != null) {
                        Addresses addresses = Addresses.toAddresses(adds);
                        ices = RTCSession.getInstance().turnUris(addresses);
                    }
                    RTCSession.getInstance().accept(senderPid, ices);
                    break;
                }
                case SESSION_REJECT: {
                    RTCSession.getInstance().reject(senderPid);
                    break;
                }
                case SESSION_OFFER: {
                    String sdp = content.get(Content.SDP);
                    checkNotNull(sdp);
                    RTCSession.getInstance().offer(senderPid, sdp);
                    break;
                }
                case SESSION_ANSWER: {
                    String sdp = content.get(Content.SDP);
                    checkNotNull(sdp);
                    String esk = content.get(Content.TYPE);
                    checkNotNull(esk);
                    RTCSession.getInstance().answer(senderPid, sdp, esk);
                    break;
                }
                case SESSION_CANDIDATE_REMOVE: {
                    String sdp = content.get(Content.SDP);
                    checkNotNull(sdp);
                    String mid = content.get(Content.MID);
                    checkNotNull(mid);
                    String index = content.get(Content.INDEX);
                    checkNotNull(index);
                    RTCSession.getInstance().candidate_remove(
                            senderPid, sdp, mid, index);
                    break;
                }
                case SESSION_CANDIDATE: {
                    String sdp = content.get(Content.SDP);
                    checkNotNull(sdp);
                    String mid = content.get(Content.MID);
                    checkNotNull(mid);
                    String index = content.get(Content.INDEX);
                    checkNotNull(index);
                    RTCSession.getInstance().candidate(senderPid, sdp, mid, index);
                    break;
                }
                case SESSION_CLOSE: {
                    RTCSession.getInstance().close(senderPid);
                    NotificationManager notificationManager = (NotificationManager)
                            context.getSystemService(Context.NOTIFICATION_SERVICE);
                    if (notificationManager != null) {
                        notificationManager.cancel(RTC_CALL_ID);
                    }
                    break;
                }
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
    }

    private static boolean shouldRun(long start, long timeout) {
        checkArgument(start > 0);
        checkArgument(timeout > 0);
        long passedTime = System.currentTimeMillis() - start;
        return passedTime < timeout;
    }

    public static boolean connectUserTimeout(@NonNull Context context,
                                             @NonNull PID user,
                                             @NonNull String aesKey) {
        checkNotNull(context);
        checkNotNull(user);
        checkNotNull(aesKey);

        final int timeout = Preferences.getConnectionTimeout(context);
        final long start = System.currentTimeMillis();
        boolean value = false;

        try {
            do {

                value = IdentityService.connectPeer(context, user, aesKey,
                        timeout, true, true, true);
            } while (shouldRun(start, timeout) && !value);
        } catch (Throwable e) {
            Log.e(TAG, e.getLocalizedMessage(), e);
        }
        return value;
    }

    @Nullable
    public String[] turnUris(@Nullable Addresses addresses) {
        if (addresses == null) {
            return null;
        }
        Collection<String> values = addresses.values();
        List<String> uris = new ArrayList<>();
        for (String address : values) {
            try {

                String[] parts = address.split("/");
                String protocol = parts[3];
                if (protocol.equals("udp")) { // TODO "turn" not working
                    uris.add("stun:" + parts[2] + ":" + parts[4] + "?transport=udp");
                } else {
                    uris.add("stun:" + parts[2] + ":" + parts[4]);
                }

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage());
            }
        }
        return uris.toArray(new String[uris.size()]);
    }

    public void emitSessionAnswer(@NonNull Context context,
                                  @NonNull PID user,
                                  @NonNull SessionDescription message) {
        checkNotNull(context);
        checkNotNull(user);

        checkNotNull(message);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {

            try {


                HashMap<String, String> map = new HashMap<>();
                map.put(Content.EST, Event.SESSION_ANSWER.name());
                map.put(Content.SDP, message.description);
                map.put(Content.TYPE, message.type.name());

                ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }
    }

    public void busy(@NonNull PID pid) {
        if (listener != null) {
            listener.busy(pid);
        }
    }

    public void accept(@NonNull PID pid, @Nullable String[] ices) {
        if (listener != null) {
            listener.accept(pid, ices);
        }
    }

    public void close(@NonNull PID pid) {
        if (listener != null) {
            listener.close(pid);
        }
    }

    public void setListener(@Nullable Listener listener) {
        this.listener = listener;
    }

    public void reject(@NonNull PID pid) {
        if (listener != null) {
            listener.reject(pid);
        }
    }

    public void offer(@NonNull PID pid, @NonNull String sdp) {
        if (listener != null) {
            listener.offer(pid, sdp);
        }
    }

    public void answer(@NonNull PID pid, @NonNull String sdp, @NonNull String type) {
        if (listener != null) {
            listener.answer(pid, sdp, type);
        }
    }

    public void candidate(@NonNull PID pid, @NonNull String sdp,
                          @NonNull String mid, @NonNull String index) {
        if (listener != null) {
            listener.candidate(pid, sdp, mid, index);
        }
    }

    public void candidate_remove(@NonNull PID pid, @NonNull String sdp,
                                 @NonNull String mid, @NonNull String index) {
        if (listener != null) {
            listener.candidate_remove(pid, sdp, mid, index);
        }
    }

    public void timeout(PID pid) {
        if (listener != null) {
            listener.timeout(pid);
        }
    }

    public void emitSessionOffer(@NonNull Context context,
                                 @NonNull PID user,
                                 @NonNull SessionDescription message) {
        checkNotNull(context);
        checkNotNull(user);

        checkNotNull(message);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {

            try {


                HashMap<String, String> map = new HashMap<>();
                map.put(Content.EST, Event.SESSION_OFFER.name());
                map.put(Content.SDP, message.description);

                ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }
    }

    public void emitSessionVideoCall(@NonNull Context context, @NonNull PID host, @NonNull PID user) {
        checkNotNull(context);
        checkNotNull(host);
        checkNotNull(user);
        try {
            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            THREADS threads = Singleton.getInstance(context).getThreads();
            Preconditions.checkNotNull(ipfs);
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_VIDEO_CALL.name());
            threads.core.api.PeerInfo peer = threads.getPeerInfoByPID(host);
            if (peer != null) {
                String addresses = Addresses.toString(peer.getAddresses());
                map.put(Content.ICES, addresses);
            }
            ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

    }

    public void emitSessionCall(@NonNull Context context, @NonNull PID host, @NonNull PID user) {
        checkNotNull(context);
        checkNotNull(host);
        checkNotNull(user);
        try {
            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            THREADS threads = Singleton.getInstance(context).getThreads();
            Preconditions.checkNotNull(ipfs);
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_CALL.name());
            threads.core.api.PeerInfo peer = threads.getPeerInfoByPID(host);
            if (peer != null) {
                String addresses = Addresses.toString(peer.getAddresses());
                map.put(Content.ICES, addresses);
            }
            ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

    }

    public void emitSessionBusy(@NonNull Context context,
                                @NonNull PID user) {
        checkNotNull(context);
        checkNotNull(user);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {

            try {

                HashMap<String, String> map = new HashMap<>();
                map.put(Content.EST, Event.SESSION_BUSY.name());
                ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }
    }

    public void emitSessionTimeout(@NonNull Context context,
                                   @NonNull PID user) {
        checkNotNull(context);
        checkNotNull(user);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {

            try {

                HashMap<String, String> map = new HashMap<>();
                map.put(Content.EST, Event.SESSION_TIMEOUT.name());
                ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }

    }

    public void emitSessionReject(@NonNull Context context,
                                  @NonNull PID user) {
        checkNotNull(context);
        checkNotNull(user);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            ExecutorService executor = Executors.newSingleThreadExecutor();
            executor.submit(() -> {
                try {

                    HashMap<String, String> map = new HashMap<>();
                    map.put(Content.EST, Event.SESSION_REJECT.name());
                    ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);

                } catch (Throwable e) {
                    Log.e(TAG, "" + e.getLocalizedMessage(), e);
                }
            });
        }

    }

    public void emitSessionAccept(@NonNull Context context,
                                  @NonNull PID host,
                                  @NonNull PID user) {
        checkNotNull(context);
        checkNotNull(host);
        checkNotNull(user);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        if (ipfs != null) {
            ExecutorService executor = Executors.newSingleThreadExecutor();
            executor.submit(() -> {
                try {

                    HashMap<String, String> map = new HashMap<>();
                    map.put(Content.EST, Event.SESSION_ACCEPT.name());

                    threads.core.api.PeerInfo peer = threads.getPeerInfoByPID(host);
                    if (peer != null) {
                        String addresses = Addresses.toString(peer.getAddresses());
                        map.put(Content.ICES, addresses);
                    }

                    ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);

                } catch (Throwable e) {
                    Log.e(TAG, "" + e.getLocalizedMessage(), e);
                }

            });
        }


    }

    public void emitSessionClose(@NonNull Context context,
                                 @NonNull PID user) {
        checkNotNull(context);
        checkNotNull(user);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {

            try {

                HashMap<String, String> map = new HashMap<>();
                map.put(Content.EST, Event.SESSION_CLOSE.name());
                ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }
    }

    public void emitIceCandidatesRemove(@NonNull Context context,
                                        @NonNull PID user,
                                        @NonNull IceCandidate[] candidates) {
        checkNotNull(context);
        checkNotNull(user);

        checkNotNull(candidates);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {

            try {

                for (IceCandidate candidate : candidates) {
                    HashMap<String, String> map = new HashMap<>();
                    map.put(Content.EST, Event.SESSION_CANDIDATE_REMOVE.name());
                    map.put(Content.SDP, candidate.sdp);
                    map.put(Content.MID, candidate.sdpMid);
                    map.put(Content.INDEX, String.valueOf(candidate.sdpMLineIndex));

                    ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);
                }

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }

    }

    public void emitIceCandidate(@NonNull Context context,
                                 @NonNull PID user,
                                 @NonNull IceCandidate candidate) {
        checkNotNull(context);
        checkNotNull(user);
        checkNotNull(candidate);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            try {

                HashMap<String, String> map = new HashMap<>();
                map.put(Content.EST, Event.SESSION_CANDIDATE.name());
                map.put(Content.SDP, candidate.sdp);
                map.put(Content.MID, candidate.sdpMid);
                map.put(Content.INDEX, String.valueOf(candidate.sdpMLineIndex));

                ipfs.pubsubPub(user.getPid(), gson.toJson(map), PUBSUB_TIMEOUT);

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }

    }

    public boolean isBusy() {
        return busy.get();
    }

    public void setBusy(boolean flag) {
        busy.set(flag);
    }

    public enum Event {
        SESSION_CALL, SESSION_VIDEO_CALL, SESSION_ACCEPT, SESSION_BUSY,
        SESSION_REJECT, SESSION_OFFER, SESSION_TIMEOUT,
        SESSION_ANSWER, SESSION_CANDIDATE,
        SESSION_CLOSE, SESSION_CANDIDATE_REMOVE
    }

    public interface Listener {
        void busy(@NonNull PID pid);

        void accept(@NonNull PID pid, @Nullable String[] ices);

        void reject(@NonNull PID pid);

        void offer(@NonNull PID pid, @NonNull String sdp);

        void answer(@NonNull PID pid, @NonNull String sdp, @NonNull String type);

        void candidate(@NonNull PID pid, @NonNull String sdp,
                       @NonNull String mid, @NonNull String index);

        void candidate_remove(@NonNull PID pid, @NonNull String sdp,
                              @NonNull String mid, @NonNull String index);

        void close(@NonNull PID pid);

        void timeout(@NonNull PID pid);
    }
}
