package threads.share;

import android.media.MediaDataSource;
import android.util.Log;

import androidx.annotation.NonNull;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import threads.ipfs.IPFS;
import threads.ipfs.api.CID;

import static androidx.core.util.Preconditions.checkNotNull;

public class IPFSMediaDataSource extends MediaDataSource {
    private static final String TAG = IPFSMediaDataSource.class.getSimpleName();
    private final long length;
    private RandomAccessFile memoryFile;


    IPFSMediaDataSource(long length) {
        this.length = length;
    }

    void init(@NonNull IPFS ipfs,
              @NonNull CID cid,
              @NonNull String key,
              @NonNull Callback listener,
              int timeout) {
        checkNotNull(ipfs);
        checkNotNull(cid);
        checkNotNull(key);
        checkNotNull(listener);

        ExecutorService executor = Executors.newSingleThreadExecutor();

        executor.submit(() -> {
            try {
                File file = new File(ipfs.getCacheDir(), cid.getCid());
                if (!file.exists()) {
                    ipfs.storeToFile(file, cid, key, timeout, true);
                }
                memoryFile = new RandomAccessFile(file, "r");
                listener.done();
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }
        });
    }


    private int read(long position, byte[] buffer, int offset, int size) throws IOException {
        memoryFile.seek(position);
        return memoryFile.read(buffer, offset, size);

    }


    @Override
    public int readAt(long position, byte[] buffer, int offset, int size) throws IOException {
        try {
            return read(position, buffer, offset, size);
        } catch (Throwable e) {
            throw new IOException(e);
        }

    }

    @Override
    public long getSize() {
        return length;
    }

    @Override
    public void close() {
        try {
            memoryFile.close();
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

    }

    public interface Callback {
        void done();
    }
}
