package threads.share;

import android.Manifest;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.SystemClock;
import android.util.Log;
import android.view.View;
import android.view.Window;
import android.widget.LinearLayout;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.app.ActivityCompat;
import androidx.core.app.NotificationCompat;
import androidx.core.app.TaskStackBuilder;
import androidx.core.content.ContextCompat;
import androidx.lifecycle.ViewModelProviders;

import com.google.android.material.floatingactionbutton.FloatingActionButton;

import org.webrtc.Camera1Enumerator;
import org.webrtc.Camera2Enumerator;
import org.webrtc.CameraEnumerator;
import org.webrtc.EglBase;
import org.webrtc.IceCandidate;
import org.webrtc.PeerConnection;
import org.webrtc.PeerConnectionFactory;
import org.webrtc.RendererCommon.ScalingType;
import org.webrtc.SessionDescription;
import org.webrtc.SurfaceViewRenderer;
import org.webrtc.VideoCapturer;
import org.webrtc.VideoFrame;
import org.webrtc.VideoSink;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import threads.core.Preferences;
import threads.core.Singleton;
import threads.core.mdl.EventViewModel;
import threads.ipfs.api.PID;
import threads.share.RTCAudioManager.AudioDevice;
import threads.share.RTCPeerConnection.PeerConnectionParameters;

import static androidx.core.util.Preconditions.checkNotNull;

public class RTCVideoCallActivity extends AppCompatActivity implements
        RTCClient.SignalingEvents,
        RTCPeerConnection.PeerConnectionEvents,
        RTCCallingDialogFragment.ActionListener,
        RTCCallDialogFragment.ActionListener {

    public static final String ACTION_INCOMING_CALL = "ACTION_INCOMING_CALL";
    public static final String ACTION_OUTGOING_CALL = "ACTION_OUTGOING_CALL";
    public static final String EXTRA_CAMERA2 = "CAMERA2";
    public static final String EXTRA_VIDEO_WIDTH = "VIDEO_WIDTH";
    public static final String EXTRA_VIDEO_HEIGHT = "VIDEO_HEIGHT";
    public static final String EXTRA_VIDEO_FPS = "VIDEO_FPS";
    public static final String EXTRA_VIDEO_BITRATE = "VIDEO_BITRATE";
    public static final String EXTRA_AUDIO_BITRATE = "AUDIO_BITRATE";
    private static final String TAG = RTCVideoCallActivity.class.getSimpleName();
    private static final String CALL_NAME = "CALL_NAME";
    private static final String CALL_PID = "CALL_PID";
    private static final String CALL_ICES = "CALL_ICES";
    private static final String INITIATOR = "INITIATOR";

    private static final int DISCONNECT_TIMEOUT = 2500;
    private static final int REQUEST_VIDEO_CAPTURE = 1;
    private static final int REQUEST_AUDIO_CAPTURE = 2;
    private static final int REQUEST_MODIFY_AUDIO_SETTINGS = 3;

    private final AtomicBoolean disconnect = new AtomicBoolean(false);
    private final ProxyVideoSink remoteProxyRenderer = new ProxyVideoSink();
    private final ProxyVideoSink localProxyVideoSink = new ProxyVideoSink();
    private final List<VideoSink> remoteSinks = new ArrayList<>();
    @Nullable
    private RTCPeerConnection peerConnectionClient;
    @Nullable
    private RTCClient appRtcClient;
    @Nullable
    private RTCAudioManager audioManager;
    private SurfaceViewRenderer pipRenderer;
    private SurfaceViewRenderer fullscreenRenderer;
    private PeerConnectionParameters peerConnectionParameters;
    private boolean callControlFragmentVisible = true;
    private boolean micEnabled = true;
    private boolean speakerEnabled = true;
    private boolean initiator;

    private long mLastClickTime = 0;
    private String callee;
    private String calleeName;

    private List<PeerConnection.IceServer> peerIceServers = new ArrayList<>();
    private LinearLayout fab_video_layout;
    private RTCCallDialogFragment callDialog;
    private FloatingActionButton fab_toggle_speaker;
    private FloatingActionButton fab_toggle_mic;
    private FloatingActionButton fab_switch_camera;
    private FloatingActionButton fab_hangup;

    public static void createVideoCallNotification(@NonNull Context context,
                                                   @NonNull String pid,
                                                   @NonNull String name,
                                                   @Nullable String[] ices) {

        NotificationCompat.Builder builder = new NotificationCompat.Builder(context,
                RTCSession.RTC_CHANNEL_ID);

        builder.setContentTitle(context.getString(R.string.incoming_video_call));
        builder.setContentText(context.getString(R.string.is_calling, name));
        builder.setSmallIcon(R.drawable.video);
        builder.setPriority(NotificationCompat.PRIORITY_HIGH);

        Intent intent = RTCVideoCallActivity.createIntent(
                context, pid, name, ices, false, false);
        intent.setAction(RTCVideoCallActivity.ACTION_INCOMING_CALL);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

        TaskStackBuilder stackBuilder = TaskStackBuilder.create(context);
        stackBuilder.addNextIntentWithParentStack(intent);


        int requestID = (int) System.currentTimeMillis();
        PendingIntent pendingIntent = stackBuilder.getPendingIntent(
                requestID, PendingIntent.FLAG_UPDATE_CURRENT);

        int timeout = Preferences.getConnectionTimeout(context);
        builder.setTimeoutAfter(timeout);
        builder.setContentIntent(pendingIntent);
        builder.setVisibility(NotificationCompat.VISIBILITY_PUBLIC);
        builder.setAutoCancel(true);

        builder.setCategory(NotificationCompat.CATEGORY_CALL);
        builder.setFullScreenIntent(pendingIntent, true);

        Notification notification = builder.build();

        NotificationManager notificationManager = (NotificationManager) context.getSystemService(
                Context.NOTIFICATION_SERVICE);
        if (notificationManager != null) {
            notificationManager.notify(RTCSession.RTC_CALL_ID, notification);
        }
    }


    public static Intent createIntent(@NonNull Context context,
                                      @NonNull String pid,
                                      @NonNull String name,
                                      @Nullable String[] ices,
                                      boolean initiator,
                                      boolean pubsubCheck) {
        checkNotNull(context);
        checkNotNull(pid);
        Intent intent = new Intent(context, RTCVideoCallActivity.class);
        intent.putExtra(RTCVideoCallActivity.CALL_PID, pid);
        intent.putExtra(RTCVideoCallActivity.CALL_NAME, name);
        intent.putExtra(RTCVideoCallActivity.INITIATOR, initiator);
        intent.putExtra(RTCVideoCallActivity.CALL_ICES, ices);

        intent.putExtra(RTCVideoCallActivity.EXTRA_CAMERA2, true);
        intent.putExtra(RTCVideoCallActivity.EXTRA_VIDEO_WIDTH, 1024);
        intent.putExtra(RTCVideoCallActivity.EXTRA_VIDEO_HEIGHT, 720);
        intent.putExtra(RTCVideoCallActivity.EXTRA_VIDEO_FPS, 30);
        intent.putExtra(RTCVideoCallActivity.EXTRA_VIDEO_BITRATE, 0);
        intent.putExtra(RTCVideoCallActivity.EXTRA_AUDIO_BITRATE, 0);

        return intent;
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        handleActionIntents(intent);
    }

    @Override
    protected void onResume() {
        super.onResume();
        hideSystemUI();
    }


    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        requestWindowFeature(Window.FEATURE_NO_TITLE);

        hideSystemUI();


        setContentView(R.layout.activity_video_call);


        final Intent intent = getIntent();
        callee = intent.getStringExtra(CALL_PID);
        calleeName = intent.getStringExtra(CALL_NAME);
        PID user = PID.create(callee);
        String[] ices = intent.getStringArrayExtra(CALL_ICES);

        // TODO maybe remove in the future
        PeerConnection.IceServer peerIceServer =
                PeerConnection.IceServer.builder("stun:stun.l.google.com:19302").createIceServer();
        peerIceServers.add(peerIceServer);

        if (ices != null) {
            for (String turn : ices) {
                peerIceServers.add(
                        PeerConnection.IceServer.builder(turn).createIceServer());
            }
        }

        initiator = intent.getBooleanExtra(INITIATOR, true);

        // Create UI controls.
        pipRenderer = findViewById(R.id.pip_video_view);

        fullscreenRenderer = findViewById(R.id.fullscreen_video_view);


        // Show/hide call control fragment on view click.
        fullscreenRenderer.setOnClickListener((view) -> toggleCallControls());

        remoteSinks.add(remoteProxyRenderer);

        final EglBase eglBase = EglBase.create();


        pipRenderer.init(eglBase.getEglBaseContext(), null);
        pipRenderer.setScalingType(ScalingType.SCALE_ASPECT_FIT);

        fullscreenRenderer.init(eglBase.getEglBaseContext(), null);
        fullscreenRenderer.setScalingType(ScalingType.SCALE_ASPECT_FILL);

        pipRenderer.setZOrderMediaOverlay(true);
        pipRenderer.setEnableHardwareScaler(true /* enabled */);
        fullscreenRenderer.setEnableHardwareScaler(false /* enabled */);
        // Start with local feed in fullscreen and swap it to the pip when the call is connected.
        setSwappedFeeds(true /* isSwappedFeeds */);


        int videoWidth = intent.getIntExtra(EXTRA_VIDEO_WIDTH, 0);
        int videoHeight = intent.getIntExtra(EXTRA_VIDEO_HEIGHT, 0);

        String audioCodec = Preferences.getAudioCodec(getApplicationContext());
        String videoCodec = Preferences.getVideoCodec(getApplicationContext());
        boolean audioProcessingEnabled =
                Preferences.isAudioProcessingEnabled(getApplicationContext());
        boolean openSlES = Preferences.isOpenSlESEnabled(getApplicationContext());
        boolean aec = Preferences.isAcousticEchoCancelerEnabled(getApplicationContext());
        boolean agc = Preferences.isAutomaticGainControlEnabled(getApplicationContext());
        boolean hns = Preferences.isHardwareNoiseSuppressorEnabled(getApplicationContext());

        peerConnectionParameters =
                new PeerConnectionParameters(
                        true,
                        false,
                        false,
                        videoWidth,
                        videoHeight,
                        intent.getIntExtra(EXTRA_VIDEO_FPS, 0),
                        intent.getIntExtra(EXTRA_VIDEO_BITRATE, 0),
                        videoCodec,
                        true,
                        false, // TODO check
                        intent.getIntExtra(EXTRA_AUDIO_BITRATE, 0),
                        audioCodec,
                        !audioProcessingEnabled,
                        false, // TODO check
                        openSlES,
                        !aec,
                        !agc,
                        !hns,
                        !agc,
                        false, null);


        appRtcClient = new RTCClient(user, this);


        // Create peer connection client.
        peerConnectionClient = new RTCPeerConnection(
                getApplicationContext(), eglBase, peerConnectionParameters, RTCVideoCallActivity.this);
        PeerConnectionFactory.Options options = new PeerConnectionFactory.Options();

        peerConnectionClient.createPeerConnectionFactory(options);


        audioManager = RTCAudioManager.create(getApplicationContext(), AudioDevice.SPEAKER_PHONE);

        audioManager.start((audioDevice, availableAudioDevices) -> {
            // This method will be called each time the number of available audio
            // devices has changed.

        });


        fab_video_layout = findViewById(R.id.fab_video_layout);


        fab_hangup = findViewById(R.id.fab_hangup);
        fab_hangup.setOnClickListener((view) -> {


            // mis-clicking prevention, using threshold of 1000 ms
            if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                return;
            }
            mLastClickTime = SystemClock.elapsedRealtime();

            RTCSession.getInstance().emitSessionClose(getApplicationContext(),
                    PID.create(callee));

            disconnect(DISCONNECT_TIMEOUT, true);

        });


        fab_switch_camera = findViewById(R.id.fab_switch_camera);
        fab_switch_camera.setOnClickListener((view) -> {


            // mis-clicking prevention, using threshold of 1000 ms
            if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                return;
            }
            mLastClickTime = SystemClock.elapsedRealtime();

            onCameraSwitch();

        });


        fab_toggle_speaker = findViewById(R.id.fab_toggle_speaker);
        fab_toggle_speaker.setOnClickListener((view) -> {


            // mis-clicking prevention, using threshold of 1000 ms
            if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                return;
            }
            mLastClickTime = SystemClock.elapsedRealtime();

            onToggleSpeaker();

        });

        if (speakerEnabled) {
            audioManager.setDefaultAudioDevice(AudioDevice.SPEAKER_PHONE);
            fab_toggle_speaker.setImageResource(R.drawable.volume_high);
        } else {
            audioManager.setDefaultAudioDevice(AudioDevice.EARPIECE);
            fab_toggle_speaker.setImageResource(R.drawable.volume_medium);
        }

        fab_toggle_mic = findViewById(R.id.fab_toggle_mic);
        fab_toggle_mic.setOnClickListener((view) -> {


            // mis-clicking prevention, using threshold of 1000 ms
            if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                return;
            }
            mLastClickTime = SystemClock.elapsedRealtime();

            boolean enabled = onToggleMic();
            if (enabled) {
                fab_toggle_mic.setImageResource(R.drawable.microphone);
            } else {
                fab_toggle_mic.setImageResource(R.drawable.microphone_off);
            }


        });


        EventViewModel eventViewModel = ViewModelProviders.of(this).get(EventViewModel.class);
        eventViewModel.getInfo().observe(this, (event) -> {
            try {
                if (event != null) {
                    Toast.makeText(
                            getApplicationContext(), event.getContent(), Toast.LENGTH_LONG).show();
                    eventViewModel.removeEvent(event);

                }
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        });


        handleActionIntents(intent);

        NotificationManager notificationManager = (NotificationManager) getSystemService(
                Context.NOTIFICATION_SERVICE);
        if (notificationManager != null) {
            notificationManager.cancel(RTCSession.RTC_CALL_ID);
        }

        onEnableFabs(false);
        Singleton singleton = Singleton.getInstance(getApplicationContext());
        Preferences.info(singleton.getThreads(), "");


    }

    public void onEnableFabs(boolean enabled) {
        fab_toggle_speaker.setEnabled(enabled);
        fab_hangup.setEnabled(enabled);
        fab_toggle_mic.setEnabled(enabled);
        fab_switch_camera.setEnabled(enabled);
    }


    private void handleActionIntents(Intent intent) {
        if (intent != null && intent.getAction() != null) {
            if (intent.getAction().equals(RTCVideoCallActivity.ACTION_INCOMING_CALL)) {
                String pid = intent.getStringExtra(RTCVideoCallActivity.CALL_PID);
                if (pid != null && !pid.isEmpty()) {
                    receiveUserCall();
                    intent.removeExtra(RTCVideoCallActivity.CALL_PID);
                }
            }
            if (intent.getAction().equals(RTCVideoCallActivity.ACTION_OUTGOING_CALL)) {
                String pid = intent.getStringExtra(RTCVideoCallActivity.CALL_PID);
                if (pid != null && !pid.isEmpty()) {
                    userCall();
                    intent.removeExtra(RTCVideoCallActivity.CALL_PID);
                }
            }
        }
    }

    private void userCall() {
        RTCSession.getInstance().setBusy(true);
        callDialog = RTCCallDialogFragment.newInstance(
                callee, calleeName, true);
        callDialog.show(getSupportFragmentManager(), RTCCallDialogFragment.TAG);
    }

    @Override
    public void onWindowFocusChanged(boolean hasFocus) {
        super.onWindowFocusChanged(hasFocus);
        hideSystemUI();
    }

    private void hideSystemUI() {
        getWindow().getDecorView().setSystemUiVisibility(
                View.SYSTEM_UI_FLAG_IMMERSIVE
                        // Set the content to appear under the system bars so that the
                        // content doesn't resize when the system bars hide and show.
                        | View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                        | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                        | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                        // Hide the nav bar and status bar
                        | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                        | View.SYSTEM_UI_FLAG_FULLSCREEN);
    }

    @Override
    public void onRequestPermissionsResult
            (int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        switch (requestCode) {

            case REQUEST_AUDIO_CAPTURE: {
                for (int i = 0, len = permissions.length; i < len; i++) {
                    if (grantResults[i] == PackageManager.PERMISSION_DENIED) {
                        Singleton singleton = Singleton.getInstance(getApplicationContext());
                        Preferences.info(singleton.getThreads(),
                                getString(R.string.permission_audio_denied));
                        RTCSession.getInstance().emitSessionClose(getApplicationContext(),
                                PID.create(callee));
                        disconnect(DISCONNECT_TIMEOUT, true);
                    }
                    if (grantResults[i] == PackageManager.PERMISSION_GRANTED) {
                        receiveUserCall();
                    }
                }

                break;
            }
            case REQUEST_MODIFY_AUDIO_SETTINGS: {
                for (int i = 0, len = permissions.length; i < len; i++) {
                    if (grantResults[i] == PackageManager.PERMISSION_DENIED) {
                        Singleton singleton = Singleton.getInstance(getApplicationContext());
                        Preferences.info(singleton.getThreads(),
                                getString(R.string.permission_audio_settings_denied));
                        RTCSession.getInstance().emitSessionClose(getApplicationContext(),
                                PID.create(callee));
                        disconnect(DISCONNECT_TIMEOUT, true);
                    }
                    if (grantResults[i] == PackageManager.PERMISSION_GRANTED) {
                        receiveUserCall();
                    }
                }

                break;
            }
            case REQUEST_VIDEO_CAPTURE: {
                for (int i = 0, len = permissions.length; i < len; i++) {
                    if (grantResults[i] == PackageManager.PERMISSION_DENIED) {
                        Singleton singleton = Singleton.getInstance(getApplicationContext());
                        Preferences.info(singleton.getThreads(),
                                getString(R.string.permission_camera_denied));
                        RTCSession.getInstance().emitSessionClose(getApplicationContext(),
                                PID.create(callee));
                        disconnect(DISCONNECT_TIMEOUT, true);
                    }
                    if (grantResults[i] == PackageManager.PERMISSION_GRANTED) {
                        receiveUserCall();
                    }
                }

                break;
            }
        }
    }

    public void receiveUserCall() {
        if (ContextCompat.checkSelfPermission(this,
                Manifest.permission.RECORD_AUDIO)
                != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(this,
                    new String[]{Manifest.permission.RECORD_AUDIO},
                    REQUEST_AUDIO_CAPTURE);
            return;
        }

        if (ContextCompat.checkSelfPermission(this,
                Manifest.permission.CAMERA)
                != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(this,
                    new String[]{Manifest.permission.CAMERA},
                    REQUEST_VIDEO_CAPTURE);
            return;
        }

        if (ContextCompat.checkSelfPermission(this,
                Manifest.permission.MODIFY_AUDIO_SETTINGS)
                != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(this,
                    new String[]{Manifest.permission.MODIFY_AUDIO_SETTINGS},
                    REQUEST_MODIFY_AUDIO_SETTINGS);
            return;
        }


        RTCCallingDialogFragment.newInstance(callee, calleeName, true)
                .show(getSupportFragmentManager(), RTCCallingDialogFragment.TAG);

    }


    private boolean useCamera2() {
        return Camera2Enumerator.isSupported(this) && getIntent().getBooleanExtra(EXTRA_CAMERA2, true);
    }


    @Nullable
    private VideoCapturer createCameraCapturer(CameraEnumerator enumerator) {
        final String[] deviceNames = enumerator.getDeviceNames();


        for (String deviceName : deviceNames) {
            if (enumerator.isFrontFacing(deviceName)) {
                VideoCapturer videoCapturer = enumerator.createCapturer(
                        deviceName, null);

                if (videoCapturer != null) {
                    return videoCapturer;
                }
            }
        }

        for (String deviceName : deviceNames) {
            if (!enumerator.isFrontFacing(deviceName)) {

                VideoCapturer videoCapturer = enumerator.createCapturer(
                        deviceName, null);

                if (videoCapturer != null) {
                    return videoCapturer;
                }
            }
        }

        return null;
    }


    @Override
    public void onStop() {
        super.onStop();

        // Don't stop the video when using screencapture to allow user to show other apps to the remote
        // end.
        if (peerConnectionClient != null) {
            peerConnectionClient.stopVideoSource();
        }

    }

    @Override
    public void onStart() {
        super.onStart();

        // Video is not paused for screen-capture. See onPause.
        if (peerConnectionClient != null) {
            peerConnectionClient.startVideoSource();
        }
    }

    @Override
    protected void onDestroy() {

        if (!disconnect.get()) {
            RTCSession.getInstance().emitSessionClose(getApplicationContext(),
                    PID.create(callee));
            disconnect(0, true);
        }
        super.onDestroy();

    }


    public void onCameraSwitch() {
        if (peerConnectionClient != null) {
            peerConnectionClient.switchCamera();
        }
    }


    public boolean onToggleMic() {
        if (peerConnectionClient != null) {
            micEnabled = !micEnabled;
            peerConnectionClient.setAudioEnabled(micEnabled);
        }
        return micEnabled;
    }

    public void onToggleSpeaker() {
        if (audioManager != null) {
            speakerEnabled = !speakerEnabled;
            if (speakerEnabled) {
                audioManager.setDefaultAudioDevice(AudioDevice.SPEAKER_PHONE);
                fab_toggle_speaker.setImageResource(R.drawable.volume_high);
            } else {
                audioManager.setDefaultAudioDevice(AudioDevice.EARPIECE);
                fab_toggle_speaker.setImageResource(R.drawable.volume_medium);
            }

        }
    }

    private void toggleCallControls() {

        // Show/hide call control fragment
        callControlFragmentVisible = !callControlFragmentVisible;

        if (callControlFragmentVisible) {
            fab_video_layout.setVisibility(View.VISIBLE);
        } else {
            fab_video_layout.setVisibility(View.GONE);
        }
    }


    // Disconnect from remote resources, dispose of local resources, and exit.
    private void disconnect(long timeout, boolean closeCallDialog) {

        onEnableFabs(false);

        if (!disconnect.getAndSet(true)) {

            if (closeCallDialog) {
                closeCallDialog();
            }

            remoteProxyRenderer.setTarget(null);
            localProxyVideoSink.setTarget(null);

            if (appRtcClient != null) {
                appRtcClient = null;
            }
            if (pipRenderer != null) {
                pipRenderer.release();
                pipRenderer = null;
            }
            if (fullscreenRenderer != null) {
                fullscreenRenderer.release();
                fullscreenRenderer = null;
            }
            if (peerConnectionClient != null) {
                peerConnectionClient.close();
                peerConnectionClient = null;
            }
            if (audioManager != null) {
                audioManager.stop();
                audioManager = null;
            }

            RTCSession.getInstance().setBusy(false);

            if (timeout > 0) {
                new Handler().postDelayed(this::finish, timeout);
            } else {
                finish();
            }
        }

    }


    private void closeCallDialog() {
        try {
            if (callDialog != null) {
                callDialog.dismiss();
                callDialog = null;
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
    }


    @Nullable
    private VideoCapturer createVideoCapturer() {
        final VideoCapturer videoCapturer;

        if (useCamera2()) {
            videoCapturer = createCameraCapturer(new Camera2Enumerator(this));
        } else {
            videoCapturer = createCameraCapturer(new Camera1Enumerator(false));
        }
        if (videoCapturer == null) {
            Singleton singleton = Singleton.getInstance(getApplicationContext());
            Preferences.info(singleton.getThreads(), getString(R.string.camera_error));
            disconnect(DISCONNECT_TIMEOUT, true);
            return null;
        }
        return videoCapturer;
    }

    private void setSwappedFeeds(boolean isSwappedFeeds) {
        localProxyVideoSink.setTarget(isSwappedFeeds ? fullscreenRenderer : pipRenderer);
        remoteProxyRenderer.setTarget(isSwappedFeeds ? pipRenderer : fullscreenRenderer);
        fullscreenRenderer.setMirror(isSwappedFeeds);
        pipRenderer.setMirror(!isSwappedFeeds);

    }


    @Override
    public void onChannelOffer(@NonNull final SessionDescription offerSdp) {
        checkNotNull(offerSdp);
        runOnUiThread(() -> {
            try {

                VideoCapturer videoCapturer = null;
                if (peerConnectionParameters.videoCallEnabled) {
                    videoCapturer = createVideoCapturer();
                }
                checkNotNull(peerConnectionClient);

                peerConnectionClient.createPeerConnection(
                        localProxyVideoSink, remoteSinks, videoCapturer, peerIceServers);

                peerConnectionClient.setRemoteDescription(offerSdp);

                // Create answer. Answer SDP will be sent to offering client in
                // PeerConnectionEvents.onLocalDescription event.
                peerConnectionClient.createAnswer();
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        });
    }

    @Override
    public void onRemoteDescription(@NonNull final SessionDescription sdp) {
        runOnUiThread(() -> {

            if (peerConnectionClient == null) {
                return;
            }
            peerConnectionClient.setRemoteDescription(sdp);
            if (!initiator) {
                // Create answer. Answer SDP will be sent to offering client in
                // PeerConnectionEvents.onLocalDescription event.
                peerConnectionClient.createAnswer();
            }

        });
    }

    @Override
    public void onRemoteIceCandidate(final IceCandidate candidate) {
        runOnUiThread(() -> {
            if (peerConnectionClient != null) {
                peerConnectionClient.addRemoteIceCandidate(candidate);
            }
        });
    }

    @Override
    public void onRemoteIceCandidateRemoved(final IceCandidate candidate) {
        runOnUiThread(() -> {
            if (peerConnectionClient != null) {
                IceCandidate[] candidates = {candidate};
                peerConnectionClient.removeRemoteIceCandidates(candidates);
            }
        });
    }


    @Override
    public void onChannelAccepted(@Nullable String[] ices) {
        runOnUiThread(() -> {
            try {

                if (ices != null) {
                    for (String turn : ices) {
                        peerIceServers.add(
                                PeerConnection.IceServer.builder(turn).createIceServer());
                    }
                }

                VideoCapturer videoCapturer = null;
                if (peerConnectionParameters.videoCallEnabled) {
                    videoCapturer = createVideoCapturer();
                }
                checkNotNull(peerConnectionClient); // must be defined here


                peerConnectionClient.createPeerConnection(
                        localProxyVideoSink, remoteSinks, videoCapturer, peerIceServers);


                // Create offer. Offer SDP will be sent to answering client in
                // PeerConnectionEvents.onLocalDescription event.
                peerConnectionClient.createOffer();
            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
            }
        });
    }

    // -----Implementation of RTCPeerConnection.PeerConnectionEvents.---------
    // Send local peer connection SDP and ICE candidates to remote party.
    // All callbacks are invoked from peer connection client looper thread and
    // are routed to UI thread.
    @Override
    public void onLocalDescription(final SessionDescription sdp) {

        if (appRtcClient != null) {
            if (initiator) {
                appRtcClient.sendOfferSdp(getApplicationContext(), sdp);
            } else {
                appRtcClient.sendAnswerSdp(getApplicationContext(), sdp);
            }
        }

        runOnUiThread(() -> {

            if (peerConnectionParameters.videoMaxBitrate > 0) {

                if (peerConnectionClient != null) {
                    peerConnectionClient.setVideoMaxBitrate(
                            peerConnectionParameters.videoMaxBitrate);
                }
            }

        });
    }

    @Override
    public void onIceCandidate(final IceCandidate candidate) {
        if (appRtcClient != null) {
            appRtcClient.sendLocalIceCandidate(getApplicationContext(), candidate);
        }

    }

    @Override
    public void onIceCandidatesRemoved(final IceCandidate[] candidates) {
        if (appRtcClient != null) {
            appRtcClient.sendLocalIceCandidateRemovals(getApplicationContext(), candidates);
        }
    }

    @Override
    public void onIceConnected() {
    }

    @Override
    public void onIceDisconnected() {
    }

    @Override
    public void onConnected() {
        runOnUiThread(() -> {
            onEnableFabs(true);
            closeCallDialog();
            setSwappedFeeds(false);
        });
    }

    @Override
    public void onChannelClose() {
        Singleton singleton = Singleton.getInstance(getApplicationContext());
        Preferences.info(singleton.getThreads(),
                getString(R.string.call_closed_by, calleeName));
        runOnUiThread(() -> disconnect(DISCONNECT_TIMEOUT, true));
    }

    @Override
    public void onChannelBusy() {
        Singleton singleton = Singleton.getInstance(getApplicationContext());
        Preferences.info(singleton.getThreads(),
                getString(R.string.call_busy_by, calleeName));
        runOnUiThread(() -> disconnect(DISCONNECT_TIMEOUT, true));
    }

    @Override
    public void onChannelTimeout() {
        Singleton singleton = Singleton.getInstance(getApplicationContext());
        Preferences.info(singleton.getThreads(),
                getString(R.string.call_timeout_by, calleeName));
        runOnUiThread(() -> disconnect(DISCONNECT_TIMEOUT, true));
    }

    @Override
    public void onChannelReject() {
        Singleton singleton = Singleton.getInstance(getApplicationContext());
        Preferences.info(singleton.getThreads(),
                getString(R.string.call_busy_by, calleeName));
        runOnUiThread(() -> disconnect(DISCONNECT_TIMEOUT, true));
    }


    @Override
    public void onDisconnected() {
        runOnUiThread(() -> disconnect(DISCONNECT_TIMEOUT, true));
    }

    @Override
    public void onPeerConnectionClosed() {
        runOnUiThread(() -> disconnect(DISCONNECT_TIMEOUT, true));
    }

    @Override
    public void onPeerConnectionError(final String description) {

        // for now connection errors are ignored
    }


    @Override
    public void acceptCall(@NonNull String pid) {
        checkNotNull(pid);
        try {
            runOnUiThread(() -> onEnableFabs(true));
            PID host = Preferences.getPID(getApplicationContext());
            checkNotNull(host);
            RTCSession.getInstance().emitSessionAccept(getApplicationContext(),
                    host, PID.create(pid));
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
    }

    @Override
    public void rejectCall(@NonNull String pid) {
        checkNotNull(pid);
        try {
            runOnUiThread(() -> onEnableFabs(true));
            RTCSession.getInstance().emitSessionReject(getApplicationContext(),
                    PID.create(pid));

            disconnect(DISCONNECT_TIMEOUT, true);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        } finally {
            RTCSession.getInstance().setBusy(false);
        }
    }

    @Override
    public void abortCall(@NonNull String pid) {
        checkNotNull(pid);
        try {
            runOnUiThread(() -> onEnableFabs(true));
            RTCSession.getInstance().emitSessionClose(getApplicationContext(),
                    PID.create(pid));
            disconnect(DISCONNECT_TIMEOUT, false);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        } finally {
            RTCSession.getInstance().setBusy(false);
        }
    }

    @Override
    public void timeoutCall(@NonNull String pid) {
        checkNotNull(pid);
        try {
            runOnUiThread(() -> onEnableFabs(true));
            Singleton singleton = Singleton.getInstance(getApplicationContext());
            Preferences.info(singleton.getThreads(),
                    getString(R.string.timeout_call));
            RTCSession.getInstance().emitSessionTimeout(getApplicationContext(),
                    PID.create(pid));
            runOnUiThread(() -> disconnect(DISCONNECT_TIMEOUT, false));
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

    }


    @Override
    public void failCall(@NonNull String pid) {
        checkNotNull(pid);
        try {
            Singleton singleton = Singleton.getInstance(getApplicationContext());
            Preferences.info(singleton.getThreads(),
                    getString(R.string.call_fail_by, calleeName));
            runOnUiThread(() -> disconnect(DISCONNECT_TIMEOUT, false));
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
    }


    private static class ProxyVideoSink implements VideoSink {
        private VideoSink target;

        @Override
        synchronized public void onFrame(VideoFrame frame) {
            if (target == null) {
                return;
            }
            target.onFrame(frame);
        }

        synchronized void setTarget(VideoSink target) {
            this.target = target;
        }
    }


}
