package threads.share;

import android.content.Context;
import android.os.SystemClock;
import android.text.Html;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.DiffUtil;
import androidx.recyclerview.widget.RecyclerView;

import com.amulyakhare.textdrawable.util.ColorGenerator;
import com.bumptech.glide.Glide;
import com.bumptech.glide.load.MultiTransformation;
import com.bumptech.glide.load.resource.bitmap.CenterCrop;
import com.bumptech.glide.load.resource.bitmap.RoundedCorners;
import com.bumptech.glide.request.RequestOptions;

import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;

import threads.core.Preferences;
import threads.core.Singleton;
import threads.core.THREADS;
import threads.core.api.Note;
import threads.core.api.NoteType;
import threads.ipfs.IPFS;

import static androidx.core.util.Preconditions.checkNotNull;

public class NotesViewAdapter extends RecyclerView.Adapter<NotesViewAdapter.ViewHolder> {
    private static final String TAG = NotesViewAdapter.class.getSimpleName();
    private final Context context;
    private final List<Note> notes = new ArrayList<>();
    private final NotesViewAdapterListener listener;
    private long mLastClickTime = 0;


    public NotesViewAdapter(@NonNull Context context, @NonNull NotesViewAdapterListener listener) {
        this.context = context;
        this.listener = listener;
    }


    @NonNull
    @Override
    public NotesViewAdapter.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {

        View v;
        switch (viewType) {
            case 1:
                v = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.thread_message, parent, false);
                return new MessageHolder(v);
            case 2:
                v = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.thread_info, parent, false);
                return new InfoHolder(v);
            case 3:
                v = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.thread_link, parent, false);
                return new LinkHolder(v);
            case 4:
                v = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.thread_data, parent, false);
                return new DataHolder(v);
            case 5:
                v = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.thread_audio, parent, false);
                return new AudioHolder(v);
            case 6:
                v = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.thread_html, parent, false);
                return new HtmlHolder(v);
            case 7:
                v = LayoutInflater.from(parent.getContext())
                        .inflate(R.layout.thread_location, parent, false);
                return new LocationHolder(v);

        }
        throw new RuntimeException("View Type not supported !!!");
    }

    @Override
    public void onBindViewHolder(@NonNull NotesViewAdapter.ViewHolder holder, final int position) {
        Note note = notes.get(position);
        checkNotNull(note);
        try {
            if (holder instanceof InfoHolder) {
                InfoHolder noteHolder = (InfoHolder) holder;
                handleInfoNote(noteHolder, note);
            }
            if (holder instanceof HtmlHolder) {
                HtmlHolder htmlHolder = (HtmlHolder) holder;
                handleHtmlNote(htmlHolder, note);
            }
            if (holder instanceof AudioHolder) {
                AudioHolder audioHolder = (AudioHolder) holder;
                handleAudioNote(audioHolder, note);
            }
            if (holder instanceof MessageHolder) {
                MessageHolder messageHolder = (MessageHolder) holder;
                handleMessageNote(messageHolder, note);
            }
            if (holder instanceof LinkHolder) {
                LinkHolder linkHolder = (LinkHolder) holder;
                handleLinkNote(linkHolder, note);
            }
            if (holder instanceof DataHolder) {
                DataHolder dataHolder = (DataHolder) holder;
                handleDataNote(dataHolder, note);
            }
            if (holder instanceof LocationHolder) {
                LocationHolder locationHolder = (LocationHolder) holder;
                handleLocationNote(locationHolder, note);
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

    }

    private void handleInfoNote(@NonNull InfoHolder infoHolder, @NonNull Note note) {
        checkNotNull(infoHolder);
        checkNotNull(note);

        TextView textView = infoHolder.message_body;
        textView.setTextAppearance(android.R.style.TextAppearance_Small);
        String content = listener.getContent(note);
        textView.setText(content);

        if (listener.showDate(note)) {
            infoHolder.date.setText(Preferences.getDate(note.getDate()));
        }
    }

    private void handleHtmlNote(@NonNull HtmlHolder htmlHolder, @NonNull Note note) {
        checkNotNull(htmlHolder);
        checkNotNull(note);

        TextView textView = htmlHolder.message_body;
        textView.setTextAppearance(android.R.style.TextAppearance_Medium);

        String content = listener.getContent(note);
        textView.setTextAppearance(android.R.style.TextAppearance_Small);
        textView.setText(Html.fromHtml(content, Html.FROM_HTML_MODE_LEGACY));

        if (listener.showDate(note)) {
            htmlHolder.date.setText(Preferences.getDate(note.getDate()));
        }
        htmlHolder.image_action.setVisibility(View.INVISIBLE);


        switch (note.getStatus()) {
            case SEEDING:
            case PUBLISHING:
            case OFFLINE:
                htmlHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, R.drawable.upload, 0);
                htmlHolder.date.setCompoundDrawablePadding(0);
                break;
            case LEACHING:
                htmlHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, R.drawable.download, 0);
                htmlHolder.date.setCompoundDrawablePadding(0);
                break;
            case EXPIRED:
                htmlHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, R.drawable.timer_off, 0);
                htmlHolder.date.setCompoundDrawablePadding(0);
                break;
            case DELETING:
                htmlHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, R.drawable.delete, 0);
                htmlHolder.date.setCompoundDrawablePadding(0);
                break;
            case ERROR: {
                htmlHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, 0, 0);
                htmlHolder.date.setCompoundDrawablePadding(0);
                htmlHolder.image_action.setVisibility(View.VISIBLE);
                htmlHolder.image_action.setImageResource(R.drawable.sync_alert);
                htmlHolder.image_action.setOnClickListener((v) -> {

                    if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                        return;
                    }
                    mLastClickTime = SystemClock.elapsedRealtime();

                    listener.invokeHtmlActionError(note);
                });
                break;
            }
            default:
                htmlHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, 0, 0);
                htmlHolder.date.setCompoundDrawablePadding(0);
                break;
        }
    }

    private void handleMessageNote(@NonNull MessageHolder messageHolder, @NonNull Note note) {
        checkNotNull(messageHolder);
        checkNotNull(note);

        if (listener.showName(note)) {
            String author = note.getSenderAlias();
            messageHolder.user.setText(author);
            int color = ColorGenerator.MATERIAL.getColor(author);
            messageHolder.user.setTextColor(color);
        }
        if (listener.showDate(note)) {
            messageHolder.date.setText(Preferences.getDate(note.getDate()));
        }

        TextView textView = messageHolder.message_body;
        if (note.getNoteType() == NoteType.MESSAGE) {
            textView.setTextAppearance(android.R.style.TextAppearance_Medium);
        } else {
            textView.setTextAppearance(android.R.style.TextAppearance_Small);
        }

        String content = listener.getContent(note);
        textView.setText(content);

        if (listener.generalActionSupport(note)) {
            messageHolder.general_action.setVisibility(View.VISIBLE);
            messageHolder.general_action.setOnClickListener((v) ->
                    listener.invokeGeneralAction(note)
            );
        } else {
            messageHolder.general_action.setVisibility(View.GONE);
        }
        switch (note.getStatus()) {
            case OFFLINE: {
                messageHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, R.drawable.upload, 0);
                messageHolder.date.setCompoundDrawablePadding(0);
                messageHolder.image_action.setVisibility(View.INVISIBLE);
                break;
            }
            case LEACHING: {
                messageHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, R.drawable.download, 0);
                messageHolder.date.setCompoundDrawablePadding(0);
                messageHolder.image_action.setVisibility(View.INVISIBLE);
                break;
            }
            case EXPIRED: {
                messageHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, R.drawable.timer_off, 0);
                messageHolder.date.setCompoundDrawablePadding(0);
                messageHolder.image_action.setVisibility(View.INVISIBLE);
                break;
            }
            case DELETING: {
                messageHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, R.drawable.delete, 0);
                messageHolder.date.setCompoundDrawablePadding(0);
                messageHolder.image_action.setVisibility(View.INVISIBLE);
                break;
            }
            case PUBLISHING:
            case SENDING: {
                messageHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, R.drawable.upload, 0);
                messageHolder.date.setCompoundDrawablePadding(0);
                messageHolder.image_action.setVisibility(View.INVISIBLE);
                break;
            }
            case SEEDING: {
                messageHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, 0, 0);
                messageHolder.date.setCompoundDrawablePadding(0);
                messageHolder.image_action.setVisibility(View.INVISIBLE);
                break;
            }
            case ERROR: {
                messageHolder.image_action.setVisibility(View.VISIBLE);
                messageHolder.image_action.setImageResource(R.drawable.sync_alert);
                messageHolder.image_action.setOnClickListener((v) -> {
                    if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                        return;
                    }
                    mLastClickTime = SystemClock.elapsedRealtime();

                    listener.invokeMessageActionError(note);
                });

                break;
            }
            default: {
                messageHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, 0, 0);
                messageHolder.date.setCompoundDrawablePadding(0);
                messageHolder.image_action.setVisibility(View.INVISIBLE);
                break;
            }
        }
    }


    private void handleAudioNote(@NonNull AudioHolder audioHolder, @NonNull Note note) {
        checkNotNull(audioHolder);
        checkNotNull(note);


        if (listener.showName(note)) {
            String author = note.getSenderAlias();
            audioHolder.user.setText(author);
            int color = ColorGenerator.MATERIAL.getColor(author);
            audioHolder.user.setTextColor(color);
        }

        audioHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                0, 0, 0, 0);
        audioHolder.date.setCompoundDrawablePadding(0);

        if (listener.showDate(note)) {
            audioHolder.date.setText(Preferences.getDate(note.getDate()));
        }
        String title = listener.getTitle(note);
        if (listener.generalActionSupport(note)) {
            audioHolder.general_action.setVisibility(View.VISIBLE);
            audioHolder.general_action.setOnClickListener((v) ->
                    listener.invokeGeneralAction(note)
            );
        } else {
            audioHolder.general_action.setVisibility(View.GONE);
        }
        switch (note.getStatus()) {
            case OFFLINE: {
                String downloading = context.getString(R.string.start_uploading);
                audioHolder.audio_title.setText(downloading.concat(title));
                audioHolder.image_action.setVisibility(View.VISIBLE);
                audioHolder.image_action.setImageResource(R.drawable.upload);
                audioHolder.progress_bar.setVisibility(View.VISIBLE);
                audioHolder.progress_bar.setIndeterminate(true);
                break;
            }


            case SEEDING: {
                audioHolder.audio_title.setText(title);
                if (note.getCid() != null) {
                    audioHolder.image_action.setVisibility(View.VISIBLE);
                    audioHolder.image_action.setImageResource(R.drawable.play_circle);
                    audioHolder.image_action.setClickable(true);
                    audioHolder.image_action.setOnClickListener((view) -> {
                        if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                            return;
                        }
                        mLastClickTime = SystemClock.elapsedRealtime();

                        listener.invokeAudioAction(note);
                    });
                }
                audioHolder.progress_bar.setVisibility(View.INVISIBLE);
                break;
            }
            case PUBLISHING:
            case SENDING: {
                audioHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, R.drawable.upload, 0);
                audioHolder.date.setCompoundDrawablePadding(0);
                audioHolder.audio_title.setText(title);
                if (note.getCid() != null) {
                    audioHolder.image_action.setVisibility(View.VISIBLE);
                    audioHolder.image_action.setImageResource(R.drawable.play_circle);
                    audioHolder.image_action.setClickable(true);
                    audioHolder.image_action.setOnClickListener((view) -> {
                        if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                            return;
                        }
                        mLastClickTime = SystemClock.elapsedRealtime();

                        listener.invokeAudioAction(note);
                    });
                }
                audioHolder.progress_bar.setVisibility(View.INVISIBLE);
                break;
            }
            case LEACHING: {
                String downloading = context.getString(R.string.start_downloading);
                audioHolder.audio_title.setText(downloading.concat(title));
                audioHolder.image_action.setVisibility(View.VISIBLE);
                audioHolder.image_action.setImageResource(R.drawable.download);
                audioHolder.progress_bar.setVisibility(View.VISIBLE);
                audioHolder.progress_bar.setIndeterminate(true);
                break;
            }
            case ERROR: {
                audioHolder.progress_bar.setVisibility(View.INVISIBLE);
                audioHolder.image_action.setVisibility(View.VISIBLE);
                audioHolder.image_action.setImageResource(R.drawable.sync_alert);
                audioHolder.image_action.setOnClickListener((v) -> {

                    if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                        return;
                    }
                    mLastClickTime = SystemClock.elapsedRealtime();

                    listener.invokeAudioActionError(note);
                });
                audioHolder.audio_title.setText(title);
                break;
            }
            case EXPIRED:
                audioHolder.progress_bar.setVisibility(View.INVISIBLE);
                audioHolder.image_action.setVisibility(View.VISIBLE);
                audioHolder.image_action.setImageDrawable(context.getDrawable(R.drawable.timer_off));
                audioHolder.audio_title.setText(title);
                break;
            case DELETING:
                audioHolder.progress_bar.setVisibility(View.INVISIBLE);
                audioHolder.image_action.setVisibility(View.VISIBLE);
                audioHolder.image_action.setImageDrawable(context.getDrawable(R.drawable.delete));
                audioHolder.audio_title.setText(title);
                break;
            default:
                audioHolder.image_action.setVisibility(View.INVISIBLE);
                audioHolder.progress_bar.setVisibility(View.INVISIBLE);
                audioHolder.audio_title.setText(title);
                break;
        }
    }


    private void handleLocationNote(@NonNull LocationHolder locationHolder, @NonNull Note note) {
        checkNotNull(locationHolder);
        checkNotNull(note);

        if (listener.showDate(note)) {
            locationHolder.date.setText(Preferences.getDate(note.getDate()));
        }

        if (listener.showName(note)) {
            String author = note.getSenderAlias();
            locationHolder.user.setText(author);
            int color = ColorGenerator.MATERIAL.getColor(author);
            locationHolder.user.setTextColor(color);
        }
        if (listener.generalActionSupport(note)) {
            locationHolder.general_action.setVisibility(View.VISIBLE);
            locationHolder.general_action.setOnClickListener((v) ->
                    listener.invokeGeneralAction(note)
            );
        } else {
            locationHolder.general_action.setVisibility(View.GONE);
        }


        MultiTransformation transform = new MultiTransformation(
                new CenterCrop(), new RoundedCorners(30));
        checkNotNull(transform);
        if (note.getImage() != null) {
            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            IPFSData data = IPFSData.create(ipfs, note.getImage(), note.getSesKey());
            Glide.with(context).load(data).apply(
                    new RequestOptions().timeout(15000).transform(transform))
                    .into(locationHolder.image_view);

        } else {
            locationHolder.image_view.setImageResource(R.drawable.action_map_marker);
        }

        switch (note.getStatus()) {
            case LEACHING: {
                locationHolder.image_action.setVisibility(View.VISIBLE);
                locationHolder.image_action.setImageResource(R.drawable.download);
                locationHolder.progress_bar.setVisibility(View.VISIBLE);
                locationHolder.progress_bar.setIndeterminate(true);
                break;
            }
            case OFFLINE: {
                locationHolder.image_action.setVisibility(View.VISIBLE);
                locationHolder.image_action.setImageResource(R.drawable.upload);
                locationHolder.progress_bar.setVisibility(View.VISIBLE);
                locationHolder.progress_bar.setIndeterminate(true);
                break;
            }
            case PUBLISHING:
            case SEEDING:
            case SENDING: {
                locationHolder.progress_bar.setVisibility(View.INVISIBLE);

                locationHolder.image_action.setVisibility(View.INVISIBLE);
                locationHolder.image_view.setClickable(true);
                locationHolder.image_view.setOnClickListener((v) -> {
                    if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                        return;
                    }
                    mLastClickTime = SystemClock.elapsedRealtime();
                    listener.invokeLocationAction(note);
                });


                break;
            }
            case ERROR: {
                locationHolder.progress_bar.setVisibility(View.INVISIBLE);
                locationHolder.image_action.setVisibility(View.VISIBLE);
                locationHolder.image_action.setImageResource(R.drawable.sync_alert);
                locationHolder.image_action.setOnClickListener((v) -> {
                    if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                        return;
                    }
                    mLastClickTime = SystemClock.elapsedRealtime();

                    listener.invokeLocationActionError(note);
                });

                break;
            }
            case EXPIRED: {
                locationHolder.image_action.setVisibility(View.INVISIBLE);
                locationHolder.progress_bar.setVisibility(View.INVISIBLE);
                locationHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, R.drawable.timer_off, 0);
                locationHolder.date.setCompoundDrawablePadding(0);
                break;
            }
            case DELETING: {
                locationHolder.image_action.setVisibility(View.INVISIBLE);
                locationHolder.progress_bar.setVisibility(View.INVISIBLE);
                locationHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, R.drawable.delete, 0);
                locationHolder.date.setCompoundDrawablePadding(0);
                break;
            }
            default:
                locationHolder.image_action.setVisibility(View.INVISIBLE);
                locationHolder.progress_bar.setVisibility(View.INVISIBLE);
                break;
        }

    }

    private void handleDataNote(@NonNull DataHolder dataHolder, @NonNull Note note) {
        checkNotNull(dataHolder);
        checkNotNull(note);

        if (listener.showDate(note)) {
            dataHolder.date.setText(Preferences.getDate(note.getDate()));
        }

        if (listener.showName(note)) {
            String author = note.getSenderAlias();
            dataHolder.user.setText(author);
            int color = ColorGenerator.MATERIAL.getColor(author);
            dataHolder.user.setTextColor(color);
        }
        if (listener.generalActionSupport(note)) {
            dataHolder.general_action.setVisibility(View.VISIBLE);
            dataHolder.general_action.setOnClickListener((v) ->
                    listener.invokeGeneralAction(note)
            );
        } else {
            dataHolder.general_action.setVisibility(View.GONE);
        }


        MultiTransformation transform = new MultiTransformation(
                new CenterCrop(), new RoundedCorners(30));
        checkNotNull(transform);
        if (note.getImage() != null) {
            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            IPFSData data = IPFSData.create(ipfs, note.getImage(), note.getSesKey());
            Glide.with(context).load(data).apply(
                    new RequestOptions().timeout(15000).transform(transform))
                    .into(dataHolder.image_view);

        } else {
            dataHolder.image_view.setImageResource(R.drawable.file_document);
        }

        switch (note.getStatus()) {
            case LEACHING: {
                dataHolder.image_action.setVisibility(View.VISIBLE);
                dataHolder.image_action.setImageResource(R.drawable.download);
                dataHolder.progress_bar.setVisibility(View.VISIBLE);
                dataHolder.progress_bar.setIndeterminate(true);
                break;
            }
            case OFFLINE: {
                dataHolder.image_action.setVisibility(View.VISIBLE);
                dataHolder.image_action.setImageResource(R.drawable.upload);
                dataHolder.progress_bar.setVisibility(View.VISIBLE);
                dataHolder.progress_bar.setIndeterminate(true);
                break;
            }
            case PUBLISHING:
            case SEEDING:
            case SENDING: {
                dataHolder.progress_bar.setVisibility(View.INVISIBLE);
                if (note.getMimeType().startsWith("video")) {
                    if (note.getCid() != null) {
                        dataHolder.image_action.setVisibility(View.VISIBLE);
                        dataHolder.image_action.setImageDrawable(context.getDrawable(R.drawable.play_circle));
                        dataHolder.image_action.setClickable(true);
                        dataHolder.image_action.setOnClickListener((v) -> {

                            if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                                return;
                            }
                            mLastClickTime = SystemClock.elapsedRealtime();
                            listener.invokeVideoAction(note);
                        });
                    }
                } else {
                    if (note.getCid() != null) {
                        dataHolder.image_action.setVisibility(View.INVISIBLE);
                        dataHolder.image_view.setClickable(true);
                        dataHolder.image_view.setOnClickListener((v) -> {

                            if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                                return;
                            }
                            mLastClickTime = SystemClock.elapsedRealtime();
                            listener.invokeDataAction(note);
                        });
                    }
                }

                break;
            }
            case ERROR: {
                dataHolder.progress_bar.setVisibility(View.INVISIBLE);
                dataHolder.image_action.setVisibility(View.VISIBLE);
                dataHolder.image_action.setImageResource(R.drawable.sync_alert);
                dataHolder.image_action.setOnClickListener((v) -> {
                    if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                        return;
                    }
                    mLastClickTime = SystemClock.elapsedRealtime();

                    listener.invokeDataActionError(note);
                });

                break;
            }
            case EXPIRED: {
                dataHolder.image_action.setVisibility(View.INVISIBLE);
                dataHolder.progress_bar.setVisibility(View.INVISIBLE);
                dataHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, R.drawable.timer_off, 0);
                dataHolder.date.setCompoundDrawablePadding(0);
                break;
            }
            case DELETING: {
                dataHolder.image_action.setVisibility(View.INVISIBLE);
                dataHolder.progress_bar.setVisibility(View.INVISIBLE);
                dataHolder.date.setCompoundDrawablesRelativeWithIntrinsicBounds(
                        0, 0, R.drawable.delete, 0);
                dataHolder.date.setCompoundDrawablePadding(0);
                break;
            }
            default:
                dataHolder.image_action.setVisibility(View.INVISIBLE);
                dataHolder.progress_bar.setVisibility(View.INVISIBLE);
                break;
        }

    }

    @Override
    public int getItemViewType(int position) {
        Note notification = getItemAtPosition(position);
        checkNotNull(notification);
        NoteType noteType = notification.getNoteType();
        checkNotNull(noteType);

        if (noteType == NoteType.LOCATION) {
            return 7;
        }

        if (noteType == NoteType.HTML) {
            return 6;
        }

        if (noteType == NoteType.AUDIO) {
            return 5;
        }

        if (noteType == NoteType.DATA) {
            return 4;
        }

        if (noteType == NoteType.LINK) {
            return 3;
        }

        if (noteType == NoteType.INFO) {
            return 2;
        }

        return 1;
    }


    private Note getItemAtPosition(int position) {
        try {
            return notes.get(position);
        } catch (Throwable e) {
            // ignore exception
        }
        return null;
    }


    @Override
    public int getItemCount() {
        return notes.size();
    }

    public void updateNotes(@NonNull List<Note> notes) {

        final NoteDiffCallback diffCallback = new NoteDiffCallback(this.notes, notes);
        final DiffUtil.DiffResult diffResult = DiffUtil.calculateDiff(diffCallback);

        this.notes.clear();
        this.notes.addAll(notes);
        diffResult.dispatchUpdatesTo(this);
    }

    private void handleLinkNote(@NonNull LinkHolder linkHolder, @NonNull Note note) {
        checkNotNull(linkHolder);
        checkNotNull(note);

        if (listener.showName(note)) {
            String author = StringUtils.left(note.getSenderAlias(), Preferences.MAX_NAME_SIZE);
            linkHolder.user.setText(author);
            int color = ColorGenerator.MATERIAL.getColor(author);
            linkHolder.user.setTextColor(color);
        }
        if (listener.showDate(note)) {
            linkHolder.date.setText(Preferences.getDate(note.getDate()));
        }

        TextView textView = linkHolder.message_body;
        if (note.getNoteType() == NoteType.MESSAGE) {
            textView.setTextAppearance(android.R.style.TextAppearance_Medium);
        } else {
            textView.setTextAppearance(android.R.style.TextAppearance_Small);
        }

        linkHolder.mime_type.setImageResource(listener.getMediaResource(note));

        THREADS threads = Singleton.getInstance(context).getThreads();
        String fileName = threads.getFileName(note);
        String fileSize = String.valueOf(threads.getFileSize(note) / 1000);
        textView.setText(context.getString(R.string.link_format, fileName, fileSize));

        if (listener.generalActionSupport(note)) {
            linkHolder.general_action.setVisibility(View.VISIBLE);
            linkHolder.general_action.setOnClickListener((v) ->
                    listener.invokeGeneralAction(note)
            );
        } else {
            linkHolder.general_action.setVisibility(View.GONE);
        }
        switch (note.getStatus()) {
            case LEACHING: {
                linkHolder.image_action.setVisibility(View.VISIBLE);
                linkHolder.image_action.setImageResource(R.drawable.download);
                break;
            }
            case ONLINE: {
                if (note.getCid() != null) {
                    linkHolder.image_action.setVisibility(View.VISIBLE);
                    linkHolder.image_action.setImageResource(R.drawable.sync_download);
                    linkHolder.image_action.setOnClickListener((v) -> {

                        try {
                            if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                                return;
                            }
                            mLastClickTime = SystemClock.elapsedRealtime();

                            listener.invokeLinkAction(note);

                        } catch (Throwable e) {
                            Log.e(TAG, "" + e.getLocalizedMessage(), e);
                        }
                    });
                }
                break;
            }
            case ERROR: {
                linkHolder.image_action.setVisibility(View.VISIBLE);
                linkHolder.image_action.setImageResource(R.drawable.sync_alert);
                linkHolder.image_action.setOnClickListener((v) -> {

                    if (SystemClock.elapsedRealtime() - mLastClickTime < 1000) {
                        return;
                    }
                    mLastClickTime = SystemClock.elapsedRealtime();

                    listener.invokeLinkActionError(note);
                });

                break;
            }
            case OFFLINE: {
                linkHolder.image_action.setVisibility(View.VISIBLE);
                linkHolder.image_action.setImageResource(R.drawable.upload);
                break;
            }
            case EXPIRED: {
                linkHolder.image_action.setVisibility(View.VISIBLE);
                linkHolder.image_action.setImageResource(R.drawable.timer_off);
                break;
            }
            case DELETING: {
                linkHolder.image_action.setVisibility(View.VISIBLE);
                linkHolder.image_action.setImageResource(R.drawable.delete);
                break;
            }
            default: {
                linkHolder.image_action.setVisibility(View.GONE);
                break;
            }
        }
    }

    public interface NotesViewAdapterListener {

        boolean generalActionSupport(@NonNull Note note);

        void invokeGeneralAction(@NonNull Note note);

        void invokeLinkAction(@NonNull Note note);

        void invokeAudioAction(@NonNull Note note);

        void invokeVideoAction(@NonNull Note note);

        void invokeLocationAction(@NonNull Note note);

        void invokeDataAction(@NonNull Note note);

        void invokeAudioActionError(@NonNull Note note);

        void invokeLinkActionError(@NonNull Note note);

        void invokeHtmlActionError(@NonNull Note note);

        void invokeDataActionError(@NonNull Note note);

        void invokeLocationActionError(@NonNull Note note);

        void invokeMessageActionError(@NonNull Note note);

        @NonNull
        String getTitle(@NonNull Note note);

        @NonNull
        String getContent(@NonNull Note note);

        boolean showDate(@NonNull Note note);

        boolean showName(@NonNull Note note);

        int getMediaResource(@NonNull Note note);
    }

    static class ViewHolder extends RecyclerView.ViewHolder {

        final TextView date;


        ViewHolder(View v) {
            super(v);
            date = v.findViewById(R.id.date);

        }

    }

    static class LinkHolder extends NotesViewAdapter.ViewHolder {
        final ImageView image_action;
        final TextView user;
        final TextView message_body;
        final ImageView general_action;
        final ImageView mime_type;

        LinkHolder(View v) {
            super(v);
            image_action = v.findViewById(R.id.image_action);
            general_action = v.findViewById(R.id.general_action);
            user = v.findViewById(R.id.user);
            message_body = v.findViewById(R.id.message_body);
            mime_type = v.findViewById(R.id.mime_type);
        }


    }

    static class InfoHolder extends NotesViewAdapter.ViewHolder {
        final TextView message_body;

        InfoHolder(View v) {
            super(v);
            message_body = v.findViewById(R.id.message_body);
        }


    }

    static class MessageHolder extends NotesViewAdapter.ViewHolder {
        final ImageView image_action;
        final TextView user;
        final TextView message_body;
        final ImageView general_action;

        MessageHolder(View v) {
            super(v);
            image_action = v.findViewById(R.id.image_action);
            message_body = v.findViewById(R.id.message_body);
            general_action = v.findViewById(R.id.general_action);
            user = v.findViewById(R.id.user);
        }


    }

    static class HtmlHolder extends NotesViewAdapter.ViewHolder {

        final TextView message_body;
        final ImageView image_action;

        HtmlHolder(View v) {
            super(v);
            image_action = v.findViewById(R.id.image_action);
            message_body = v.findViewById(R.id.message_body);
        }

    }

    static class AudioHolder extends NotesViewAdapter.ViewHolder {
        final ImageView image_action;
        final ProgressBar progress_bar;
        final ImageView general_action;
        final TextView user;
        final TextView audio_title;


        AudioHolder(View v) {
            super(v);
            user = v.findViewById(R.id.user);
            image_action = v.findViewById(R.id.image_action);
            general_action = v.findViewById(R.id.general_action);
            progress_bar = v.findViewById(R.id.progress_bar);
            progress_bar.setMax(100);
            audio_title = v.findViewById(R.id.audio_title);

        }
    }

    static class DataHolder extends NotesViewAdapter.ViewHolder {
        final ImageView image_action;
        final ImageView general_action;
        final ProgressBar progress_bar;
        final ImageView image_view;
        final TextView user;

        DataHolder(View v) {
            super(v);
            image_view = v.findViewById(R.id.image_view);
            general_action = v.findViewById(R.id.general_action);
            image_action = v.findViewById(R.id.image_action);
            user = v.findViewById(R.id.user);
            progress_bar = v.findViewById(R.id.progress_bar);
            progress_bar.setMax(100);
        }
    }

    static class LocationHolder extends NotesViewAdapter.ViewHolder {
        final ImageView image_action;
        final ImageView general_action;
        final ProgressBar progress_bar;
        final ImageView image_view;
        final TextView user;

        LocationHolder(View v) {
            super(v);
            image_view = v.findViewById(R.id.image_view);
            general_action = v.findViewById(R.id.general_action);
            image_action = v.findViewById(R.id.image_action);
            user = v.findViewById(R.id.user);
            progress_bar = v.findViewById(R.id.progress_bar);
            progress_bar.setMax(100);
        }
    }

}
