package threads.share;

import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import threads.core.Network;
import threads.core.Preferences;
import threads.core.Singleton;
import threads.core.THREADS;
import threads.core.api.Addresses;
import threads.core.api.Content;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.Peer;

import static androidx.core.util.Preconditions.checkNotNull;

public class PeerService {
    private static final int MAX_RELAYS = 3;
    private static final String TAG = PeerService.class.getSimpleName();
    private static final String SUPPORT_PEER_DISCOVERY_KEY = "supportPeerDiscoveryKey";

    public static boolean isSupportPeerDiscovery(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(TAG, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(SUPPORT_PEER_DISCOVERY_KEY, true);
    }

    public static void setSupportPeerDiscovery(@NonNull Context context, boolean enable) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(TAG, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(SUPPORT_PEER_DISCOVERY_KEY, enable);
        editor.apply();
    }

    public static void publishPeer(@NonNull Context context) {
        checkNotNull(context);

        if (!Network.isConnected(context)) {
            return;
        }

        if (!PeerService.isSupportPeerDiscovery(context)) {
            return;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        if (ipfs != null) {
            try {

                PID host = Preferences.getPID(context);
                if (host != null) {
                    threads.core.api.Peer peer = threads.getPeerByPID(host);
                    if (peer != null) {
                        boolean isDirectPeerValid = isDirectPeerValid(context, peer);
                        int numRelays = updatePeerRelays(context, peer);
                        if (numRelays < MAX_RELAYS || !isDirectPeerValid) {
                            Addresses addresses = peer.getAddresses();
                            List<Peer> relays = getRelayPeers(context, addresses.keySet());

                            Peer direct = null;
                            if (!isDirectPeerValid) {
                                direct = getDirectPeer(context);
                            }
                            updatePeer(context, peer, relays, direct);
                        }
                    } else {
                        List<Peer> relays = getRelayPeers(context, new HashSet<>());
                        Peer direct = getDirectPeer(context);
                        createPeer(context, host, relays, direct);
                    }
                }

            } catch (Throwable e) {
                Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
            }
        }
    }


    private static boolean isDirectPeerValid(@NonNull Context context,
                                             @NonNull threads.core.api.Peer peer) {
        checkNotNull(context);
        checkNotNull(peer);
        final int thresholdMax = Preferences.getMaxThreshold(context);
        AtomicBoolean peerValid = new AtomicBoolean(false);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();

        if (ipfs != null) {
            String ma = peer.getAdditional(Content.PEER);
            if (!ma.isEmpty()) {
                String pid = peer.getAdditional(Content.PID);

                if (!ipfs.isConnected(PID.create(pid))) {
                    boolean connect = ipfs.swarmConnect(
                            ma + "/" + IPFS.Style.ipfs.name() + "/" + pid,
                            thresholdMax);
                    if (connect) {
                        peerValid.set(true);
                    } else {
                        Singleton.getInstance(context).getConsoleListener().debug(
                                "Direct peer not valid anymore : " + pid);
                    }
                } else {
                    peerValid.set(true);
                }
            }
        }

        return peerValid.get();
    }

    private static int updatePeerRelays(@NonNull Context context,
                                        @NonNull threads.core.api.Peer peer) {
        checkNotNull(context);
        checkNotNull(peer);
        final int thresholdMax = Preferences.getMaxThreshold(context);
        AtomicInteger connected = new AtomicInteger(0);
        AtomicBoolean peerChanged = new AtomicBoolean(false);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        Addresses addresses = peer.getAddresses();
        if (ipfs != null) {
            for (String relay : addresses.keySet()) {
                try {
                    String ma = addresses.get(relay);
                    checkNotNull(ma);
                    PID relayPID = PID.create(relay);

                    if (!ipfs.isConnected(relayPID)) {
                        boolean connect = ipfs.swarmConnect(
                                ma + "/" + IPFS.Style.ipfs.name() + "/" + relay,
                                thresholdMax);
                        if (connect) {
                            connected.incrementAndGet();
                        } else {
                            Singleton.getInstance(context).getConsoleListener().debug(
                                    "Relay not valid anymore : " + relayPID);
                            peer.removeAddress(relay); // remove entry (not valid anymore)
                            peerChanged.set(true);
                        }
                    } else {
                        connected.incrementAndGet();
                    }

                } catch (Throwable e) {
                    Singleton.getInstance(context).getConsoleListener().debug(
                            "" + e.getLocalizedMessage());
                }
            }
        }
        if (peerChanged.get()) {
            Singleton.getInstance(context).getThreads().storePeer(peer);
        }
        return connected.get();
    }


    private static void createPeer(@NonNull Context context,
                                   @NonNull PID user,
                                   @NonNull List<Peer> relays,
                                   @Nullable Peer direct) {
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        if (ipfs != null) {
            try {
                threads.core.api.Peer peer = threads.createPeer(user);
                String alias = threads.getUserAlias(user);
                peer.addAdditional(Content.ALIAS, alias, false);
                if (direct != null) {
                    peer.addAdditional(Content.PID, direct.getPid().getPid(), false);
                    peer.addAdditional(Content.PEER, direct.getMultiAddress(), false);
                }
                // add relay address
                for (Peer relay : relays) {
                    peer.addAddress(relay.getPid(), relay.getMultiAddress());
                }
                threads.storePeer(peer);

                insertPeer(context, peer);

            } catch (Throwable e) {
                Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
            }
        }
    }

    public static threads.core.api.Peer getPeer(@NonNull Context context, @NonNull PID user) {
        checkNotNull(context);
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        if (PeerService.isSupportPeerDiscovery(context)) {
            return threads.getPeer(iota, user);
        }
        return null;
    }

    private static void insertPeer(@NonNull Context context, @NonNull threads.core.api.Peer peer) {
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        checkNotNull(iota);
        threads.setHash(peer, null);

        long start = System.currentTimeMillis();

        boolean success = threads.insertPeer(iota, peer);

        long time = (System.currentTimeMillis() - start) / 1000;
        if (success) {
            Singleton.getInstance(context).getConsoleListener().info(
                    "Success store peer : " + time + " [s]");
        } else {
            Singleton.getInstance(context).getConsoleListener().error(
                    "Failed store peer : " + time + " [s]");
        }

    }

    private static void updatePeer(@NonNull Context context,
                                   @NonNull threads.core.api.Peer peer,
                                   @NonNull List<Peer> relays,
                                   @Nullable Peer direct) {
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        boolean update = false;
        if (ipfs != null) {
            try {
                if (direct != null) {
                    if (!direct.getMultiAddress().equals(peer.getAdditional(Content.PEER))) {
                        update = true;
                        peer.addAdditional(Content.PID, direct.getPid().getPid(), false);
                        peer.addAdditional(Content.PEER, direct.getMultiAddress(), false);
                    }
                }

                Addresses previous = peer.getAddresses();
                // add relay address
                for (Peer relay : relays) {
                    peer.addAddress(relay.getPid(), relay.getMultiAddress());
                }
                if (!previous.equals(peer.getAddresses())) {
                    update = true;
                }

                // check if both are differ
                if (update) {
                    threads.updatePeer(peer);
                    insertPeer(context, peer);
                }

            } catch (Throwable e) {
                Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
            }
        }
    }

    @Nullable
    private static Peer getDirectPeer(@NonNull Context context) {
        checkNotNull(context);

        // TODO find relay peers directly with a function dht_findPeers or dht_findProvs

        final int thresholdMax = Preferences.getMaxThreshold(context);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            List<Peer> peers = ipfs.swarmPeers();
            for (Peer peer : peers) {
                // check if if can work as relay
                if (!peer.isRelay()) { // NOT a relay
                    if (ipfs.swarmConnect(peer, thresholdMax)) {
                        Singleton.getInstance(context).getConsoleListener().debug(
                                "Direct Swarm Peer : " + peer.toString());
                        return peer;
                    }
                }
            }
        }
        return null;
    }

    @NonNull
    private static List<Peer> getRelayPeers(@NonNull Context context, @NonNull Set<String> ignoreList) {
        checkNotNull(context);
        checkNotNull(ignoreList);
        // TODO find relay peers directly with a function dht_findPeers or dht_findProvs

        List<Peer> relays = new ArrayList<>();

        final int thresholdMax = Preferences.getMaxThreshold(context);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            List<Peer> peers = ipfs.swarmPeers();
            for (Peer peer : peers) {
                // check if if can work as relay
                if (!ignoreList.contains(peer.getPid().getPid())) {
                    if (peer.isRelay()) {

                        if (ipfs.swarmConnect(peer, thresholdMax)) {
                            Singleton.getInstance(context).getConsoleListener().debug(
                                    "Detect Swarm Relay Peer : " + peer.toString());
                            relays.add(peer);
                        }

                    }

                }
                if (relays.size() >= MAX_RELAYS) {
                    break;
                }
            }
        }
        return relays;
    }
}
