package threads.share;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;

import threads.core.Network;
import threads.core.Preferences;
import threads.core.Singleton;
import threads.core.THREADS;
import threads.core.api.Addresses;
import threads.core.api.Content;
import threads.core.api.Members;
import threads.core.api.Note;
import threads.core.api.Peer;
import threads.core.api.Thread;
import threads.core.api.User;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkNotNull;

public class ConnectService {

    private static final String TAG = ConnectService.class.getSimpleName();


    public static List<Future<Boolean>> connectMembersAsync(
            @NonNull Context context,
            @NonNull threads.core.api.Thread thread,
            int timeout) {
        checkNotNull(context);
        checkNotNull(thread);
        ArrayList<Future<Boolean>> futures = new ArrayList<>();
        ExecutorService executorService = Executors.newFixedThreadPool(5);
        Members members = thread.getMembers();
        for (PID pid : members) {
            Future<Boolean> future = executorService.submit(() -> {
                try {
                    return connectUser(context, pid, timeout);
                } catch (Throwable e) {
                    Log.e(TAG, e.getLocalizedMessage(), e);
                    return false;
                }
            });
            futures.add(future);
        }
        return futures;
    }

    public static List<Future<Boolean>> publishNoteAsync(@NonNull Context context,
                                                         @NonNull Note note) {

        checkNotNull(context);
        checkNotNull(note);

        final THREADS threads = Singleton.getInstance(context).getThreads();

        ArrayList<Future<Boolean>> futures = new ArrayList<>();
        ExecutorService executorService = Executors.newFixedThreadPool(5);
        Thread thread = threads.getThread(note);
        checkNotNull(thread);
        Members members = thread.getMembers();
        for (PID pid : members) {
            Future<Boolean> future = executorService.submit(() -> {
                try {

                    boolean result = connectUser(context, pid);
                    if (result) {
                        User user = threads.getUserByPID(pid);
                        checkNotNull(user); // user must be defined here
                        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
                        if (ipfs != null) {
                            return threads.publishNote(ipfs, user, note);
                        } else {
                            return false;
                        }
                    }
                    return result;
                } catch (Throwable e) {
                    Log.e(TAG, e.getLocalizedMessage(), e);
                    return false;
                }
            });
            futures.add(future);
        }
        return futures;
    }

    public static List<Future<Boolean>> connectMembersAsync(
            @NonNull Context context,
            @NonNull Thread thread) {

        checkNotNull(context);
        checkNotNull(thread);

        ArrayList<Future<Boolean>> futures = new ArrayList<>();
        ExecutorService executorService = Executors.newFixedThreadPool(5);
        Members members = thread.getMembers();
        for (PID pid : members) {
            Future<Boolean> future = executorService.submit(() -> {
                try {
                    return connectUser(context, pid);
                } catch (Throwable e) {
                    Log.e(TAG, e.getLocalizedMessage(), e);
                    return false;
                }
            });
            futures.add(future);
        }
        return futures;
    }


    public static boolean publishNote(@NonNull Context context,
                                      @NonNull Note note) {
        checkNotNull(context);
        checkNotNull(note);

        AtomicBoolean allSend = new AtomicBoolean(true);

        List<Future<Boolean>> futures = publishNoteAsync(context, note);
        for (Future<Boolean> future : futures) {
            try {
                if (!future.get()) {
                    allSend.set(false);
                }
            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
                allSend.set(false);
            }
        }
        return allSend.get();
    }

    public static boolean connectMembers(@NonNull Context context,
                                         @NonNull Thread thread,
                                         boolean shortcut) {
        checkNotNull(context);
        checkNotNull(thread);

        boolean result = true;

        List<Future<Boolean>> futures = connectMembersAsync(context, thread);
        for (Future<Boolean> future : futures) {
            try {
                if (!future.get()) {
                    result = false;
                    if (shortcut) {
                        return result;
                    }
                }
            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
                result = false;
            }
        }
        return result;
    }

    public static boolean connectMembers(@NonNull Context context,
                                         @NonNull threads.core.api.Thread thread,
                                         int timeout,
                                         boolean shortcut) {
        checkNotNull(context);
        checkNotNull(thread);
        boolean result = true;
        List<Future<Boolean>> futures = connectMembersAsync(context, thread, timeout);
        for (Future<Boolean> future : futures) {
            try {
                if (!future.get()) {
                    result = false;
                    if (shortcut) {
                        return result;
                    }
                }
            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
                result = false;
            }
        }
        return result;
    }


    private static void connectDirect(@NonNull Context context,
                                      @NonNull threads.core.api.Peer peer) {
        checkNotNull(context);
        checkNotNull(peer);

        final int thresholdMax = Preferences.getMaxThreshold(context);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();

        if (ipfs != null) {
            String ma = peer.getAdditional(Content.PEER);
            if (!ma.isEmpty()) {
                String pid = peer.getAdditional(Content.PID);

                if (!ipfs.isConnected(PID.create(pid))) {
                    ipfs.swarmConnect(
                            ma + "/" + IPFS.Style.ipfs.name() + "/" + pid,
                            thresholdMax);
                }
            }
        }

    }

    private static boolean connectRelays(@NonNull Context context,
                                         @NonNull threads.core.api.Peer peer,
                                         long start,
                                         int timeout) {
        checkNotNull(context);
        checkNotNull(peer);
        final int minThreshold = Preferences.getMinThreshold(context);
        final Threshold threshold = new Threshold(minThreshold);
        final boolean dialRelay = Preferences.isDialRelay(context);
        final AtomicBoolean connected = new AtomicBoolean(false);


        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        Addresses addresses = peer.getAddresses();
        if (ipfs != null) {
            for (String relay : addresses.keySet()) {
                try {
                    String ma = addresses.get(relay);
                    checkNotNull(ma);


                    // pre-condition
                    boolean relayConnected = ipfs.isConnected(PID.create(relay));
                    if (!relayConnected) {
                        relayConnected = ipfs.swarmConnect(
                                ma + "/" + IPFS.Style.ipfs.name() + "/" + relay,
                                time(threshold, start, timeout));
                    }

                    if (dialRelay && relayConnected) {
                        ma = ma.concat("/" + IPFS.Style.ipfs.name() + "/" + relay +
                                "/p2p-circuit/" + IPFS.Style.ipfs.name() + "/" +
                                peer.getPID().getPid());
                        boolean connect = ipfs.swarmConnect(ma, time(threshold, start, timeout));
                        if (connect) {
                            connected.set(true);
                        }
                    }

                } catch (Throwable e) {
                    Log.e(TAG, e.getLocalizedMessage(), e);
                }
            }
        }

        return connected.get();
    }


    private static int time(Threshold threshold, long start, int timeout) {
        if (timeout > 0) {
            int passedTime = (int) (System.currentTimeMillis() - start);
            if (passedTime < timeout) {
                if (threshold.maxThreshold > 0) {
                    return threshold.maxThreshold;
                } else {
                    return timeout - passedTime;
                }
            }
            return threshold.minThreshold;
        }
        return timeout;
    }

    public static boolean connectUser(@NonNull Context context, @NonNull PID user) {
        checkNotNull(context);
        checkNotNull(user);
        final int timeout = Preferences.getConnectionTimeout(context);
        return connectUser(context, user, timeout);
    }

    public static boolean connectUser(@NonNull Context context, @NonNull PID user, int timeout) {
        checkNotNull(context);
        checkNotNull(user);

        if (!Network.isConnected(context)) {
            return false;
        }


        final int minThreshold = Preferences.getMinThreshold(context);

        final Threshold threshold = new Threshold(minThreshold);
        final long start = System.currentTimeMillis();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();

        if (ipfs != null) {
            // first check if already connected

            if (ipfs.isConnected(user)) {
                return true;
            }


            IOTA iota = Singleton.getInstance(context).getIota();
            checkNotNull(iota);
            Peer peer = threads.getPeer(iota, user);

            if (peer != null) {

                if (connectRelays(context, peer, start, time(threshold, start, timeout))) {
                    return true;
                }
                connectDirect(context, peer);
            }


            boolean result = ipfs.swarmConnect(user, time(threshold, start, timeout));
            if (result) {
                return true;
            }
            return ipfs.isConnected(user);
        }
        return false;
    }


    public static boolean swarm_connect(@NonNull Context context, @NonNull PID pid, int timeout) {

        checkNotNull(pid);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            if (ipfs.isConnected(pid)) {
                return true;
            }
            return ipfs.swarmConnect(pid, timeout);
        }
        return false;
    }


    private static class Threshold {

        private final int minThreshold;
        private final int maxThreshold;

        Threshold(int minThreshold) {
            this.minThreshold = minThreshold;
            this.maxThreshold = 0;
        }

        Threshold(int minThreshold, int maxThreshold) {
            this.minThreshold = minThreshold;
            this.maxThreshold = maxThreshold;
        }

        public int getMinThreshold() {
            return minThreshold;
        }

        public int getMaxThreshold() {
            return maxThreshold;
        }
    }
}
