package threads.ipfs;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.j256.simplemagic.ContentInfo;
import com.j256.simplemagic.ContentInfoUtil;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.ServerSocket;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;

import mobile.Listener;
import mobile.LoaderStream;
import mobile.LsInfoClose;
import mobile.Node;
import mobile.Reader;
import mobile.Writer;

public class IPFS implements Listener {

    private static final String RANDOM_SWARM_KEY = "randomSwarmKey";
    private static final String PREF_KEY = "prefKey";
    private static final String PID_KEY = "pidKey";
    private static final String SWARM_PORT_KEY = "swarmPortKey";
    private static final String QUIC_KEY = "quicKey";
    private static final String DISABLE_NAT_PORT_MAP_KEY = "natPortMapKey";
    private static final String NAT_SERVICE_KEY = "noFetchKey";
    private static final String ENABLE_AUTO_RELAY_KEY = "enableAutoRelayKey";
    private static final String RELAY_HOP_KEY = "relayHopKey";
    private static final String CONN_MGR_CONFIG_TYPE_KEY = "connMgrConfigTypeKey";
    private static final String ROUTING_TYPE_KEY = "routingTypeKey";
    private static final String ROUTER_ENUM_KEY = "routerEnumKey";
    private static final String HIGH_WATER_KEY = "highWaterKey";
    private static final String LOW_WATER_KEY = "lowWaterKey";
    private static final String GRACE_PERIOD_KEY = "gracePeriodKey";
    private static final String PREFER_TLS_KEY = "preferTLSKey";
    private static final String TOPIC_KEY = "prefTopicKey";
    private static final String PUBLIC_KEY = "publicKey";
    private static final String PRIVATE_KEY = "privateKey";
    private static final int BLOCK_SIZE = 262158;
    private static final String P2P_CIRCUIT = "p2p-circuit";
    private static final String CONFIG_FILE_NAME = "config";
    private static final long TIMEOUT = 30000L;
    private static final String TAG = IPFS.class.getSimpleName();
    private static IPFS INSTANCE = null;
    @Nullable
    private static PubSubHandler HANDLER = null;
    private final File baseDir;
    private final File cacheDir;
    private final Node node;
    private final PubSubReader pubsubReader;
    private final ContentInfoUtil util;
    private final Hashtable<String, Future> topics = new Hashtable<>();
    private Gson gson = new Gson();

    private IPFS(@NonNull Builder builder) throws Exception {
        this.baseDir = builder.context.getFilesDir();
        this.cacheDir = builder.context.getCacheDir();
        this.util = new ContentInfoUtil(builder.context);
        this.pubsubReader = builder.pubSubReader;

        boolean init = !existConfigFile();

        node = new Node(this, this.baseDir.getAbsolutePath());

        if (init) {
            node.init();
        }

        node.database();
        node.identity();
        setPID(builder.context, PID.create(node.getPid()));
        setPublicKey(builder.context, node.getPublicKey());
        setPrivateKey(builder.context, node.getPrivateKey());

        Config config = getConfig();

        configTune(config,
                builder.addresses,
                builder.experimental,
                builder.pubSubConfig,
                builder.discovery,
                builder.swarm,
                builder.routing);

    }

    private static void setPublicKey(@NonNull Context context, @NonNull String key) {
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(PUBLIC_KEY, key);
        editor.apply();
    }

    private static void setPrivateKey(@NonNull Context context, @NonNull String key) {
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(PRIVATE_KEY, key);
        editor.apply();
    }

    @NonNull
    public static String getPublicKey(@NonNull Context context) {
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getString(PUBLIC_KEY, "");

    }

    @NonNull
    public static String getPrivateKey(@NonNull Context context) {
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getString(PRIVATE_KEY, "");

    }

    private static void setPID(@NonNull Context context, @NonNull PID pid) {
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(PID_KEY, pid.getPid());
        editor.apply();
    }

    @Nullable
    public static PID getPID(@NonNull Context context) {
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        String pid = sharedPref.getString(PID_KEY, "");
        if (pid.isEmpty()) {
            return null;
        }
        return PID.create(pid);
    }

    public static boolean deleteConfigFile(@NonNull Context context) {
        try {
            File dir = context.getFilesDir();
            if (dir.exists() && dir.isDirectory()) {
                File config = new File(dir, CONFIG_FILE_NAME);
                if (config.exists()) {
                    return config.delete();
                }
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return false;
    }

    @Nullable
    public static PubSubHandler getPubSubHandler() {
        return HANDLER;
    }

    public static void setPubSubHandler(@Nullable PubSubHandler pubsubHandler) {
        HANDLER = pubsubHandler;
    }

    @NonNull
    public static String getDefaultTopic(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getString(TOPIC_KEY, "pubsub");
    }

    public static void setDefaultTopic(@NonNull Context context, @NonNull String topic) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(TOPIC_KEY, topic);
        editor.apply();

    }

    public static void setSwarmPort(@NonNull Context context, int port) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putInt(SWARM_PORT_KEY, port);
        editor.apply();
    }

    public static int getSwarmPort(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getInt(SWARM_PORT_KEY, 4001);
    }

    public static boolean isRandomSwarmPort(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(RANDOM_SWARM_KEY, false);
    }

    public static void setRandomSwarmPort(@NonNull Context context, boolean enable) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(RANDOM_SWARM_KEY, enable);
        editor.apply();
    }

    public static boolean isQUICEnabled(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(QUIC_KEY, false);

    }

    public static void setQUICEnabled(@NonNull Context context, boolean enable) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(QUIC_KEY, enable);
        editor.apply();
    }

    public static boolean isPreferTLS(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(PREFER_TLS_KEY, false);
    }

    public static void setPreferTLS(@NonNull Context context, boolean enable) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(PREFER_TLS_KEY, enable);
        editor.apply();
    }

    public static PubSubConfig.RouterEnum getPubSubRouter(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return PubSubConfig.RouterEnum.valueOf(
                sharedPref.getString(ROUTER_ENUM_KEY, PubSubConfig.RouterEnum.floodsub.name()));
    }

    public static void setPubSubRouter(@NonNull Context context,
                                       @NonNull PubSubConfig.RouterEnum routerEnum) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(ROUTER_ENUM_KEY, routerEnum.name());
        editor.apply();
    }

    @NonNull
    public static RoutingConfig.TypeEnum getRoutingType(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return RoutingConfig.TypeEnum.valueOf(
                sharedPref.getString(ROUTING_TYPE_KEY, RoutingConfig.TypeEnum.dhtclient.name()));
    }

    public static void setRoutingType(@NonNull Context context,
                                      @NonNull RoutingConfig.TypeEnum routingType) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(ROUTING_TYPE_KEY, routingType.name());
        editor.apply();
    }

    @NonNull
    public static ConnMgrConfig.TypeEnum getConnMgrConfigType(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return ConnMgrConfig.TypeEnum.valueOf(
                sharedPref.getString(CONN_MGR_CONFIG_TYPE_KEY, ConnMgrConfig.TypeEnum.basic.name()));
    }

    public static void setConnMgrConfigType(@NonNull Context context, ConnMgrConfig.TypeEnum typeEnum) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(CONN_MGR_CONFIG_TYPE_KEY, typeEnum.name());
        editor.apply();
    }

    public static void setLowWater(@NonNull Context context, int lowWater) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putInt(LOW_WATER_KEY, lowWater);
        editor.apply();
    }

    public static int getLowWater(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getInt(LOW_WATER_KEY, 20);
    }


    public static boolean isDisableNatPortMap(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(DISABLE_NAT_PORT_MAP_KEY, false);

    }

    public static void setDisableNatPortMap(@NonNull Context context, boolean enable) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(DISABLE_NAT_PORT_MAP_KEY, enable);
        editor.apply();
    }

    public static boolean isAutoRelayEnabled(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(ENABLE_AUTO_RELAY_KEY, false);

    }

    public static void setAutoRelayEnabled(@NonNull Context context, boolean enable) {

        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(ENABLE_AUTO_RELAY_KEY, enable);
        editor.apply();
    }


    public static boolean isAutoNATServiceEnabled(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(NAT_SERVICE_KEY, false);
    }

    public static void setAutoNATServiceEnabled(@NonNull Context context, boolean natService) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(NAT_SERVICE_KEY, natService);
        editor.apply();
    }

    public static boolean isRelayHopEnabled(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(RELAY_HOP_KEY, false);
    }

    public static void setRelayHopEnabled(@NonNull Context context, boolean relayHop) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(RELAY_HOP_KEY, relayHop);
        editor.apply();
    }

    @NonNull
    public static String getGracePeriod(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getString(GRACE_PERIOD_KEY, "30s");
    }

    public static void setGracePeriod(@NonNull Context context, @NonNull String gracePeriod) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(GRACE_PERIOD_KEY, gracePeriod);
        editor.apply();

    }

    public static void setHighWater(@NonNull Context context, int highWater) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putInt(HIGH_WATER_KEY, highWater);
        editor.apply();
    }

    public static int getHighWater(@NonNull Context context) {

        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getInt(HIGH_WATER_KEY, 40);
    }

    @NonNull
    public static IPFS getInstance(@NonNull Context context) {
        if (INSTANCE == null) {
            synchronized (IPFS.class) {
                if (INSTANCE == null) {
                    PubSubReader pubsubReader = (message) -> {
                        if (HANDLER != null) {
                            HANDLER.receive(message);
                        }
                    };
                    int swarmPort = getSwarmPort(context);

                    boolean changedPort = false;
                    if (isRandomSwarmPort(context)) {
                        changedPort = true;
                        swarmPort = nextFreePort();
                    }


                    Integer quicPort = null;
                    if (isQUICEnabled(context)) {
                        quicPort = swarmPort;
                    }

                    if (changedPort) {
                        setSwarmPort(context, swarmPort);
                    }
                    AddressesConfig addresses = AddressesConfig.create(
                            swarmPort, quicPort);

                    ExperimentalConfig experimental = ExperimentalConfig.create();
                    experimental.setQUIC(isQUICEnabled(context));
                    experimental.setPreferTLS(isPreferTLS(context));


                    PubSubConfig pubsubConfig = PubSubConfig.create();
                    pubsubConfig.setRouter(getPubSubRouter(context));


                    SwarmConfig swarmConfig = SwarmConfig.create();
                    swarmConfig.setDisableBandwidthMetrics(true);
                    swarmConfig.setDisableNatPortMap(isDisableNatPortMap(context));
                    swarmConfig.setDisableRelay(false);
                    swarmConfig.setEnableAutoRelay(isAutoRelayEnabled(context));
                    swarmConfig.setEnableAutoNATService(isAutoNATServiceEnabled(context));
                    swarmConfig.setEnableRelayHop(isRelayHopEnabled(context));

                    ConnMgrConfig mgr = swarmConfig.getConnMgr();
                    mgr.setGracePeriod(getGracePeriod(context));
                    mgr.setHighWater(getHighWater(context));
                    mgr.setLowWater(getLowWater(context));
                    mgr.setType(getConnMgrConfigType(context));

                    DiscoveryConfig discoveryConfig = DiscoveryConfig.create();
                    discoveryConfig.getMdns().setEnabled(false);


                    RoutingConfig routingConfig = RoutingConfig.create();
                    routingConfig.setType(getRoutingType(context));


                    try {
                        INSTANCE = new Builder().
                                context(context).
                                pubSubReader(pubsubReader).
                                addresses(addresses).
                                experimental(experimental).
                                pubSubConfig(pubsubConfig).
                                discovery(discoveryConfig).
                                swarm(swarmConfig).
                                routing(routingConfig).build();

                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        return INSTANCE;
    }

    public static String getDeviceName() {
        try {
            String manufacturer = Build.MANUFACTURER;
            String model = Build.MODEL;
            if (model.startsWith(manufacturer)) {
                return capitalize(model);
            }
            return capitalize(manufacturer) + " " + model;
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return "";
    }

    private static String capitalize(String str) {
        if (TextUtils.isEmpty(str)) {
            return str;
        }
        char[] arr = str.toCharArray();
        boolean capitalizeNext = true;
        String phrase = "";
        for (char c : arr) {
            if (capitalizeNext && Character.isLetter(c)) {
                phrase = phrase.concat("" + Character.toUpperCase(c));
                capitalizeNext = false;
                continue;
            } else if (Character.isWhitespace(c)) {
                capitalizeNext = true;
            }
            phrase = phrase.concat("" + c);
        }
        return phrase;
    }

    public static int nextFreePort() {
        return nextFreePort(4001, 65535);
    }

    public static int nextFreePort(int from, int to) {
        int port = ThreadLocalRandom.current().nextInt(from, to);
        while (true) {
            if (isLocalPortFree(port)) {
                return port;
            } else {
                port = ThreadLocalRandom.current().nextInt(from, to);
            }
        }
    }

    public static boolean isLocalPortFree(int port) {
        try {
            new ServerSocket(port).close();
            return true;
        } catch (IOException e) {
            return false;
        }
    }

    public static void logBaseDir(@NonNull Context context) {
        try {
            File[] files = context.getFilesDir().listFiles();
            if (files != null) {
                for (File file : files) {
                    Log.e(TAG, "" + file.length() + " " + file.getAbsolutePath());
                    if (file.isDirectory()) {
                        File[] children = file.listFiles();
                        if (children != null) {
                            for (File child : children) {
                                Log.e(TAG, "" + child.length() + " " + child.getAbsolutePath());
                            }
                        }
                    }
                }
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
    }

    private static void cleanDir(@NonNull File dir) {
        try {
            File[] files = dir.listFiles();
            if (files != null) {
                for (File file : files) {
                    if (file.isDirectory()) {
                        cleanDir(file);
                        boolean result = file.delete();
                        if (!result) {
                            Log.e(TAG, "File not deleted.");
                        }
                    } else {
                        boolean result = file.delete();
                        if (!result) {
                            Log.e(TAG, "File not deleted.");
                        }
                    }
                }
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
    }

    public static void cleanBaseDir(@NonNull Context context) {
        try {
            File[] files = context.getFilesDir().listFiles();
            if (files != null) {
                for (File file : files) {
                    if (file.isDirectory()) {
                        cleanDir(file);
                        boolean result = file.delete();
                        if (!result) {
                            Log.e(TAG, "File not deleted.");
                        }
                    } else {
                        boolean result = file.delete();
                        if (!result) {
                            Log.e(TAG, "File not deleted.");
                        }
                    }
                }
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
    }

    public static void cleanCacheDir(@NonNull Context context) {

        try {
            File[] files = context.getCacheDir().listFiles();
            if (files != null) {
                for (File file : files) {
                    if (file.isDirectory()) {
                        cleanDir(file);
                        boolean result = file.delete();
                        if (!result) {
                            Log.e(TAG, "File not deleted.");
                        }
                    } else {
                        boolean result = file.delete();
                        if (!result) {
                            Log.e(TAG, "File not deleted.");
                        }
                    }
                }
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
    }

    public void bootstrap(int timeout) {

        if (!isDaemonRunning()) {
            throw new RuntimeException("Daemon is not running");
        }

        for (String address : Config.Bootstrap) {

            boolean result = swarmConnect(address, timeout);
            if (result) {
                Log.w(TAG, result + " \n Bootstrap : " + address);
            }

        }
    }

    public boolean addPubSubTopic(@NonNull Context context, @NonNull String topic) {

        if (!isDaemonRunning()) {
            return false;
        }
        // check already listen to topic
        if (topics.containsKey(topic)) {
            return true;
        }


        ExecutorService executor = Executors.newSingleThreadExecutor();
        topics.put(topic, executor.submit(() -> {
            try {
                pubSubSub(topic);
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage());
            }
        }));
        return false;
    }

    public boolean relay(@NonNull Peer relay, @NonNull PID pid, int timeout) {

        return swarmConnect(relayAddress(relay, pid), timeout);

    }

    @NonNull
    private File getConfigFile() {
        return new File(baseDir, CONFIG_FILE_NAME);
    }

    private boolean existConfigFile() {
        return getConfigFile().exists();
    }

    @NonNull
    public String config_show() throws Exception {
        return getConfigAsString();
    }

    @NonNull
    public String getConfigAsString() throws Exception {

        if (!existConfigFile()) {
            throw new RuntimeException("Config file does not exist.");
        }
        return FileUtils.readFileToString(getConfigFile(), StandardCharsets.UTF_8);

    }


    public List<PID> dhtFindProviders(@NonNull CID cid, int numProvs, int timeout) {


        List<PID> providers = new ArrayList<>();
        if (!isDaemonRunning()) {
            return providers;
        }
        try {

            node.dhtFindProvs(cid.getCid(), (pid) ->
                            providers.add(PID.create(pid))
                    , numProvs, true, timeout);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return providers;
    }

    public void dhtPublish(@NonNull CID cid, boolean recursive, int timeout) {

        if (!isDaemonRunning()) {
            throw new RuntimeException("Daemon is not running");
        }
        try {

            node.dhtProvide(cid.getCid(), recursive, true, timeout);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
    }

    @Nullable
    public PeerInfo id(@NonNull Peer peer, int timeout) {

        if (!isDaemonRunning()) {
            return null;
        }
        return id(peer.getPid(), timeout);
    }

    @Nullable
    public PeerInfo id(@NonNull PID pid, int timeout) {


        try {
            if (!isDaemonRunning()) {
                return null;
            }
            String json = node.idWithTimeout(pid.getPid(), timeout);
            Map map = gson.fromJson(json, Map.class);
            return PeerInfo.create(map);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }

        return null;

    }

    @Nullable
    public PeerInfo id() {

        try {
            if (!isDaemonRunning()) {
                return null;
            }
            String json = node.id();
            Map map = gson.fromJson(json, Map.class);
            return PeerInfo.create(map);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    public boolean checkRelay(@NonNull Peer peer, int timeout) {


        AtomicBoolean success = new AtomicBoolean(false);
        //noinspection CatchMayIgnoreException
        try {
            if (!isDaemonRunning()) {
                return false;
            }
            PID pid = PID.create("QmWATWQ7fVPP2EFGu71UkfnqhYXDYH566qy47CnJDgvs8u"); // DUMMY

            String address = relayAddress(peer, pid);
            node.swarmConnect(address, timeout);

        } catch (Throwable e) {
            String line = e.getLocalizedMessage();
            if (line != null) {
                if (line.contains("HOP_NO_CONN_TO_DST")) {
                    success.set(true);
                }
            }
        }
        return success.get();
    }

    public void pubSubPub(@NonNull final String topic, @NonNull final String message, int timeout) {

        if (!isDaemonRunning()) {
            throw new RuntimeException("Daemon is not running");
        }
        try {
            byte[] data = Base64.encodeBase64(message.getBytes());
            node.pubsubPub(topic, data);
            Thread.sleep(timeout);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }


    public void pubSubSub(@NonNull String topic) throws Exception {

        if (!isDaemonRunning()) {
            throw new RuntimeException("Daemon is not running");
        }
        node.pubsubSub(topic);
    }

    @NonNull
    public String relayAddress(@NonNull Peer relay, @NonNull PID pid) {

        return relayAddress(relay.getMultiAddress(), relay.getPid(), pid);
    }

    @NonNull
    public String relayAddress(@NonNull String address, @NonNull PID relay, @NonNull PID pid) {

        return address + "/" + Style.p2p.name() + "/" + relay.getPid() +
                "/" + P2P_CIRCUIT + "/" + Style.p2p.name() + "/" + pid.getPid();
    }

    public boolean specificRelay(@NonNull PID relay,
                                 @NonNull PID pid,
                                 int timeout) {


        if (!isDaemonRunning()) {
            return false;
        }
        if (swarmConnect(relay, timeout)) {
            Peer peer = swarmPeer(relay);
            if (peer != null) {
                return swarmConnect(relayAddress(peer, pid), timeout);
            }
        }


        return false;
    }

    public boolean relay(@NonNull PID relay, @NonNull PID pid, int timeout) {


        return specificRelay(relay, pid, timeout);

    }

    public boolean arbitraryRelay(@NonNull PID pid, int timeout) {
        if (!isDaemonRunning()) {
            return false;
        }
        return swarmConnect("/" + P2P_CIRCUIT + "/" +
                Style.p2p.name() + "/" + pid.getPid(), timeout);
    }

    public boolean swarmConnect(@NonNull PID pid, int timeout) {
        if (!isDaemonRunning()) {
            return false;
        }
        return swarmConnect("/" + Style.p2p.name() + "/" +
                pid.getPid(), timeout);
    }

    public boolean swarmConnect(@NonNull Peer peer, int timeout) {
        if (!isDaemonRunning()) {
            return false;
        }
        String ma = peer.getMultiAddress() + "/" + Style.p2p.name() + "/" + peer.getPid();
        return swarmConnect(ma, timeout);

    }

    public boolean swarmConnect(@NonNull String multiAddress, int timeout) {
        if (!isDaemonRunning()) {
            return false;
        }
        try {
            return node.swarmConnect(multiAddress, timeout);
        } catch (Throwable e) {
            Log.e(TAG, "swarmConnect " + e.getLocalizedMessage());
        }
        return false;
    }


    public boolean isConnected(@NonNull PID pid) {
        if (!isDaemonRunning()) {
            return false;
        }
        try {
            return node.isConnected(pid.getPid());
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return false;
    }

    @Nullable
    public Peer swarmPeer(@NonNull PID pid) {
        if (!isDaemonRunning()) {
            return null;
        }
        try {
            String json = node.swarmPeer(pid.getPid());
            if (json != null && !json.isEmpty()) {
                Map map = gson.fromJson(json, Map.class);
                if (map != null) {
                    return Peer.create(map);
                }
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return null;
    }

    @NonNull
    public List<Peer> swarmPeers() {
        if (!isDaemonRunning()) {
            return new ArrayList<>();
        }
        List<Peer> peers = swarm_peers();
        peers.sort(Peer::compareTo);
        return peers;
    }

    @NonNull
    private List<Peer> swarm_peers() {

        List<Peer> peers = new ArrayList<>();
        try {


            String json = node.swarmPeers();
            if (json != null && !json.isEmpty()) {
                Map map = gson.fromJson(json, Map.class);
                if (map != null) {

                    Object object = map.get("Peers");
                    if (object instanceof List) {
                        List list = (List) object;
                        if (!list.isEmpty()) {
                            for (Object entry : list) {
                                if (entry instanceof Map) {
                                    Map peer = (Map) entry;
                                    peers.add(Peer.create(peer));
                                }
                            }
                        }
                    }
                }
            }

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return peers;
    }

    public boolean swarmDisconnect(@NonNull PID pid) {
        if (!isDaemonRunning()) {
            return false;
        }
        try {
            return node.swarmDisconnect("/" + Style.p2p.name() + "/" + pid.getPid());
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }

        return false;
    }

    public void rm(@NonNull CID cid) {
        try {
            node.rm(cid.getCid());
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
    }

    @NonNull
    public List<String> pubSubPeers() {

        List<String> peers = new ArrayList<>();
        if (!isDaemonRunning()) {
            return peers;
        }
        try {
            //noinspection Convert2MethodRef
            node.pubsubPeers((peer) -> peers.add(peer));

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return peers;
    }

    @NonNull
    public List<PID> pubSubPeers(@NonNull String topic) {

        List<PID> pidList = new ArrayList<>();
        if (!isDaemonRunning()) {
            return pidList;
        }
        try {

            node.pubsubPeers((peer) -> pidList.add(PID.create(peer)));

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return pidList;
    }

    @NonNull
    public Config getConfig() throws Exception {
        String result = getConfigAsString();
        Gson gson = new Gson();
        return gson.fromJson(result, Config.class);
    }

    public void saveConfig(@NonNull Config config) throws Exception {

        File file = getConfigFile();

        try (FileWriter writer = new FileWriter(file)) {
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            gson.toJson(config, writer);
        }

    }

    private void configTune(@NonNull Config config,
                            @Nullable AddressesConfig addresses,
                            @Nullable ExperimentalConfig experimental,
                            @Nullable PubSubConfig pubsub,
                            @Nullable DiscoveryConfig discovery,
                            @Nullable SwarmConfig swarm,
                            @Nullable RoutingConfig routing) throws Exception {
        boolean changedConfig = false;
        if (experimental != null) {
            changedConfig = true;
            config.setExperimental(experimental);
        }
        if (pubsub != null) {
            changedConfig = true;
            config.setPubsub(pubsub);
        }
        if (addresses != null) {
            changedConfig = true;
            config.setAddresses(addresses);
        }
        if (discovery != null) {
            changedConfig = true;
            config.setDiscovery(discovery);
        }
        if (swarm != null) {
            changedConfig = true;
            config.setSwarm(swarm);
        }
        if (routing != null) {
            changedConfig = true;
            config.setRouting(routing);
        }
        if (changedConfig) {
            saveConfig(config);
        }
    }

    public void daemon(String agent, boolean pubSubEnable) {
        if (isDaemonRunning()) {
            throw new RuntimeException("Daemon is already running");
        }
        AtomicBoolean failure = new AtomicBoolean(false);
        ExecutorService executor = Executors.newSingleThreadExecutor();
        AtomicReference<String> exception = new AtomicReference<>("");
        executor.submit(() -> {
            try {
                node.daemon(agent, pubSubEnable);
            } catch (Throwable e) {
                failure.set(true);
                exception.set("" + e.getLocalizedMessage());
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }
        });
        long time = 0L;
        while (!node.getRunning() && time <= TIMEOUT && !failure.get()) {
            time = time + 100L;
            try {
                Thread.sleep(100);
            } catch (Throwable e) {
                throw new RuntimeException(exception.get());
            }
        }
        if (failure.get()) {
            throw new RuntimeException(exception.get());
        }
    }

    @Nullable
    public CID storeData(@NonNull byte[] data) {

        try (InputStream inputStream = new ByteArrayInputStream(data)) {
            return storeInputStream(inputStream);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
        return null;
    }

    @Nullable
    public CID storeText(@NonNull String content) {
        return storeText(content, "");

    }

    @Nullable
    public CID storeText(@NonNull String content, @NonNull String key) {

        try (InputStream inputStream = new ByteArrayInputStream(content.getBytes())) {
            return storeInputStream(inputStream, key);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
        return null;
    }


    @Nullable
    public List<LinkInfo> info(@NonNull CID cid) {
        if (!isDaemonRunning()) {
            return null;
        }
        List<LinkInfo> infoList = new ArrayList<>();
        try {

            node.info(cid.getCid(), (json) -> {

                try {
                    Map map = gson.fromJson(json, Map.class);
                    LinkInfo info = LinkInfo.create(map);
                    infoList.add(info);
                } catch (Throwable e) {
                    Log.e(TAG, "" + e.getLocalizedMessage());
                }

            });


        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
            return null;
        }
        return infoList;
    }

    @Nullable
    public List<LinkInfo> ls(@NonNull CID cid, @NonNull Closeable closeable) {
        if (!isDaemonRunning()) {
            return null;
        }
        List<LinkInfo> infoList = new ArrayList<>();
        try {

            node.ls(cid.getCid(), new LsInfoClose() {
                @Override
                public boolean close() {
                    return closeable.isClosed();
                }

                @Override
                public void lsLink(String json) {
                    try {
                        Map map = gson.fromJson(json, Map.class);
                        LinkInfo info = LinkInfo.create(map);
                        infoList.add(info);
                    } catch (Throwable e) {
                        Log.e(TAG, "" + e.getLocalizedMessage(), e);
                    }
                }
            });

        } catch (Throwable e) {
            Log.e(TAG, "ls " + e.getLocalizedMessage());
            return null;
        }
        return infoList;
    }


    @Nullable
    public CID storeFile(@NonNull File target) {

        try (InputStream io = new FileInputStream(target)) {
            return storeInputStream(io);
        } catch (Throwable e) {
            Log.e(TAG, "storeFile " + e.getLocalizedMessage());
        }
        return null;
    }

    @NonNull
    public Reader getReader(@NonNull CID cid) throws Exception {
        return node.getReader(cid.getCid());
    }

    @NonNull
    public Writer getWriter() throws Exception {
        return node.getWriter();
    }

    private boolean loadToOutputStream(@NonNull OutputStream outputStream, @NonNull CID cid,
                                       @NonNull String key, @NonNull Progress progress) {

        try (InputStream inputStream = loadStream(cid, progress, key)) {
            if (inputStream == null) {
                return false;
            }
            IOUtils.copy(inputStream, outputStream);
        } catch (Throwable e) {
            Log.e(TAG, "loadToOutputStream " + e.getLocalizedMessage());
            return false;
        }
        return progress.isDone();

    }

    private void getToOutputStream(@NonNull OutputStream outputStream, @NonNull CID cid,
                                   @NonNull String key) throws Exception {
        try (InputStream inputStream = getInputStream(cid, key)) {
            IOUtils.copy(inputStream, outputStream);
        }
    }

    public boolean loadToFile(@NonNull File file, @NonNull CID cid, @NonNull Progress progress) {
        if (!isDaemonRunning()) {
            return false;
        }

        try (FileOutputStream outputStream = new FileOutputStream(file)) {
            return loadToOutputStream(outputStream, cid, "", progress);
        } catch (Throwable e) {
            Log.e(TAG, "loadToFile " + e.getLocalizedMessage());
            return false;
        }
    }

    public void storeToOutputStream(@NonNull OutputStream os, @NonNull CID cid) throws Exception {

        Reader reader = getReader(cid);
        try {

            reader.load(BLOCK_SIZE);
            long read = reader.getRead();
            while (read > 0) {
                byte[] bytes = reader.getData();
                os.write(bytes, 0, bytes.length);

                reader.load(BLOCK_SIZE);
                read = reader.getRead();
            }
        } finally {
            reader.close();
        }

    }

    @NonNull
    private InputStream loadStream(@NonNull CID cid, @NonNull Progress progress) throws Exception {


        PipedOutputStream pos = new PipedOutputStream();
        PipedInputStream pis = new PipedInputStream(pos);

        List<LinkInfo> info = ls(cid, progress);
        if (info == null) {
            throw new RuntimeException("closed " + progress.isClosed());
        }


        final AtomicInteger atomicProgress = new AtomicInteger(0);

        progress.setProgress(0);
        final AtomicBoolean close = new AtomicBoolean(false);
        ExecutorService executor = Executors.newSingleThreadExecutor();
        executor.submit(() -> {


            try {
                node.getStream(cid.getCid(), new LoaderStream() {
                    int totalRead = 0;
                    long size = 0;

                    @Override
                    public boolean close() {
                        return close.get() || progress.isClosed();
                    }


                    @Override
                    public long read(byte[] bytes) {
                        try {
                            pos.write(bytes);

                            int bytesRead = bytes.length;
                            if (bytesRead > 0) {

                                totalRead += bytesRead;
                                if (size > 0) {
                                    int percent = (int) ((totalRead * 100.0f) / size);
                                    if (atomicProgress.getAndSet(percent) < percent) {
                                        progress.setProgress(percent);
                                    }
                                }
                            }
                            progress.setDone(totalRead == size);

                        } catch (Throwable e) {
                            close.set(true);
                            progress.setDone(false);
                            // Ignore exception might be on pipe is closed
                            Log.e(TAG, "loadStream " + e.getLocalizedMessage(), e);
                        }
                        return -1;
                    }

                    @Override
                    public void size(long size) {
                        progress.setSize(size);
                        this.size = size;
                        if (size == 0) { // special case (empty file)
                            progress.setDone(true);
                        }
                    }

                });
            } catch (Throwable e) {
                progress.setDone(false);
                Log.e(TAG, "loadStream " + e.getLocalizedMessage());
            } finally {
                try {
                    pos.close();
                } catch (Throwable ec) {
                    Log.e(TAG, "loadStream " + ec.getLocalizedMessage());
                }
            }


        });


        return pis;


    }

    public void storeToFile(@NonNull File file, @NonNull CID cid) throws Exception {

        try (FileOutputStream fileOutputStream = new FileOutputStream(file)) {
            storeToOutputStream(fileOutputStream, cid);
        }

    }

    @Nullable
    public ContentInfo getContentInfo(@NonNull CID cid) {

        try {
            try (InputStream inputStream = new BufferedInputStream(getInputStream(cid))) {

                return util.findMatch(inputStream);

            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return null;
    }

    @Nullable
    public ContentInfo getContentInfo(@NonNull File file) {

        try {
            return util.findMatch(file);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }
        return null;
    }

    @NonNull
    public File getCacheDir() {
        return this.cacheDir;
    }

    @NonNull
    public File createCacheFile() throws IOException {
        return File.createTempFile("temp", ".cid", getCacheDir());
    }


    @NonNull
    public File createCacheFile(@NonNull CID cid) throws IOException {

        File file = new File(getCacheDir(), cid.getCid());
        if (file.exists()) {
            boolean result = file.delete();
            if (!result) {
                Log.e(TAG, "Deleting failed");
            }
        }
        boolean succes = file.createNewFile();
        if (!succes) {
            Log.e(TAG, "Failed create a new file");
        }
        return file;
    }

    @Nullable
    public CID storeInputStream(@NonNull InputStream inputStream) {


        String res = "";
        try {
            Writer writer = getWriter();
            res = writer.stream(new WriterStream(writer, inputStream));
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }

        if (!res.isEmpty()) {
            return CID.create(res);
        }
        return null;
    }

    @Nullable
    private CID storeInputStream(@NonNull InputStream inputStream, @NonNull String key) throws Exception {

        if (!key.isEmpty()) {
            Key aesKey = Encryption.getKey(key);
            Cipher cipher = Cipher.getInstance("AES");
            cipher.init(Cipher.ENCRYPT_MODE, aesKey);
            try (CipherInputStream cipherStream = new CipherInputStream(inputStream, cipher)) {
                return storeInputStream(cipherStream);
            }
        } else {
            return storeInputStream(inputStream);
        }

    }

    @Nullable
    public String loadText(@NonNull CID cid, @NonNull Progress progress) {
        if (!isDaemonRunning()) {
            return null;
        }
        return loadText(cid, "", progress);
    }

    @Nullable
    public String loadText(@NonNull CID cid, @NonNull String key, @NonNull Progress progress) {
        if (!isDaemonRunning()) {
            return null;
        }
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            boolean success = loadToOutputStream(outputStream, cid, key, progress);
            if (success) {
                return new String(outputStream.toByteArray());
            } else {
                return null;
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
            return null;
        }
    }

    @Nullable
    public String getText(@NonNull CID cid) {
        return getText(cid, "");
    }

    @Nullable
    public String getText(@NonNull CID cid, @NonNull String key) {

        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            getToOutputStream(outputStream, cid, key);
            return new String(outputStream.toByteArray());
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
            return null;
        }
    }

    @Nullable
    public byte[] getData(@NonNull CID cid, @NonNull String key) {

        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            getToOutputStream(outputStream, cid, key);
            return outputStream.toByteArray();
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
            return null;
        }
    }

    @Nullable
    public byte[] getData(@NonNull CID cid) {
        return getData(cid, "");
    }

    @Nullable
    public byte[] loadData(@NonNull CID cid, @NonNull String key, @NonNull Progress progress) {
        if (!isDaemonRunning()) {
            return null;
        }
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            boolean success = loadToOutputStream(outputStream, cid, key, progress);
            if (success) {
                return outputStream.toByteArray();
            } else {
                return null;
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
            return null;
        }

    }

    @Nullable
    public byte[] loadData(@NonNull CID cid, @NonNull Progress progress) {
        if (!isDaemonRunning()) {
            return null;
        }
        return loadData(cid, "", progress);
    }

    @Nullable
    private InputStream loadStream(@NonNull CID cid, @NonNull Progress progress, @NonNull String key) {
        try {
            if (key.isEmpty()) {
                return loadStream(cid, progress);
            } else {
                Key aesKey = Encryption.getKey(key);
                Cipher cipher = Cipher.getInstance("AES");
                cipher.init(Cipher.DECRYPT_MODE, aesKey);
                return new CipherInputStream(loadStream(cid, progress), cipher);
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
        return null;
    }

    @NonNull
    public InputStream getInputStream(@NonNull CID cid, @NonNull String key) throws Exception {

        if (key.isEmpty()) {
            return getInputStream(cid);
        } else {
            Key aesKey = Encryption.getKey(key);
            Cipher cipher = Cipher.getInstance("AES");
            cipher.init(Cipher.DECRYPT_MODE, aesKey);
            return new CipherInputStream(getInputStream(cid), cipher);
        }
    }


    public boolean swarmDisconnect(@NonNull Peer peer) {
        if (!isDaemonRunning()) {
            return false;
        }
        return swarmDisconnect(peer.getPid());
    }

    public void gc() {
        try {
            node.repoGC();
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }

    }

    @Override
    public void error(String message) {
        if (message != null && !message.isEmpty()) {
            Log.e(TAG, "" + message);
        }
    }

    @Override
    public void info(String message) {
        if (message != null && !message.isEmpty()) {
            Log.i(TAG, "" + message);
        }
    }


    @Override
    public void pubsub(final String message, final byte[] data) {
        if (!isDaemonRunning()) {
            throw new RuntimeException("Daemon is not running");
        }
        try {
            PubSubInfo pubSub = PubSubInfo.create(message, data);

            if (pubsubReader != null) {
                pubsubReader.receive(pubSub);
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
        }

    }

    @Override
    public void verbose(String s) {
        Log.i(TAG, "" + s);
    }

    @NonNull
    public InputStream getInputStream(@NonNull CID cid) throws Exception {
        Reader reader = getReader(cid);
        return new ReaderInputStream(reader);

    }

    public boolean isDaemonRunning() {
        return node.getRunning();
    }

    public enum Style {
        ipfs, ipns, p2p
    }

    public interface PubSubHandler {
        void receive(@NonNull PubSubInfo message);
    }

    private static class Builder {

        private Context context;
        private AddressesConfig addresses = null;
        private ExperimentalConfig experimental = null;
        private PubSubConfig pubSubConfig = null;
        private DiscoveryConfig discovery = null;
        private SwarmConfig swarm = null;
        private RoutingConfig routing = null;
        private PubSubReader pubSubReader = null;

        IPFS build() throws Exception {
            return new IPFS(this);
        }


        Builder context(@NonNull Context context) {
            this.context = context;
            return this;
        }

        Builder pubSubReader(@NonNull PubSubReader pubsubReader) {
            this.pubSubReader = pubsubReader;
            return this;
        }


        Builder experimental(@Nullable ExperimentalConfig experimental) {
            this.experimental = experimental;
            return this;
        }

        Builder addresses(@Nullable AddressesConfig addresses) {
            this.addresses = addresses;
            return this;
        }

        Builder pubSubConfig(@Nullable PubSubConfig pubSubConfig) {
            this.pubSubConfig = pubSubConfig;
            return this;
        }

        Builder discovery(@Nullable DiscoveryConfig discovery) {
            this.discovery = discovery;
            return this;
        }

        Builder swarm(@Nullable SwarmConfig swarm) {
            this.swarm = swarm;
            return this;
        }

        Builder routing(@Nullable RoutingConfig routing) {
            this.routing = routing;
            return this;
        }

    }

    private class ReaderInputStream extends InputStream implements AutoCloseable {
        private final Reader mReader;
        private int position = 0;
        private byte[] data = null;

        ReaderInputStream(@NonNull Reader reader) {
            mReader = reader;
        }

        @Override
        public int read() throws IOException {

            try {
                if (data == null) {
                    invalidate();
                    preLoad();
                }
                if (data == null) {
                    return -1;
                }
                if (position < data.length) {
                    byte value = data[position];
                    position++;
                    return (value & 0xff);
                } else {
                    invalidate();
                    if (preLoad()) {
                        byte value = data[position];
                        position++;
                        return (value & 0xff);
                    } else {
                        return -1;
                    }
                }


            } catch (Throwable e) {
                throw new IOException(e);
            }
        }

        private void invalidate() {
            position = 0;
            data = null;
        }


        private boolean preLoad() throws Exception {
            mReader.load(BLOCK_SIZE);
            int read = (int) mReader.getRead();
            if (read > 0) {
                data = new byte[read];
                byte[] values = mReader.getData();
                System.arraycopy(values, 0, data, 0, read);
                return true;
            }
            return false;
        }

        public void close() {
            try {
                mReader.close();
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage());
            }
        }
    }

    private class WriterStream implements mobile.WriterStream {
        private final InputStream mInputStream;
        private final Writer mWriter;

        WriterStream(Writer writer, InputStream inputStream) {
            this.mWriter = writer;
            this.mInputStream = inputStream;
        }


        @Override
        public void load(long size) {
            try {
                byte[] data = new byte[(int) size];

                int read = mInputStream.read(data);
                mWriter.setWritten(read);
                mWriter.setData(data);
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage());
            }

        }
    }
}
