package threads.ipfs;

import android.content.Context;
import android.net.Uri;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.j256.simplemagic.ContentInfo;
import com.j256.simplemagic.ContentInfoUtil;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;

import mobile.FileReader;
import mobile.Listener;
import mobile.Node;
import mobile.PubsubPeer;
import mobile.Stream;
import threads.ipfs.api.AddressesConfig;
import threads.ipfs.api.ApiListener;
import threads.ipfs.api.CID;
import threads.ipfs.api.Config;
import threads.ipfs.api.DiscoveryConfig;
import threads.ipfs.api.Encryption;
import threads.ipfs.api.ExperimentalConfig;
import threads.ipfs.api.LinkInfo;
import threads.ipfs.api.LogListener;
import threads.ipfs.api.PID;
import threads.ipfs.api.Peer;
import threads.ipfs.api.PeerInfo;
import threads.ipfs.api.PubsubConfig;
import threads.ipfs.api.PubsubInfo;
import threads.ipfs.api.PubsubReader;
import threads.ipfs.api.ReproviderConfig;
import threads.ipfs.api.RoutingConfig;
import threads.ipfs.api.SwarmConfig;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class IPFS implements Listener {

    private static final int BLOCK_SIZE = 262158;
    private static final String P2P_CIRCUIT = "p2p-circuit";
    private static final String CONFIG_FILE_NAME = "config";
    private static final String CACHE = "cache";
    private static final long TIMEOUT = 30000L;
    private static final String TAG = IPFS.class.getSimpleName();
    private static IPFS INSTANCE = null;
    private final File baseDir;
    private final File cacheDir;
    private final ApiListener listener;
    private final Node node;
    private final PID pid;
    private final ContentInfoUtil util;
    private Gson gson = new Gson();
    private Hashtable<String, PubsubReader> pubsubReader = new Hashtable<>();

    private IPFS(@NonNull Builder builder) throws Exception {
        checkNotNull(builder);
        this.baseDir = builder.context.getFilesDir();
        checkNotNull(this.baseDir);
        checkArgument(this.baseDir.isDirectory());
        this.cacheDir = builder.context.getCacheDir();
        checkNotNull(this.cacheDir);
        checkArgument(this.cacheDir.isDirectory());
        this.listener = builder.listener;
        this.util = new ContentInfoUtil(builder.context);

        boolean init = !existConfigFile();

        node = new Node(this, this.baseDir.getAbsolutePath());

        if (init) {
            node.init();
        }

        Config config = getConfig();

        configTune(config,
                builder.addresses,
                builder.experimental,
                builder.pubsub,
                builder.discovery,
                builder.swarm,
                builder.routing,
                builder.reprovider);

        pid = config.getIdentity().getPID();
    }

    public static IPFS getInstance(@NonNull Context context,
                                   @NonNull ApiListener listener,
                                   @Nullable AddressesConfig addresses,
                                   @Nullable ExperimentalConfig experimental,
                                   @Nullable PubsubConfig pubsub,
                                   @Nullable DiscoveryConfig discovery,
                                   @Nullable SwarmConfig swarm,
                                   @Nullable RoutingConfig routing,
                                   @Nullable ReproviderConfig reprovider) throws Exception {
        if (INSTANCE == null) {
            INSTANCE = new Builder().
                    context(context).
                    listener(listener).
                    addresses(addresses).
                    experimental(experimental).
                    pubsub(pubsub).
                    discovery(discovery).
                    swarm(swarm).
                    routing(routing).
                    reprovider(reprovider).build();
        }
        return INSTANCE;
    }

    @NonNull
    public PID getPid() {
        return pid;
    }


    public boolean relay(@NonNull Peer relay, @NonNull PID pid, int timeout) {
        checkNotNull(relay);
        checkNotNull(pid);
        checkArgument(timeout > 0);
        return swarmConnect(relayAddress(relay, pid), timeout);

    }

    @NonNull
    private File getConfigFile() {
        return new File(baseDir, CONFIG_FILE_NAME);
    }


    private boolean existConfigFile() {
        return getConfigFile().exists();
    }

    @NonNull
    public String config_show() throws Exception {
        return getConfigAsString();
    }

    @NonNull
    public String getConfigAsString() throws Exception {
        checkArgument(existConfigFile(), "Config file does not exist.");

        return FileUtils.readFileToString(getConfigFile(), StandardCharsets.UTF_8);

    }

    @NonNull
    public String getPrivateKey() {
        checkDaemonRunning();
        return node.getPrivateKey();
    }

    @NonNull
    public String getPublicKey() {
        checkDaemonRunning();
        return node.getPublicKey();
    }


    public List<PID> dhtFindProvs(@NonNull CID cid, int numProvs, int timeout) {
        checkNotNull(cid);
        checkArgument(timeout > 0);
        List<PID> providers = new ArrayList<>();
        try {
            node.dhtFindProvs(cid.getCid(), (pid) ->
                            providers.add(PID.create(pid))
                    , numProvs, true, timeout);
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return providers;
    }


    public void dhtPublish(@NonNull CID cid, boolean recursive, int timeout) {
        checkNotNull(cid);
        checkArgument(timeout > 0);
        checkArgument(checkDaemonRunning());
        try {
            node.dhtProvide(cid.getCid(), recursive, true, timeout);
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
    }

    @Nullable
    public PeerInfo id(@NonNull Peer peer, int timeout) {
        checkNotNull(peer);
        checkArgument(timeout > 0);
        return id(peer.getPid(), timeout);
    }

    private boolean checkDaemonRunning() {
        boolean result = isDaemonRunning();
        if (!result) {
            listener.error("Daemon not running");
        }
        return result;
    }

    @Nullable
    public PeerInfo id(@NonNull PID pid, int timeout) {
        checkNotNull(pid);
        checkArgument(timeout > 0);
        try {
            if (checkDaemonRunning()) {
                String json = node.idWithTimeout(pid.getPid(), timeout);
                Map map = gson.fromJson(json, Map.class);
                if (map != null) {
                    return PeerInfo.create(map);
                }
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }

        return null;

    }

    @NonNull
    public PID getPeerID() {
        checkNotNull(pid);
        return pid;
    }

    @NonNull
    public PeerInfo id() {
        try {
            String json = node.id();
            Map map = gson.fromJson(json, Map.class);
            checkNotNull(map);
            return PeerInfo.create(map);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }


    public boolean checkRelay(@NonNull Peer peer, int timeout) {
        checkNotNull(peer);
        checkArgument(timeout > 0);
        AtomicBoolean success = new AtomicBoolean(false);
        try {
            PID pid = PID.create("QmWATWQ7fVPP2EFGu71UkfnqhYXDYH566qy47CnJDgvs8u"); // DUMMY

            String address = relayAddress(peer, pid);
            node.swarmConnect(address, timeout);

        } catch (Throwable e) {
            String line = e.getLocalizedMessage();
            if (line != null) {
                if (line.contains("HOP_NO_CONN_TO_DST")) {
                    success.set(true);
                }
            }
        }
        return success.get();
    }


    public void pubsubPub(@NonNull final String topic, @NonNull final String message, int timeout) {
        checkNotNull(topic);
        checkNotNull(message);
        checkArgument(timeout > 0);
        if (checkDaemonRunning()) {

            try {
                byte[] data = Base64.encodeBase64(message.getBytes());
                node.pubsubPub(topic, data);
                Thread.sleep(timeout);
            } catch (Throwable e) {
                listener.error("" + e.getLocalizedMessage());
            }

        }
    }

    public void logs() throws Exception {
        node.logs();
    }

    public void pubsubSub(@NonNull PubsubReader reader) throws Exception {
        checkNotNull(reader);

        if (checkDaemonRunning()) {
            String topic = reader.getTopic();
            checkNotNull(topic);
            pubsubReader.put(topic, reader);

            node.pubsubSub(topic, false);
        }

    }

    @NonNull
    public String relayAddress(@NonNull Peer relay, @NonNull PID pid) {
        checkNotNull(relay);
        checkNotNull(pid);
        return relayAddress(relay.getMultiAddress(), relay.getPid(), pid);
    }

    @NonNull
    public String relayAddress(@NonNull String address, @NonNull PID relay, @NonNull PID pid) {
        checkNotNull(address);
        checkNotNull(relay);
        checkNotNull(pid);
        return address + "/" + Style.p2p.name() + "/" + relay.getPid() +
                "/" + P2P_CIRCUIT + "/" + Style.p2p.name() + "/" + pid.getPid();
    }

    public boolean specificRelay(@NonNull PID relay,
                                 @NonNull PID pid,
                                 int timeout) {
        checkNotNull(relay);
        checkNotNull(pid);
        checkArgument(timeout > 0);
        boolean result = false;

        if (swarmConnect(relay, timeout)) {
            Peer peer = swarmPeer(relay);
            if (peer != null) {
                return swarmConnect(relayAddress(peer, pid), timeout);
            }
        }

        return result;
    }

    public boolean relay(@NonNull PID relay, @NonNull PID pid, int timeout) {
        checkNotNull(relay);
        checkNotNull(pid);
        checkArgument(timeout > 0);
        return specificRelay(relay, pid, timeout);

    }


    public boolean blockStat(@NonNull CID cid, boolean offline) {
        checkNotNull(cid);
        try {
            if (checkDaemonRunning()) {
                String res = node.blockStat(cid.getCid(), offline);
                if (res != null && !res.isEmpty()) {
                    return true;
                }
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return false;
    }


    public void blockRm(@NonNull CID cid) {
        checkNotNull(cid);
        try {
            if (checkDaemonRunning()) {
                node.blockRm(cid.getCid());
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
    }

    public boolean arbitraryRelay(@NonNull PID pid, int timeout) {
        checkNotNull(pid);
        checkArgument(timeout > 0);
        return swarmConnect("/" + P2P_CIRCUIT + "/" +
                Style.p2p.name() + "/" + pid.getPid(), timeout);
    }

    public boolean swarmConnect(@NonNull PID pid, int timeout) {
        checkNotNull(pid);
        checkArgument(timeout > 0);
        return swarmConnect("/" + Style.p2p.name() + "/" +
                pid.getPid(), timeout);
    }


    public boolean swarmConnect(@NonNull Peer peer, int timeout) {
        checkNotNull(peer);
        checkArgument(timeout > 0);
        String ma = peer.getMultiAddress() + "/" + Style.p2p.name() + "/" + peer.getPid();
        return swarmConnect(ma, timeout);

    }


    public boolean swarmConnect(@NonNull String multiAddress, int timeout) {
        checkNotNull(multiAddress);
        checkArgument(timeout > 0);
        try {
            if (checkDaemonRunning()) {
                return node.swarmConnect(multiAddress, timeout);

            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return false;
    }

    public boolean unProtectPeer(@NonNull PID pid, @NonNull String tag) {
        checkNotNull(pid);
        checkNotNull(tag);
        try {
            return node.unProtectPeer(pid.getPid(), tag);
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return false;
    }

    public void protectPeer(@NonNull PID pid, @NonNull String tag) {
        checkNotNull(pid);
        checkNotNull(tag);
        try {
            node.protectPeer(pid.getPid(), tag);
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
    }


    public boolean isConnected(@NonNull PID pid) {
        checkNotNull(pid);
        try {
            return node.isConnected(pid.getPid());
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return false;
    }

    @Nullable
    public Peer swarmPeer(@NonNull PID pid) {
        checkNotNull(pid);

        try {
            String json = node.swarmPeer(pid.getPid());
            if (json != null && !json.isEmpty()) {
                Map map = gson.fromJson(json, Map.class);
                if (map != null) {
                    return Peer.create(map);
                }
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return null;
    }

    @NonNull
    public List<Peer> swarmPeers() {
        List<Peer> peers = swarm_peers();
        peers.sort(Peer::compareTo);
        return peers;
    }


    @NonNull
    private List<Peer> swarm_peers() {

        List<Peer> peers = new ArrayList<>();
        try {
            if (checkDaemonRunning()) {

                String json = node.swarmPeers();
                if (json != null && !json.isEmpty()) {
                    Map map = gson.fromJson(json, Map.class);
                    if (map != null) {

                        Object object = map.get("Peers");
                        if (object instanceof List) {
                            List list = (List) object;
                            if (!list.isEmpty()) {
                                for (Object entry : list) {
                                    if (entry instanceof Map) {
                                        Map peer = (Map) entry;
                                        peers.add(Peer.create(peer));
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (Throwable e) {
            listener.debug("" + e.getLocalizedMessage());
        }
        return peers;
    }

    public void logBaseDir() {
        try {
            File[] files = baseDir.listFiles();
            if (files != null) {
                for (File file : files) {
                    Log.e(TAG, "" + file.length() + " " + file.getAbsolutePath());
                    if (file.isDirectory()) {
                        File[] children = file.listFiles();
                        if (children != null) {
                            for (File child : children) {
                                Log.e(TAG, "" + child.length() + " " + child.getAbsolutePath());
                            }
                        }
                    }
                }
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
    }

    public void cleanCacheDir() {
        try {
            File[] files = getCacheDir().listFiles();
            if (files != null) {
                for (File file : files) {
                    checkArgument(file.delete());
                }
            }
        } catch (Throwable e) {
           listener.error("" + e.getLocalizedMessage());
        }
    }

    public boolean swarmDisconnect(@NonNull PID pid) {
        checkNotNull(pid);
        if (checkDaemonRunning()) {
            try {
                return node.swarmDisconnect("/" + Style.p2p.name() + "/" + pid.getPid());
            } catch (Throwable e) {
                listener.error("" + e.getLocalizedMessage());
            }
        }
        return false;
    }


    public void rm(@NonNull CID cid) {
        checkNotNull(cid);
        List<LinkInfo> links = ls(cid, 10, true);

        if (links != null) {
            for (LinkInfo link : links) {
                blockRm(link.getCid());
            }
        }
        blockRm(cid);
    }


    @NonNull
    public List<String> pubsubPeers() {
        List<String> peers = new ArrayList<>();
        try {
            if (checkDaemonRunning()) {

                node.pubsubPeers(new PubsubPeer() {
                    @Override
                    public void peer(String peer) {
                        peers.add(peer);
                    }
                });

            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return peers;
    }

    @NonNull
    public List<PID> pubsubPeers(@NonNull String topic) {
        checkNotNull(topic);
        List<PID> pidList = new ArrayList<>();

        try {
            if (checkDaemonRunning()) {
                node.pubsubPeers(new PubsubPeer() {
                    @Override
                    public void peer(String peer) {
                        pidList.add(PID.create(peer));
                    }
                });
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return pidList;
    }


    @NonNull
    public Config getConfig() throws Exception {
        String result = getConfigAsString();
        Gson gson = new Gson();
        return gson.fromJson(result, Config.class);
    }

    public void saveConfig(@NonNull Config config) throws Exception {
        checkNotNull(config);

        File file = getConfigFile();

        try (FileWriter writer = new FileWriter(file)) {
            Gson gson = new GsonBuilder().setPrettyPrinting().create();
            gson.toJson(config, writer);
        }

    }


    private void configTune(@NonNull Config config,
                            @Nullable AddressesConfig addresses,
                            @Nullable ExperimentalConfig experimental,
                            @Nullable PubsubConfig pubsub,
                            @Nullable DiscoveryConfig discovery,
                            @Nullable SwarmConfig swarm,
                            @Nullable RoutingConfig routing,
                            @Nullable ReproviderConfig reprovider) throws Exception {
        checkNotNull(config);
        checkArgument(existConfigFile(), "Config file does not exist.");


        boolean changedConfig = false;
        if (experimental != null) {
            changedConfig = true;
            config.setExperimental(experimental);
        }
        if (pubsub != null) {
            changedConfig = true;
            config.setPubsub(pubsub);
        }
        if (addresses != null) {
            changedConfig = true;
            config.setAddresses(addresses);
        }
        if (discovery != null) {
            changedConfig = true;
            config.setDiscovery(discovery);
        }
        if (swarm != null) {
            changedConfig = true;
            config.setSwarm(swarm);
        }
        if (routing != null) {
            changedConfig = true;
            config.setRouting(routing);
        }
        if (reprovider != null) {
            changedConfig = true;
            config.setReprovider(reprovider);
        }
        if (changedConfig) {
            saveConfig(config);
        }
    }


    public void daemon(@Nullable PubsubReader reader, boolean pubsub) throws Exception {

        AtomicBoolean failure = new AtomicBoolean(false);
        ExecutorService executor = Executors.newSingleThreadExecutor();
        AtomicReference<String> exception = new AtomicReference<>("");
        executor.submit(() -> {
            try {
                String topic = pid.getPid();
                if (reader != null) {
                    topic = reader.getTopic();
                    checkNotNull(topic);
                    pubsubReader.put(topic, reader);
                }

                node.daemon(topic, pubsub);
            } catch (Throwable e) {
                failure.set(true);
                exception.set("" + e.getLocalizedMessage());
                listener.error(exception.get());
            }
        });
        long time = 0L;
        while (!node.getRunning() && time <= TIMEOUT && !failure.get()) {
            time = time + 100L;
            Thread.sleep(100);
        }
        if (failure.get()) {
            throw new RuntimeException(exception.get());
        }
    }


    @Nullable
    public CID storeData(@NonNull byte[] data) throws Exception {

        checkNotNull(data);
        return storeData(data, true);
    }

    @Nullable
    public CID storeText(@NonNull String text, @NonNull String key) throws Exception {
        checkNotNull(text);
        checkNotNull(key);
        return storeText(text, key, true);
    }


    @Nullable
    private CID storeStream(@NonNull InputStream inputStream) throws Exception {
        checkNotNull(inputStream);

        return storeStream(inputStream, true);
    }

    @Nullable
    public CID storeData(@NonNull byte[] data, boolean offline) throws Exception {
        checkNotNull(data);
        checkArgument(data.length > 0);
        try (InputStream inputStream = new ByteArrayInputStream(data)) {
            return storeStream(inputStream, offline);
        }
    }

    @Nullable
    public CID storeText(@NonNull String content, @NonNull String key, boolean offline) throws Exception {
        checkNotNull(key);
        checkNotNull(content);
        checkArgument(!content.isEmpty());
        try (InputStream inputStream = new ByteArrayInputStream(content.getBytes())) {
            return addFile(inputStream, key, offline);
        }
    }


    @Nullable
    public List<LinkInfo> ls(@NonNull CID cid, int timeout, boolean offline) {
        checkNotNull(cid);
        checkArgument(timeout > 0);
        checkArgument(checkDaemonRunning());
        List<LinkInfo> infos = new ArrayList<>();
        try {
            node.ls(cid.getCid(), (json) -> {

                try {
                    Map map = gson.fromJson(json, Map.class);
                    checkNotNull(map);
                    LinkInfo info = LinkInfo.create(map);
                    infos.add(info);
                } catch (Throwable e) {
                    listener.error("" + e.getLocalizedMessage());
                }

            }, timeout, offline);


        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
            return null;
        }
        return infos;
    }


    @Nullable
    public CID addFile(@NonNull File target, boolean offline) {
        checkNotNull(target);
        if (!checkDaemonRunning()) {
            throw new RuntimeException("Daemon not running");
        }


        String res = "";
        try {
            res = node.add(target.getAbsolutePath(), offline);
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }

        if (!res.isEmpty()) {
            return CID.create(res);
        }
        return null;
    }

    @NonNull
    public FileReader getFileReader(@NonNull CID cid, boolean offline) {
        return node.getFileReader(cid.getCid(), offline);
    }

    private boolean stream(@NonNull OutputStream outputStream,
                           @NonNull Progress progress,
                           @NonNull CID cid,
                           @NonNull String key,
                           boolean offline,
                           int timeout,
                           long size) {
        checkNotNull(outputStream);
        checkNotNull(progress);
        checkNotNull(cid);
        checkNotNull(key);
        checkArgument(timeout > 0);
        final AtomicInteger atomicProgress = new AtomicInteger(0);
        int totalRead = 0;
        try (InputStream inputStream = stream(cid, key, timeout, offline)) {

            byte[] data = new byte[BLOCK_SIZE];
            progress.setProgress(0);
            int bytesRead = 1;


            // Read data with timeout
            ExecutorService executor = Executors.newSingleThreadExecutor();
            Callable<Integer> readTask = () ->
                    inputStream.read(data);
            while (bytesRead >= 0) {
                Future<Integer> future = executor.submit(readTask);
                bytesRead = future.get(timeout, TimeUnit.SECONDS);
                if (bytesRead >= 0) {
                    outputStream.write(data, 0, bytesRead);
                    totalRead += bytesRead;
                    if (size > 0) {
                        int percent = (int) ((totalRead * 100.0f) / size);
                        if (atomicProgress.getAndSet(percent) < percent) {
                            progress.setProgress(percent);
                        }
                    }
                }

            }

        } catch (Throwable e) {
            progress.isStopped();
            return false;
        }
        if (size > 0) {
            return !progress.isStopped() && (totalRead == size);
        }
        return !progress.isStopped();
    }


    public boolean storeToFile(@NonNull File file,
                               @NonNull CID cid,
                               @NonNull Progress progress,
                               boolean offline, int timeout,
                               long size) {
        checkNotNull(file);
        checkNotNull(cid);
        checkNotNull(progress);
        checkArgument(timeout > 0);

        // make sure file path exists
        try {
            if (!file.exists()) {
                File parent = file.getParentFile();
                checkNotNull(parent);
                if (!parent.exists()) {
                    checkArgument(parent.mkdirs());
                }
                checkArgument(file.createNewFile());
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
            return false;
        }

        try (FileOutputStream outputStream = new FileOutputStream(file)) {
            return stream(outputStream, progress, cid, "", offline, timeout, size);
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
            return false;
        }
    }

    @NonNull
    private InputStream stream(@NonNull CID cid, int timeout, boolean offline) throws Exception {
        checkArgument(timeout > 0);
        PipedOutputStream pos = new PipedOutputStream();
        PipedInputStream pis = new PipedInputStream(pos);

        List<LinkInfo> info = ls(cid, timeout, offline);
        if (info == null) {
            throw new TimeoutException("timeout " + timeout + " [s]");
        }

        final AtomicBoolean close = new AtomicBoolean(false);
        new Thread(() -> {
            try {

                node.getStream(cid.getCid(), new Stream() {
                    @Override
                    public boolean close() {
                        return close.get();
                    }

                    @Override
                    public void size(long l) {
                        listener.debug("CID " + cid + " size : " + l);
                    }

                    @Override
                    public void write(byte[] bytes) {
                        try {
                            pos.write(bytes);
                        } catch (Throwable e) {
                            close.set(true);
                            // Ignore exception might be on pipe is closed
                            listener.debug("CID " + cid + " error : " + e.getLocalizedMessage());
                        }
                    }

                }, offline);

            } catch (Throwable e) {
                listener.debug("CID " + cid + " error : " + e.getLocalizedMessage());
            } finally {
                try {
                    pos.close();
                } catch (Throwable e) {
                    // Ignore exception might be on pipe is closed
                    listener.debug("CID " + cid + " error : " + e.getLocalizedMessage());
                }
            }
        }).start();


        return pis;
    }


    public boolean storeToFile(@NonNull File file,
                               @NonNull CID cid,
                               boolean offline, int timeout,
                               long size) {
        checkNotNull(file);
        checkNotNull(cid);
        checkArgument(timeout > 0);
        if (!checkDaemonRunning()) {
            throw new RuntimeException("Daemon not running");
        }

        // make sure file path exists
        try {
            if (!file.exists()) {
                File parent = file.getParentFile();
                checkNotNull(parent);
                if (!parent.exists()) {
                    checkArgument(parent.mkdirs());
                }
                checkArgument(file.createNewFile());
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
            return false;
        }

        try (FileOutputStream fileOutputStream = new FileOutputStream(file)) {
            return stream(fileOutputStream, new Progress() {
                @Override
                public void setProgress(int percent) {

                }

                @Override
                public boolean isStopped() {
                    return false;
                }
            }, cid, "", offline, timeout, size);
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage());
            return false;
        }

    }


    @Nullable
    public ContentInfo getContentInfo(@NonNull CID cid,
                                      @NonNull String key,
                                      int timeout,
                                      boolean offline) {
        checkNotNull(cid);
        checkNotNull(key);
        checkArgument(timeout > 0);
        try {
            try (InputStream inputStream = new BufferedInputStream(
                    stream(cid, key, timeout, offline))) {

                return util.findMatch(inputStream);

            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return null;
    }

    @Nullable
    public ContentInfo getContentInfo(@NonNull File file) {
        checkNotNull(file);

        try {
            return util.findMatch(file);
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }
        return null;
    }

    @NonNull
    public File getCacheDir() {
        File cacheDir = new File(this.cacheDir, CACHE);
        if (!cacheDir.exists()) {
            checkArgument(cacheDir.mkdir());
        }
        return cacheDir;
    }

    @NonNull
    public File getTempCacheFile() throws IOException {
        return File.createTempFile("temp", ".cid", getCacheDir());
    }


    @Nullable
    public CID storeStream(@NonNull InputStream inputStream, boolean offline) throws Exception {
        checkNotNull(inputStream);

        if (!checkDaemonRunning()) {
            throw new RuntimeException("Daemon not running");
        }
        File target = getTempCacheFile();
        try (OutputStream outputStream = new FileOutputStream(target)) {

            IOUtils.copy(inputStream, outputStream);

            return addFile(target, offline);
        } finally {
            if (target.exists()) {
                target.deleteOnExit();
            }
        }
    }


    @Nullable
    private CID addFile(@NonNull InputStream inputStream, @NonNull String key,
                        boolean offline) throws Exception {
        checkNotNull(inputStream);
        checkNotNull(key);

        if (!checkDaemonRunning()) {
            throw new RuntimeException("Daemon not running");
        }
        File target = getTempCacheFile();
        try (OutputStream outputStream = new FileOutputStream(target)) {
            if (!key.isEmpty()) {
                Key aesKey = Encryption.getKey(key);
                Cipher cipher = Cipher.getInstance("AES");
                cipher.init(Cipher.ENCRYPT_MODE, aesKey);
                CipherInputStream cipherStream = new CipherInputStream(inputStream, cipher);
                IOUtils.copy(cipherStream, outputStream);
            } else {
                IOUtils.copy(inputStream, outputStream);
            }
            return addFile(target, offline);
        } finally {
            if (target.exists()) {
                target.deleteOnExit();
            }
        }
    }

    @Nullable
    public String getText(@NonNull CID cid, @NonNull String key, int timeout, boolean offline) {
        checkNotNull(cid);
        checkNotNull(key);
        checkArgument(timeout > 0);

        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            boolean success = stream(outputStream, new Progress() {
                @Override
                public void setProgress(int percent) {

                }

                @Override
                public boolean isStopped() {
                    return false;
                }
            }, cid, key, offline, timeout, -1);
            if (success) {
                return new String(outputStream.toByteArray());
            } else {
                return null;
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
            return null;
        }
    }

    @Nullable
    public byte[] getData(@NonNull CID cid, int timeout, boolean offline) {
        checkNotNull(cid);
        checkArgument(timeout > 0);

        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            boolean success = stream(outputStream, new Progress() {
                @Override
                public void setProgress(int percent) {

                }

                @Override
                public boolean isStopped() {
                    return false;
                }
            }, cid, "", offline, timeout, -1);
            if (success) {
                return outputStream.toByteArray();
            } else {
                return null;
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
            return null;
        }

    }


    @Nullable
    public CID storeUri(@NonNull Context context, @NonNull Uri uri) throws Exception {
        checkNotNull(context);
        checkNotNull(uri);

        InputStream inputStream = context.getContentResolver().openInputStream(uri);
        checkNotNull(inputStream);

        return storeStream(inputStream, true);

    }

    @NonNull
    public InputStream getStream(@NonNull CID cid, int timeout, boolean offline) throws Exception {
        checkNotNull(cid);
        checkArgument(timeout > 0);
        return stream(cid, "", timeout, offline);
    }

    @NonNull
    private InputStream stream(@NonNull CID cid, @NonNull String key, int timeout,
                               boolean offline) throws Exception {
        checkNotNull(cid);
        checkArgument(timeout > 0);
        if (!checkDaemonRunning()) {
            throw new RuntimeException("Daemon not running");
        }


        if (key.isEmpty()) {
            return stream(cid, timeout, offline);
        } else {
            Key aesKey = Encryption.getKey(key);
            Cipher cipher = Cipher.getInstance("AES");
            cipher.init(Cipher.DECRYPT_MODE, aesKey);
            return new CipherInputStream(stream(cid, timeout, offline), cipher);
        }
    }


    @NonNull
    public String version() throws Exception {
        String json = node.version();
        Map map = gson.fromJson(json, Map.class);
        checkNotNull(map);
        String version = (String) map.get("Version");
        checkNotNull(version);
        return version;
    }

    public boolean swarmDisconnect(@NonNull Peer peer) {
        checkNotNull(peer);
        return swarmDisconnect(peer.getPid());
    }

    public void gc() {
        if (checkDaemonRunning()) {
            try {
                node.repoGC();
            } catch (Throwable e) {
                listener.error("" + e.getLocalizedMessage());
            }
        }
    }


    public boolean isDaemonRunning() {
        return node.getRunning();
    }


    @Override
    public void error(String message) {
        if (message != null && !message.isEmpty()) {
            listener.error(message);
        }
    }

    @Override
    public void info(String message) {
        if (message != null && !message.isEmpty()) {
            listener.info(message);
        }
    }

    @Override
    public void log(String message) {
        if (message != null && !message.isEmpty()) {
            try {
                if (message.startsWith("{")) {
                    Map map = gson.fromJson(message, Map.class);
                    if (map != null) {
                        Object logs = map.get("Logs");
                        if (logs instanceof List) {
                            List list = (List) map.get("Logs");
                            checkNotNull(list);
                            for (Object entry : list) {

                                if (entry instanceof Map) {
                                    Map logsMap = (Map) entry;
                                    checkNotNull(logsMap);

                                    Object fields = logsMap.get("Fields");
                                    if (fields instanceof List) {
                                        List fieldList = (List) logsMap.get("Fields");
                                        checkNotNull(fieldList);

                                        for (Object fieldEntry : fieldList) {
                                            if (fieldEntry instanceof Map) {
                                                Map fieldMap = (Map) fieldEntry;
                                                checkNotNull(fieldMap);
                                                Object key = fieldMap.get("Key");
                                                if (key instanceof String) {

                                                    if (key.equals("error")) {
                                                        listener.error("Logs : " + entry.toString());
                                                    } else if (key.equals("debug")) {
                                                        listener.debug("Logs : " + entry.toString());
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                } else {
                    listener.debug(message);
                }
            } catch (Throwable e) {
                listener.debug("" + message);
            }
        }
    }

    @Override
    public void pubsub(final String message, final byte[] data) {

        try {
            PubsubInfo pubsub = PubsubInfo.create(message, data);

            PubsubReader reader = pubsubReader.get(pubsub.getTopic());
            if (reader != null) {
                listener.debug(pubsub.toString());
                reader.receive(pubsub);
            }
        } catch (Throwable e) {
            listener.error("" + e.getLocalizedMessage());
        }

    }

    @Override
    public void verbose(String s) {
        Log.i(TAG, "" + s);
    }

    public enum Style {
        ipfs, ipns, p2p
    }

    public interface Progress {
        /**
         * Setter for percent
         *
         * @param percent Value between 0-100 percent
         */
        void setProgress(int percent);

        boolean isStopped();
    }

    private static class Builder {

        private Context context;
        private ApiListener listener = new LogListener();
        private AddressesConfig addresses = null;
        private ExperimentalConfig experimental = null;
        private PubsubConfig pubsub = null;
        private DiscoveryConfig discovery = null;
        private SwarmConfig swarm = null;
        private RoutingConfig routing = null;
        private ReproviderConfig reprovider = null;

        IPFS build() throws Exception {
            checkNotNull(context);
            checkNotNull(listener);
            return new IPFS(this);
        }


        Builder context(@NonNull Context context) {
            checkNotNull(context);
            this.context = context;
            return this;
        }

        Builder listener(@NonNull ApiListener listener) {
            checkNotNull(listener);
            this.listener = listener;
            return this;
        }


        Builder experimental(@Nullable ExperimentalConfig experimental) {
            this.experimental = experimental;
            return this;
        }

        Builder addresses(@Nullable AddressesConfig addresses) {
            this.addresses = addresses;
            return this;
        }

        Builder pubsub(@Nullable PubsubConfig pubsub) {
            this.pubsub = pubsub;
            return this;
        }

        Builder discovery(@Nullable DiscoveryConfig discovery) {
            this.discovery = discovery;
            return this;
        }

        Builder swarm(@Nullable SwarmConfig swarm) {
            this.swarm = swarm;
            return this;
        }

        Builder routing(@Nullable RoutingConfig routing) {
            this.routing = routing;
            return this;
        }

        Builder reprovider(@Nullable ReproviderConfig reprovider) {
            this.reprovider = reprovider;
            return this;
        }
    }
}
