package threads.core;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import threads.core.api.Peer;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.PeerInfo;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class GatewayService {

    public static final String TAG = GatewayService.class.getSimpleName();


    public static int evaluateAllPeers(@NonNull Context context) {
        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        THREADS threads = Singleton.getInstance(context).getThreads();

        checkNotNull(ipfs);
        threads.resetPeersConnected();
        List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();
        int size = peers.size();
        for (threads.ipfs.api.Peer peer : peers) {
            // do not store circuit addresses
            if (!peer.getMultiAddress().endsWith("/p2p-circuit")) {
                storePeer(context, peer);
            }
        }

        if (Network.isConnectedMinHighBandwidth(context)) {
            List<Peer> stored = threads.getPeers();
            for (Peer peer : stored) {
                if (!peer.isConnected() &&
                        !peer.isRelay() &&
                        !peer.isAutonat() &&
                        !peer.isPubsub()) {
                    threads.removePeer(ipfs, peer);
                }
            }
        }
        return size;
    }

    public static int evaluatePeers(@NonNull Context context, boolean pubsubs) {
        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        checkNotNull(ipfs);
        List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();
        int size = peers.size();
        for (threads.ipfs.api.Peer peer : peers) {
            // do not store circuit addresses
            if (!peer.getMultiAddress().endsWith("/p2p-circuit")) {
                if (pubsubs) {
                    if (peer.isAutonat() || peer.isRelay() || peer.isMeshSub() || peer.isFloodSub()) {
                        storePeer(context, peer);
                    }
                } else {
                    if (peer.isAutonat() || peer.isRelay()) {
                        storePeer(context, peer);
                    }
                }
            }
        }
        return size;
    }


    public synchronized static List<threads.core.api.Peer> getRelayPeers(
            @NonNull Context context, @NonNull String tag, int numRelays, int timeout) {

        checkNotNull(context);
        checkArgument(numRelays >= 0);
        checkArgument(timeout > 0);

        List<threads.core.api.Peer> result = new ArrayList<>();


        if (!Network.isConnected(context)) {
            return result;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();


        if (ipfs != null) {

            List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();

            peers.sort(threads.ipfs.api.Peer::compareTo);

            for (threads.ipfs.api.Peer peer : peers) {

                if (result.size() == numRelays) {
                    break;
                }

                if (peer.isRelay()) {

                    if (ipfs.isConnected(peer.getPid())) {

                        if (!tag.isEmpty()) {
                            ipfs.protectPeer(peer.getPid(), tag);
                        }

                        result.add(storePeer(context, peer));


                    } else if (ipfs.swarmConnect(peer, timeout)) {

                        if (!tag.isEmpty()) {
                            ipfs.protectPeer(peer.getPid(), tag);
                        }

                        result.add(storePeer(context, peer));

                    }

                }
            }

        }

        return result;
    }

    public static void fillPeerList(
            @NonNull Context context,
            @NonNull List<threads.core.api.Peer> list,
            @NonNull String tag, int numPeers) {

        checkNotNull(context);
        checkNotNull(list);
        checkNotNull(tag);

        if (!Network.isConnected(context)) {
            return;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();


        if (ipfs != null) {

            List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();

            peers.sort(threads.ipfs.api.Peer::compareTo);

            for (threads.ipfs.api.Peer peer : peers) {

                if (list.size() == numPeers) {
                    break;
                }


                if (ipfs.isConnected(peer.getPid())) {

                    if (!tag.isEmpty()) {
                        ipfs.protectPeer(peer.getPid(), tag);
                    }

                    list.add(storePeer(context, peer));


                }

            }
        }

    }

    private static Peer storePeer(@NonNull Context context,
                                  @NonNull threads.ipfs.api.Peer peer) {
        checkNotNull(context);
        checkNotNull(peer);

        // the given peer is connected (so rating will be dependent of peer
        int rating = 0;
        try {
            double latency = peer.getLatency();
            if (latency < 1000) {
                rating = (int) (1000 - latency);
            }

        } catch (Throwable e) {
            // ignore any exceptions here
        }

        // now add higher rating when peer has specific attributes
        boolean isConnected = false;
        try {
            int timeout = 5;
            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            if (ipfs != null) {
                PeerInfo info = ipfs.id(peer, timeout);
                if (info != null) {

                    String protocol = info.getProtocolVersion();
                    String agent = info.getAgentVersion();

                    if (protocol != null && protocol.equals("ipfs/0.1.0")) {
                        rating = rating + 100;
                    } else {
                        rating = rating - 100;
                    }
                    if (agent != null) {
                        if (agent.startsWith("go-ipfs/0.4.2")) {
                            rating = rating + 100;
                        } else if (agent.startsWith("go-ipfs/0.5")) {
                            rating = rating + 150;
                        }
                    }
                }
                isConnected = ipfs.isConnected(peer.getPid());
            }
        } catch (Throwable e) {
            // ignore any exceptions here
        }
        if (rating < 0) {
            rating = 0;
        }
        boolean isPubsub = peer.isFloodSub() || peer.isMeshSub();

        return storePeer(context, peer.getPid(),
                peer.getMultiAddress(), peer.isRelay(), peer.isAutonat(),
                isPubsub, isConnected, rating);
    }


    @NonNull
    private static threads.core.api.Peer storePeer(@NonNull Context context,
                                                   @NonNull PID pid,
                                                   @NonNull String multiAddress,
                                                   boolean isRelay,
                                                   boolean isAutonat,
                                                   boolean isPubsub,
                                                   boolean isConnected,
                                                   int rating) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.Peer peer = threads.getPeerByPID(pid);
        if (peer != null) {
            peer.setMultiAddress(multiAddress);
            peer.setRating(rating);
            peer.setConnected(isConnected);
            threads.updatePeer(peer);
        } else {
            peer = threads.createPeer(pid, multiAddress);
            peer.setRelay(isRelay);
            peer.setAutonat(isAutonat);
            peer.setPubsub(isPubsub);
            peer.setRating(rating);
            peer.setConnected(isConnected);
            threads.storePeer(peer);
        }
        return peer;
    }

    public static void connectStoredAutonat(@NonNull Context context,
                                            int numConnections,
                                            int timeout) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 0);

        if (!Network.isConnected(context)) {
            return;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getAutonatPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer autonat : peers) {

                if (counter.get() == numConnections) {
                    break;
                }

                if (ipfs.isConnected(PID.create(autonat.getPid()))) {
                    counter.incrementAndGet();
                    threads.setTimestamp(autonat, System.currentTimeMillis());
                } else {

                    String ma = autonat.getMultiAddress() + "/" +
                            IPFS.Style.p2p.name() + "/" + autonat.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {
                        counter.incrementAndGet();
                        threads.setTimestamp(autonat, System.currentTimeMillis());

                    } else {
                        if (Network.isConnectedMinHighBandwidth(context)) {
                            if (lifeTimeExpired(autonat)) {
                                threads.removePeer(ipfs, autonat);
                            }
                        }
                    }
                }
            }
        }
    }

    private static boolean lifeTimeExpired(@NonNull Peer peer) {
        return System.currentTimeMillis() >
                peer.getTimestamp() + (TimeUnit.HOURS.toMillis(24));
    }


    public static void connectStoredPubsub(@NonNull Context context,
                                           int numConnections,
                                           int timeout) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 0);

        if (!Network.isConnected(context)) {
            return;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getPubsubPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer pubsub : peers) {

                if (counter.get() == numConnections) {
                    break;
                }

                if (ipfs.isConnected(PID.create(pubsub.getPid()))) {
                    counter.incrementAndGet();
                    threads.setTimestamp(pubsub, System.currentTimeMillis());
                } else {

                    String ma = pubsub.getMultiAddress() + "/" +
                            IPFS.Style.p2p.name() + "/" + pubsub.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {
                        counter.incrementAndGet();
                        threads.setTimestamp(pubsub, System.currentTimeMillis());
                    } else {
                        if (Network.isConnectedMinHighBandwidth(context)) {

                            if (lifeTimeExpired(pubsub)) {
                                threads.removePeer(ipfs, pubsub);
                            }
                        }
                    }
                }
            }
        }
    }

    public static void connectStoredRelays(@NonNull Context context,
                                           int numConnections,
                                           int timeout) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 0);

        if (!Network.isConnected(context)) {
            return;
        }


        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getRelayPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer relay : peers) {

                if (counter.get() == numConnections) {
                    break;
                }

                if (ipfs.isConnected(PID.create(relay.getPid()))) {
                    counter.incrementAndGet();
                    threads.setTimestamp(relay, System.currentTimeMillis());
                } else {

                    String ma = relay.getMultiAddress() + "/" +
                            IPFS.Style.p2p.name() + "/" + relay.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {
                        counter.incrementAndGet();
                        threads.setTimestamp(relay, System.currentTimeMillis());
                    } else {

                        if (Network.isConnectedMinHighBandwidth(context)) {
                            if (lifeTimeExpired(relay)) {
                                threads.removePeer(ipfs, relay);
                            }
                        }
                    }
                }
            }
        }
    }


    public static void connectStoredPeers(@NonNull Context context, int timeout) {
        checkNotNull(context);
        checkArgument(timeout > 0);

        if (!Network.isConnected(context)) {
            return;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer peer : peers) {


                Log.e(TAG, "Stored Peer : " + peer.toString());

                if (!ipfs.isConnected(PID.create(peer.getPid()))) {

                    String ma = peer.getMultiAddress() + "/" +
                            IPFS.Style.p2p.name() + "/" + peer.getPid();

                    if (!ipfs.swarmConnect(ma, timeout)) {
                        if (Network.isConnectedMinHighBandwidth(context)) {
                            threads.removePeer(ipfs, peer);
                        }
                    }
                }
            }
        }
    }
}
