package threads.core;

import android.content.ContentResolver;
import android.content.Context;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.graphics.pdf.PdfRenderer;
import android.media.MediaMetadataRetriever;
import android.net.Uri;
import android.os.ParcelFileDescriptor;
import android.provider.DocumentsContract;
import android.util.Log;
import android.webkit.MimeTypeMap;

import androidx.annotation.DrawableRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.amulyakhare.textdrawable.TextDrawable;
import com.amulyakhare.textdrawable.util.ColorGenerator;
import com.j256.simplemagic.ContentInfo;

import org.iota.jota.utils.Constants;
import org.iota.jota.utils.TrytesConverter;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Optional;

import threads.core.api.Converter;
import threads.core.api.EventsDatabase;
import threads.core.api.LinkType;
import threads.core.api.Note;
import threads.core.api.NoteType;
import threads.core.api.Thread;
import threads.core.api.ThreadsAPI;
import threads.core.api.ThreadsDatabase;
import threads.iota.EntityService;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.CID;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class THREADS extends ThreadsAPI {
    public static final String TAG = THREADS.class.getSimpleName();
    // Is an indicator for a snapshot
    // when the number increases, previously
    // defined values can be deleted
    public static final long SNAPSHOT = 1L;
    private static final int THUMBNAIL_SIZE = 128;


    private THREADS(final THREADS.Builder builder) {
        super(builder.threadsDatabase, builder.eventsDatabase, builder.entityService);
    }

    @NonNull
    public static Bitmap getUserImage(@NonNull String name) {
        checkNotNull(name);
        Canvas canvas = new Canvas();
        String letter = name.substring(0, 1);
        int color = ColorGenerator.MATERIAL.getColor(name);
        TextDrawable drawable = TextDrawable.builder()
                .buildRound(letter, color);
        Bitmap bitmap = Bitmap.createBitmap(64, 64, Bitmap.Config.ARGB_8888);
        canvas.setBitmap(bitmap);
        drawable.setBounds(0, 0, 64, 64);

        drawable.draw(canvas);
        return bitmap;
    }

    public static Optional<String> getExtension(String filename) {
        return Optional.ofNullable(filename)
                .filter(f -> f.contains("."))
                .map(f -> f.substring(filename.lastIndexOf(".") + 1));
    }

    @NonNull
    public static String getAddress(@NonNull CID cid) {
        checkNotNull(cid);
        String address = TrytesConverter.asciiToTrytes(cid.getCid());
        return IOTA.addChecksum(address.substring(0, Constants.ADDRESS_LENGTH_WITHOUT_CHECKSUM));
    }


    @NonNull
    public static THREADS createThreads(@NonNull ThreadsDatabase threadsDatabase,
                                        @NonNull EventsDatabase eventsDatabase,
                                        @NonNull EntityService entityService) {
        checkNotNull(threadsDatabase);
        checkNotNull(eventsDatabase);
        checkNotNull(entityService);
        return new THREADS.Builder()
                .threadsDatabase(threadsDatabase)
                .eventsDatabase(eventsDatabase)
                .entityService(entityService)
                .build();
    }

    @Nullable
    public static LinkType getLinkType(@NonNull Note note) {
        checkNotNull(note);

        if (note.getNoteType() == NoteType.LINK) {
            String linkType = note.getAdditional(LinkType.class.getSimpleName());
            return LinkType.valueOf(linkType);
        }
        return null;
    }

    public static double getLatitude(@NonNull Note note) {
        checkNotNull(note);

        if (note.getNoteType() == NoteType.LOCATION) {
            return Double.valueOf(note.getAdditional(Preferences.LATITUDE));
        }
        return Double.NaN;
    }

    public static double getLongitude(@NonNull Note note) {
        checkNotNull(note);

        if (note.getNoteType() == NoteType.LOCATION) {
            return Double.valueOf(note.getAdditional(Preferences.LONGITUDE));
        }
        return Double.NaN;
    }

    public static double getZoom(@NonNull Note note) {
        checkNotNull(note);

        if (note.getNoteType() == NoteType.LOCATION) {
            return Double.valueOf(note.getAdditional(Preferences.ZOOM));
        }
        return Double.NaN;
    }

    @NonNull
    public static Date getTomorrow() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.DATE, 1);
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        return c.getTime();
    }

    @NonNull
    public static Date getYesterday() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.DATE, -1);
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        return c.getTime();
    }

    @NonNull
    public static byte[] getImage(@NonNull Drawable drawable) {
        Canvas canvas = new Canvas();
        Bitmap bitmap = Bitmap.createBitmap(64, 64, Bitmap.Config.ARGB_8888);
        canvas.setBitmap(bitmap);
        drawable.setBounds(0, 0, 64, 64);

        drawable.draw(canvas);
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        bitmap.compress(Bitmap.CompressFormat.PNG, 100, stream);
        return stream.toByteArray();
    }

    @NonNull
    public static byte[] getImage(@NonNull Context context, @NonNull String name, @DrawableRes int id) {
        checkNotNull(context);
        checkNotNull(name);
        Drawable drawable = context.getDrawable(id);
        checkNotNull(drawable);
        int color = ColorGenerator.MATERIAL.getColor(name);
        drawable.setColorFilter(color, PorterDuff.Mode.SRC_ATOP);
        return THREADS.getImage(drawable);
    }

    @NonNull
    public static byte[] getImage(@NonNull Context context, @DrawableRes int id) {
        checkNotNull(context);
        Drawable drawable = context.getDrawable(id);
        checkNotNull(drawable);
        return getImage(drawable);
    }

    @NonNull
    public static Bitmap getThumbnail(@NonNull Context context, @NonNull Uri uri) throws IOException {
        checkNotNull(context);
        checkNotNull(uri);

        InputStream input = context.getContentResolver().openInputStream(uri);
        BitmapFactory.Options onlyBoundsOptions = new BitmapFactory.Options();
        onlyBoundsOptions.inJustDecodeBounds = true;
        onlyBoundsOptions.inPreferredConfig = Bitmap.Config.ARGB_8888;//optional
        BitmapFactory.decodeStream(input, null, onlyBoundsOptions);
        input.close();
        if ((onlyBoundsOptions.outWidth == -1) || (onlyBoundsOptions.outHeight == -1))
            return null;

        int originalSize = (onlyBoundsOptions.outHeight > onlyBoundsOptions.outWidth) ?
                onlyBoundsOptions.outHeight : onlyBoundsOptions.outWidth;

        double ratio = (originalSize > THUMBNAIL_SIZE) ? (originalSize / THUMBNAIL_SIZE) : 1.0;

        int k = Integer.highestOneBit((int) Math.floor(ratio));
        int sampleSize = k;
        if (k == 0) {
            sampleSize = 1;
        }


        BitmapFactory.Options bitmapOptions = new BitmapFactory.Options();
        bitmapOptions.inSampleSize = sampleSize;
        bitmapOptions.inPreferredConfig = Bitmap.Config.ARGB_8888;//optional
        input = context.getContentResolver().openInputStream(uri);
        Bitmap bitmap = BitmapFactory.decodeStream(input, null, bitmapOptions);
        input.close();
        return bitmap;
    }

    @Nullable
    public static byte[] getPreviewImage(@NonNull Context context, @NonNull File file,
                                         @Nullable IPFS ipfs) throws Exception {
        checkNotNull(context);
        checkNotNull(file);
        Bitmap bitmap = getPreview(context, file, ipfs);
        if (bitmap != null) {
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            bitmap.compress(Bitmap.CompressFormat.WEBP, 100, stream);
            byte[] image = stream.toByteArray();
            bitmap.recycle();
            return image;
        }
        return null;
    }

    @Nullable
    public static byte[] getPreviewImage(@NonNull Context context,
                                         @NonNull File file,
                                         @NonNull String filename) throws Exception {
        checkNotNull(context);
        checkNotNull(file);
        checkNotNull(filename);

        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            Bitmap bitmap = null;

            if (!filename.isEmpty()) {
                Optional<String> result = getExtension(filename);
                if (result.isPresent()) {
                    String mimeType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(
                            result.get());
                    if (mimeType != null) {
                        bitmap = getPreview(context, file, mimeType);
                    }
                }
            }

            if (bitmap == null) {
                ContentInfo contentInfo = ipfs.getContentInfo(file);
                if (contentInfo != null) {
                    String mimeType = contentInfo.getMimeType();
                    if (mimeType != null) {
                        bitmap = getPreview(context, file, mimeType);
                    }
                }
            }


            if (bitmap != null) {
                ByteArrayOutputStream stream = new ByteArrayOutputStream();
                bitmap.compress(Bitmap.CompressFormat.WEBP, 100, stream);
                byte[] image = stream.toByteArray();
                bitmap.recycle();
                return image;
            }
        }
        return null;
    }

    @Nullable
    public static byte[] getPreviewImage(@NonNull Context context, @NonNull Uri uri) throws Exception {
        checkNotNull(context);
        checkNotNull(uri);
        Bitmap bitmap = getPreview(context, uri);
        if (bitmap != null) {
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            bitmap.compress(Bitmap.CompressFormat.WEBP, 100, stream);
            byte[] image = stream.toByteArray();
            bitmap.recycle();
            return image;
        }
        return null;
    }

    @Nullable
    public static Bitmap getPreview(@NonNull Context context, @NonNull File file,
                                    @Nullable IPFS ipfs) throws Exception {
        checkNotNull(context);
        checkNotNull(file);

        Optional<String> result = getExtension(file.getName());
        if (result.isPresent()) {
            String mimeType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(result.get());
            if (mimeType != null) {
                return getPreview(context, file, mimeType);
            }
        }


        if (ipfs != null) {
            ContentInfo contentInfo = ipfs.getContentInfo(file);
            if (contentInfo != null) {
                String mimeType = contentInfo.getMimeType();
                if (mimeType != null) {
                    return getPreview(context, file, mimeType);
                }
            }
        }

        return null;

    }

    @Nullable
    private static Bitmap getPreview(@NonNull Context context,
                                     @NonNull File file,
                                     @NonNull String mimeType) throws Exception {
        checkNotNull(context);
        checkNotNull(file);
        checkNotNull(mimeType);


        if (mimeType.startsWith("video")) {

            MediaMetadataRetriever mediaMetadataRetriever = new MediaMetadataRetriever();
            mediaMetadataRetriever.setDataSource(context, Uri.fromFile(file));

            // mediaMetadataRetriever.getPrimaryImage(); // TODO support in the future
            Bitmap bitmap;
            try {
                String time = mediaMetadataRetriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
                Long timeUs = Long.decode(time);
                long timeFrame = (timeUs / 100) * 5;

                bitmap = mediaMetadataRetriever.getFrameAtTime(timeFrame);
            } catch (Throwable e) {
                bitmap = mediaMetadataRetriever.getFrameAtTime();
            }
            mediaMetadataRetriever.release();
            return bitmap;
        } else if (mimeType.startsWith("application/pdf")) {
            return getPDFBitmap(context, Uri.fromFile(file));
        } else if (mimeType.startsWith("image")) {
            return getThumbnail(context, Uri.fromFile(file));

        }

        return null;

    }

    @Nullable
    public static Bitmap getPreview(@NonNull Context context, @NonNull Uri uri) throws Exception {
        checkNotNull(context);
        checkNotNull(uri);

        FileDetails fileDetails = getFileDetails(context, uri);
        checkNotNull(fileDetails);
        String mimeType = fileDetails.getMimeType();

        if (mimeType.startsWith("video")) {

            MediaMetadataRetriever mediaMetadataRetriever = new MediaMetadataRetriever();
            mediaMetadataRetriever.setDataSource(context, uri);
            // mediaMetadataRetriever.getPrimaryImage(); // TODO support in the future
            Bitmap bitmap;
            try {
                String time = mediaMetadataRetriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
                Long timeUs = Long.decode(time);
                long timeFrame = (timeUs / 100) * 5;

                bitmap = mediaMetadataRetriever.getFrameAtTime(timeFrame);
            } catch (Throwable e) {
                bitmap = mediaMetadataRetriever.getFrameAtTime();
            }
            mediaMetadataRetriever.release();
            return bitmap;
        } else if (mimeType.startsWith("application/pdf")) {
            return getPDFBitmap(context, uri);
        } else if (mimeType.startsWith("image")) {
            return getThumbnail(context, uri);

        }

        return null;

    }


    @NonNull
    public static Bitmap getPDFBitmap(@NonNull Context context, @NonNull Uri uri) throws Exception {
        checkNotNull(context);
        checkNotNull(uri);

        ParcelFileDescriptor fileDescriptor = context.getContentResolver().openFileDescriptor(
                uri, "r");
        checkNotNull(fileDescriptor);
        PdfRenderer pdfRenderer = new PdfRenderer(fileDescriptor);

        PdfRenderer.Page rendererPage = pdfRenderer.openPage(0);
        int rendererPageWidth = rendererPage.getWidth();
        int rendererPageHeight = rendererPage.getHeight();


        Bitmap bitmap = Bitmap.createBitmap(
                rendererPageWidth,
                rendererPageHeight,
                Bitmap.Config.ARGB_8888);
        rendererPage.render(bitmap, null, null,
                PdfRenderer.Page.RENDER_MODE_FOR_DISPLAY);


        rendererPage.close();

        pdfRenderer.close();
        fileDescriptor.close();
        return bitmap;


    }

    public static byte[] getImage(@NonNull String displayName) {
        java.lang.String letter = displayName.substring(0, 1);
        int color = ColorGenerator.MATERIAL.getColor(displayName);
        Canvas canvas = new Canvas();
        TextDrawable drawable = TextDrawable.builder()
                .buildRound(letter, color);
        Bitmap bitmap = Bitmap.createBitmap(64, 64, Bitmap.Config.ARGB_8888);
        canvas.setBitmap(bitmap);
        drawable.setBounds(0, 0, 64, 64);

        drawable.draw(canvas);
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        bitmap.compress(Bitmap.CompressFormat.PNG, 100, stream);
        return stream.toByteArray();
    }

    @NonNull
    public static String getDate(@NonNull Date date) {
        Calendar c = Calendar.getInstance();
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        Date today = c.getTime();
        c.set(Calendar.MONTH, 0);
        c.set(Calendar.DAY_OF_MONTH, 0);
        Date lastYear = c.getTime();

        if (date.before(today)) {
            if (date.before(lastYear)) {
                return android.text.format.DateFormat.format("dd.MM.yyyy", date).toString();
            } else {
                return android.text.format.DateFormat.format("dd.MMMM", date).toString();
            }
        } else {
            return android.text.format.DateFormat.format("HH:mm", date).toString();
        }
    }

    @NonNull
    public static Calendar getUserAgeOffsetCalendar(int offset) {
        Date now = new Date();
        Calendar init = new GregorianCalendar();
        init.setTime(now);
        int mYear = init.get(Calendar.YEAR);
        int mMonth = init.get(Calendar.MONTH);
        int mDay = init.get(Calendar.DAY_OF_MONTH);

        return new GregorianCalendar(mYear - offset, mMonth, mDay);
    }

    static long getOffsetDate(int offsetInYears) {
        return getUserAgeOffsetCalendar(offsetInYears).getTime().getTime();
    }

    @NonNull
    public static Date getOffsetYearDate(int offsetInYears) {
        return getUserAgeOffsetCalendar(offsetInYears).getTime();
    }

    public static long getTodayDate() {
        return getToday().getTime();
    }

    @NonNull
    public static Date getToday() {
        Calendar c = Calendar.getInstance();
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        return c.getTime();
    }

    @Nullable
    public static CID createNameImage(@NonNull THREADS threads,
                                      @NonNull IPFS ipfs,
                                      @NonNull String name,
                                      @NonNull String key) throws Exception {
        checkNotNull(name);
        checkNotNull(threads);
        checkNotNull(ipfs);
        checkNotNull(key);
        byte[] data = getImage(name);
        return threads.storeData(ipfs, data, key);
    }

    @Nullable
    public static CID createResourceImage(@NonNull Context context,
                                          @NonNull THREADS threads,
                                          @NonNull IPFS ipfs,
                                          @DrawableRes int id,
                                          @NonNull String key) throws Exception {
        checkNotNull(context);
        checkNotNull(threads);
        checkNotNull(ipfs);
        checkNotNull(key);
        byte[] data = getImage(context, id);
        return threads.storeData(ipfs, data, key);
    }

    @Nullable
    public static FileDetails getFileDetails(@NonNull Context context,
                                             @NonNull Uri uri) {
        checkNotNull(context);
        checkNotNull(uri);

        String mimeType = context.getContentResolver().getType(uri);
        if (mimeType == null) {
            mimeType = Preferences.OCTET_MIME_TYPE;
        }
        ContentResolver contentResolver = context.getContentResolver();
        try (Cursor cursor = contentResolver.query(uri, new String[]{
                DocumentsContract.Document.COLUMN_DISPLAY_NAME,
                DocumentsContract.Document.COLUMN_SIZE}, null, null, null)) {

            checkNotNull(cursor);
            cursor.moveToFirst();

            String fileName = cursor.getString(0);
            long fileSize = cursor.getLong(1);
            return FileDetails.create(fileName, mimeType, fileSize);
        } catch (Throwable e) {
            // ignore exception
        }

        return null;
    }

    public void updateThreadInfo(@NonNull Thread thread,
                                 @NonNull String mimeType,
                                 @NonNull String message,
                                 boolean increase) {
        checkNotNull(thread);
        checkNotNull(mimeType);
        checkNotNull(message);
        setAdditional(thread,
                Preferences.THREAD_MIME_TYPE, mimeType, true);
        setAdditional(thread,
                Preferences.THREAD_CONTENT, message, true);
        setAdditional(thread, Preferences.THREAD_DATE,
                "" + Converter.toLong(new Date()), true);
        if (increase) {
            incrementUnreadNotesNumber(thread);
        } else {
            refreshNotesNumber(thread);
        }
    }

    @NonNull
    public InputStream stream(@NonNull IPFS ipfs, @NonNull CID cid, @NonNull String key)
            throws Exception {
        checkNotNull(ipfs);
        checkNotNull(cid);
        checkNotNull(key);
        return ipfs.stream(cid, key, -1, true);
    }

    public boolean receive(@NonNull IPFS ipfs, @NonNull CID cid, @NonNull String key, int timeout) {
        checkNotNull(ipfs);
        checkNotNull(cid);
        checkNotNull(key);


        byte[] bytes = ipfs.get(cid, key, timeout, false);

        if (bytes.length > 0) {
            try {
                pin_add(ipfs, cid, true);
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }
            return true;
        }
        return false;
    }

    /**
     * @param ipfs     IPFS client
     * @param cid      CID object
     * @param progress Progress object
     * @param timeout  timeout when 0, there is not timeout
     * @param size     Size of the CID objects, when size  equal or less 0 it is not known
     * @return true when successful
     */
    @Nullable
    public File receive(@NonNull IPFS ipfs, @NonNull CID cid, @NonNull String key,
                        @NonNull IPFS.Progress progress, int timeout, long size) {
        checkNotNull(ipfs);
        checkNotNull(cid);
        checkNotNull(key);
        checkNotNull(progress);

        return ipfs.get(cid, key, progress, false, timeout, size, true);
    }


    @Nullable
    public Bitmap getImage(@NonNull IPFS ipfs, @NonNull Thread thread) throws Exception {
        checkNotNull(ipfs);
        checkNotNull(thread);
        CID image = thread.getImage();
        String key = thread.getSesKey();
        if (image != null) {
            return getImage(ipfs, image, key);
        }
        return null;
    }

    @Nullable
    public Bitmap getImage(@NonNull IPFS ipfs, @NonNull CID image, @NonNull String key) {
        checkNotNull(ipfs);
        checkNotNull(image);
        checkNotNull(key);

        byte[] bytes = ipfs.get(image, key, -1, true);
        return BitmapFactory.decodeByteArray(bytes, 0, bytes.length);

    }

    public boolean download(@NonNull IPFS ipfs,
                            @NonNull File file,
                            @NonNull CID cid,
                            @NonNull String key,
                            @NonNull IPFS.Progress progress,
                            boolean pin,
                            boolean offline,
                            int timeout,
                            long size) {
        checkNotNull(ipfs);
        checkNotNull(cid);
        checkNotNull(progress);
        checkNotNull(key);


        ipfs.store(file, cid, key, progress, offline, timeout, size, pin);

        return file.exists();

    }

    @Nullable
    public CID store(@NonNull Context context,
                     @NonNull IPFS ipfs,
                     @NonNull Uri uri,
                     @NonNull String key) throws Exception {
        checkNotNull(context);
        checkNotNull(ipfs);
        checkNotNull(uri);
        checkNotNull(key);

        InputStream inputStream = context.getContentResolver().openInputStream(uri);
        checkNotNull(inputStream);

        return storeStream(ipfs, key, inputStream);

    }

    public static class FileDetails {

        @NonNull
        private final String fileName;
        @NonNull
        private final String mimeType;
        private final long fileSize;

        private FileDetails(@NonNull String fileName, @NonNull String mimeType, long fileSize) {
            checkNotNull(fileName);
            checkNotNull(mimeType);
            checkArgument(fileSize > 0);
            this.fileName = fileName;
            this.mimeType = mimeType;
            this.fileSize = fileSize;
        }

        public static FileDetails create(@NonNull String fileName,
                                         @NonNull String mimeType,
                                         long fileSize) {
            return new FileDetails(fileName, mimeType, fileSize);
        }

        @NonNull
        public String getFileName() {
            return fileName;
        }

        @NonNull
        public String getMimeType() {
            return mimeType;
        }

        public long getFileSize() {
            return fileSize;
        }

    }

    public static class Builder {
        EventsDatabase eventsDatabase = null;
        ThreadsDatabase threadsDatabase = null;
        EntityService entityService = null;

        public THREADS build() {
            checkNotNull(threadsDatabase);
            checkNotNull(eventsDatabase);
            checkNotNull(entityService);
            return new THREADS(this);
        }

        public Builder threadsDatabase(@NonNull ThreadsDatabase threadsDatabase) {
            checkNotNull(threadsDatabase);
            this.threadsDatabase = threadsDatabase;
            return this;
        }

        public Builder eventsDatabase(@NonNull EventsDatabase eventsDatabase) {
            checkNotNull(eventsDatabase);
            this.eventsDatabase = eventsDatabase;
            return this;
        }

        public Builder entityService(@NonNull EntityService entityService) {
            checkNotNull(entityService);
            this.entityService = entityService;
            return this;
        }

    }
}
