/******************************************************************************
 * Copyright (C) 2016 Sebastiaan R. Hogenbirk                                 *
 * *
 * This program is free software: you can redistribute it and/or modify       *
 * it under the terms of the GNU General Public License as published by       *
 * the Free Software Foundation, either version 3 of the License, or          *
 * (at your option) any later version.                                        *
 * *
 * This program is distributed in the hope that it will be useful,            *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 * GNU General Public License for more details.                               *
 * *
 * You should have received a copy of the GNU General Public License          *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.      *
 ******************************************************************************/

package thorwin.math.accelerator;

import java.io.*;
import java.util.Arrays;
import java.util.Optional;

/**
 * Access to native CPUID instruction.
 */
public final class CpuID {

    private static final boolean AVAILABLE;

    static {
        boolean available = true;
        try {
            String extension = ".so";
            if (System.getProperty("os.name").contains("Mac")) {
                extension = ".jnilib";
            }
            byte[] buffer = new byte[1024];
            File file = File.createTempFile("lib", extension);
            file.deleteOnExit();
            try (InputStream in = Blas.class.getResourceAsStream("CpuID" + extension);
                 FileOutputStream out = new FileOutputStream(file)) {
                int len = in.read(buffer);
                while (len > 0) {
                    out.write(buffer, 0, len);
                    len = in.read(buffer);
                }
            }
            System.load(file.getAbsolutePath());
        } catch (IOException e) {
            available = false;
        }
        AVAILABLE = available;
    }

    private final String vendor;
    private final int type;
    private final int family;
    private final int model;
    private final int steppingID;
    private final int brandIndex;

    public CpuID(String vendor, int type, int family, int model, int steppingID, int brandIndex) {
        this.vendor = vendor;
        this.type = type;
        this.family = family;
        this.model = model;
        this.steppingID = steppingID;
        this.brandIndex = brandIndex;
    }

    /**
     * Returns the CPU identification for the running x86 CPU, or
     * empty if this is not a x86 CPU.
     *
     * @return CPU identification.
     */
    public static Optional<CpuID> getCpuID() {
        try {
            if (!AVAILABLE) return Optional.empty();

            int[] registers = cpuid(0);

            byte[] vendorBytes = {
                    (byte) ((registers[1] >> 0) & 0xff),
                    (byte) ((registers[1] >> 8) & 0xff),
                    (byte) ((registers[1] >> 16) & 0xff),
                    (byte) ((registers[1] >> 24) & 0xff),
                    (byte) ((registers[3] >> 0) & 0xff),
                    (byte) ((registers[3] >> 8) & 0xff),
                    (byte) ((registers[3] >> 16) & 0xff),
                    (byte) ((registers[3] >> 24) & 0xff),
                    (byte) ((registers[2] >> 0) & 0xff),
                    (byte) ((registers[2] >> 8) & 0xff),
                    (byte) ((registers[2] >> 16) & 0xff),
                    (byte) ((registers[2] >> 24) & 0xff)
            };

            String vendor = new String(vendorBytes, "US-ASCII").trim();

            registers = cpuid(1);

            int eax = registers[0];

            int steppingID = eax & 0xf;
            int model = (eax >> 4) & 0xf;
            int family = (eax >> 8) & 0xf;
            int type = (eax >> 12) & 0b11;
            int extModel = (eax >> 16) & 0xf;
            int extFamily = (eax >> 20) & 0xff;
            int brandIndex = registers[1] & 0xff;

            CpuID cpuID = new CpuID(vendor, type, family + extFamily, model + (extModel << 4), steppingID, brandIndex);

            return Optional.of(cpuID);
        } catch (UnsupportedEncodingException e) {
            // US-ASCII should be supported by Java specification
            throw new IllegalStateException(e);
        }
    }

    /**
     * Calls the CPUID instruction
     *
     * @param level level
     * @return array {eax, ebx, ecx, edx}
     */
    private static native int[] cpuid(int level);

    @Override
    public String toString() {
        return "CpuID{" +
                "vendor='" + vendor + '\'' +
                ", type=" + type +
                ", family=" + family +
                ", model=" + model +
                ", steppingID=" + steppingID +
                ", brandIndex=" + brandIndex +
                '}';
    }

    public String getVendor() {
        return vendor;
    }

    public int getType() {
        return type;
    }

    public int getFamily() {
        return family;
    }

    public int getModel() {
        return model;
    }

    public int getSteppingID() {
        return steppingID;
    }

    public int getBrandIndex() {
        return brandIndex;
    }
}
