package thevoid.iam.components.widget.ext

import android.text.Editable
import android.text.TextWatcher
import android.widget.EditText
import androidx.annotation.StringRes
import iam.thevoid.e.format
import io.reactivex.Flowable
import org.jetbrains.anko.hintTextColor
import org.jetbrains.anko.textColor
import org.jetbrains.anko.textColorResource
import thevoid.iam.components.R
import thevoid.iam.components.rx.fields.*
import thevoid.iam.components.widget.delegate.TextWatcherDelegate
import thevoid.iam.components.widget.util.adapter.TextWatcherAdapter

private val EditText.textWatcher: TextWatcherDelegate
    get() = ((getTag(R.id.textWatcher) as? TextWatcherDelegate)
        ?: TextWatcherDelegate().also { setTag(R.id.textWatcher, it) }).also {
        removeTextChangedListener(it)
        addTextChangedListener(it)
    }


fun EditText.afterTextChanges(rxEditable: RxField<Editable>) = afterTextChanges(rxEditable) { it }

fun <T : Any> EditText.afterTextChanges(rxEditable: RxField<T>, mapper : (Editable) -> T) =
    addGetter({
        textWatcher.addAfterTextChangedCallback(object : TextWatcherAdapter() {
            override fun afterTextChanged(s: Editable?) {
                s?.apply { it.invoke(mapper(this)) }
            }
        })
    }, rxEditable)

fun EditText.afterTextChanges(rxEditable: RxString) =
    addGetter({
        textWatcher.addAfterTextChangedCallback(object : TextWatcherAdapter() {
            override fun afterTextChanged(s: Editable?) {
                s?.apply { it.invoke("$this") }
            }
        })
    }, rxEditable)

fun EditText.beforeTextChanges(rxChanges: RxField<BeforeEditTextChanges>) = beforeTextChanges(rxChanges) { it }

fun <T : Any> EditText.beforeTextChanges(rxChanges: RxField<T>, mapper: (BeforeEditTextChanges) -> T) =
    addGetter({
        textWatcher.addBeforeTextChangedCallback(object : TextWatcherAdapter() {
            override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
                it.invoke(mapper(BeforeEditTextChanges(s, start, count, after)))
            }
        })
    }, rxChanges)

fun EditText.onTextChanges(rxChanges: RxField<OnEditTextChanges>) = onTextChanges(rxChanges) { it }

fun <T : Any> EditText.onTextChanges(rxChanges: RxField<T>, mapper: (OnEditTextChanges) -> T) =
    addGetter({
        textWatcher.addOnTextChangedCallback(object : TextWatcherAdapter() {
            override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
                it.invoke(mapper(OnEditTextChanges(s, start, before, count)))
            }
        })
    }, rxChanges)

fun EditText.setTextSilent(text: CharSequence) {
    removeTextChangedListener(textWatcher)
    setText(text)
    addTextChangedListener(textWatcher)
}

fun EditText.setTextResourceSilent(@StringRes text: Int) {
    removeTextChangedListener(textWatcher)
    setText(text)
    addTextChangedListener(textWatcher)
}

fun <T : CharSequence> EditText.setText(rxString: RxCharSequence<T>) =
    addSetter(rxString.observe()) { setTextSilent(it) }

fun <T : CharSequence> EditText.setText(textFlowable: Flowable<T>) =
    addSetter(textFlowable) { setTextSilent(it) }

fun EditText.setText(rxInt: RxInt) =
    addSetter(rxInt.observe()) { setTextSilent("$it") }

fun EditText.setText(rxLong: RxLong) =
    addSetter(rxLong.observe()) { setTextSilent("$it") }

fun EditText.setText(rxFloat: RxFloat, precision: Int? = null) =
    addSetter(rxFloat.observe()) { setTextSilent(it.format(precision)) }

fun EditText.setText(rxDouble: RxDouble, precision: Int? = null) =
    addSetter(rxDouble.observe()) { setTextSilent(it.format(precision)) }

fun EditText.setTextResource(rxIntResource: RxInt) =
    addSetter(rxIntResource.observe()) { setTextResourceSilent(it) }

fun EditText.setTextResource(textResourceFlowable: Flowable<Int>) =
    addSetter(textResourceFlowable) { setTextResourceSilent(it) }

fun EditText.setTextColor(colorFlowable: Flowable<Int>) =
    addSetter(colorFlowable) { textColor = it }

fun EditText.setHintColor(colorFlowable: Flowable<Int>) =
    addSetter(colorFlowable) { hintTextColor = it }

fun EditText.setTextColorResourse(colorResFlowable: Flowable<Int>) =
    addSetter(colorResFlowable) { textColorResource = it }

data class BeforeEditTextChanges(val s: CharSequence?, val start: Int, val count: Int, val after: Int)

data class OnEditTextChanges(val s: CharSequence?, val start: Int, val before: Int, val count: Int)