package tenqube.parser.core;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteException;
import android.support.annotation.NonNull;
import android.support.v4.util.ArrayMap;
import android.text.TextUtils;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import tenqube.parser.constants.Constants;
import tenqube.parser.model.ParsingRule;
import tenqube.parser.model.RegData;
import tenqube.parser.model.SMS;
import tenqube.parser.model.Sender;
import tenqube.parser.model.Transaction;

import static tenqube.parser.constants.Constants.ALL;
import static tenqube.parser.constants.Constants.NO;
import static tenqube.parser.constants.Constants.SAMSUNG_PAY;
import static tenqube.parser.constants.Constants.YES;
import static tenqube.parser.core.ParserReaderContract.TransactionsTable.INSERT_TRANSACTION;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.core.Utils.getSMSType;
import static tenqube.parser.core.Utils.isSpendSMS;
import static tenqube.parser.core.Utils.makeCardKey;
import static tenqube.parser.core.Utils.makeRegKey;
import static tenqube.parser.core.Utils.makeSpentMoneyDwTypeKey;
import static tenqube.parser.core.Utils.transformRepSender;
import static tenqube.parser.core.Utils.transformSender;
import static tenqube.parser.util.LogUtil.LOGE;
import static tenqube.parser.util.LogUtil.LOGI;
import static tenqube.parser.util.LogUtil.makeLogTag;


class ParserHelper extends BaseQueryHelper {

    private static final String TAG = makeLogTag(ParserHelper.class);

    private ParserMapper mMapper;
    private RegHandler mRegHandler;
    private Map<String, ArrayList<RegData>> mRegMap = new HashMap<>();
    private Map<String, String> mSendersMap =  new HashMap<>();
    private Map<String, ArrayList<TransactionTableData>> mDuplMap =  new HashMap<>();
    private Map<String, ArrayList<TransactionTableData>> mMovingAssetMap =  new HashMap<>();
    private Map<String, ArrayList<TransactionTableData>> mOffsetMap =  new HashMap<>();

    private Map<String, TransactionTableData> mTransactionMap =  new HashMap<>();
    private Map<String, CardTableData> mCardMap =  new HashMap<>();


    ParserHelper(Context context) throws SQLException  {
        super(context);
        mMapper = new ParserMapper();
        mRegHandler = new RegHandler(context);
        LOGI(TAG, "ParsingHelper constructor", mIsDebug);
    }

    void setParsedTransactions() {
        LOGI(TAG, "setParsedTransactions", mIsDebug);

        String maxSpentDate = getMaxSpentDate();

        if(TextUtils.isEmpty(maxSpentDate))
            return;

        if(mTransactionMap == null) mTransactionMap = new ArrayMap<>();
        if(mCardMap == null) mCardMap = new ArrayMap<>();
        Cursor c = null;
        String query =
                SELECT + "*" +
                FROM + getJoinTable() +
                WHERE+  ParserReaderContract.TransactionsTable.ALIAS + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE + "> '"+ maxSpentDate+"' "+
                ORDER_BY + ParserReaderContract.TransactionsTable.ALIAS+ ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE + ASC;

        try {
            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {
                    while (!c.isAfterLast()) {

                        TransactionTableData tran = ParserReaderContract.TransactionsTable.populateModel(c);
                        tran.cardTableData = ParserReaderContract.CardTable.populateModel(c);
                        mCardMap.put(makeCardKey(tran.cardTableData), tran.cardTableData);
                        mTransactionMap.put(tran.identifier, tran);
                        c.moveToNext();
                    }
                }
            }
        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            if (c != null)
                c.close();
        }

    }

    private String getMaxSpentDate() {
        LOGI(TAG, "getMaxSpentDate", mIsDebug);

        Cursor c = null;
        String query =
                SELECT + " MAX(" + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE +") " +
                FROM + ParserReaderContract.TransactionsTable.TABLE_NAME;

        try {
            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {
                   return c.getString(0);
                }
            }
        } catch (SQLException e) {
            LOGE(TAG, e.toString());
        } finally {
            if (c != null)
                c.close();
        }

        return "";
    }

    /**************************************      BULK         *************************************************/

    /**
     * is_success = 0 인 리스트 함께 보냄
     * 0. sender 필터
     * 1. 정규표현식을 통한 데이터 추출
     * 2. 중복 데이터 확인
     * 3. 중복 x 상쇄 Tran 확인
     * 4. 중복 x 상쇄 x 자산이동 Tran 확인
     * 5. 잔액 변경

     * @param sms 수신된 문자
     * @return 파싱된 Tran List
     */
    ArrayList<Transaction> parseWithBulk(SMS sms) {

        TransactionTableData transaction;
        TransactionTableData offsetTran = null;
        TransactionTableData movingAssetTran = null;

        if(!isSpendSMS(sms.fullSms))
            return null;
        //0. sender 필터
        String repSender = getRepSenderWithBulk(sms.sender);
        if(repSender == null) {
            repSender = transformRepSender(sms.fullSms);
        }

        LOGI(TAG, "0. sender 대표 필터" + repSender, mIsDebug);
        if (repSender == null)
            return null;

        if(isExistTranWithBulk(sms.smsId, sms.smsType)) {
            LOGI(TAG, "isExistTranWithBulk: 이미 존재", mIsDebug);
            return null;
        }

        //1. 정규표현식을 통한 데이터 확인
        transaction = getBulkTransactionWithReg(sms, repSender);
        if (transaction == null) {
            LOGI(TAG, "1. 정규표현식을 통한 데이터 확인: FALSE", mIsDebug);
            LOGI(TAG, "NOT PARSED: "+sms.toString(), mIsDebug);
            return null;
        }

        LOGI(TAG, "1. 정규표현식을 통한 데이터 확인: TRUE" + transaction.toString(), mIsDebug);
        //2. 중복 데이터 확인
        LOGI(TAG, "2. 중복 데이터 확인", mIsDebug);
        TransactionTableData duplicationTran = getDuplicationTranWithBulk(transaction);

        //중복
        if (duplicationTran != null) {

            LOGI(TAG, "2. 중복 데이터 확인: TRUE" + duplicationTran.toString(), mIsDebug);
            //중복인 경우 우선순위에 따라 정보 변경
            transaction = mRegHandler.combineTran(duplicationTran, transaction);

        } else {
            //중복 아님
            LOGI(TAG, "2. 중복 데이터 확인: FALSE", mIsDebug);

            //상쇄 확인
            offsetTran = getOffsetTranWithBulk(transaction);
            LOGI(TAG, "3. 중복 x 상쇄 Tran 확인", mIsDebug);

            //상쇄
            if (offsetTran != null) {

                LOGI(TAG, "3. 중복 x -> 상쇄 Tran 확인: TRUE" + offsetTran.toString(), mIsDebug);
                transaction = mRegHandler.combineTranWithOffset(offsetTran, transaction);


            } else {
                //상쇄아님
                LOGI(TAG, "3. 중복 x -> 상쇄 Tran 확인: FALSE", mIsDebug);

                //자산이동 확인
                LOGI(TAG, "4. 중복 x -> 상쇄 x -> 자산이동 Tran 확인", mIsDebug);

                movingAssetTran = getMovingAssetTranWithBulk(transaction);

                //자산이동
                if (movingAssetTran != null) {
                    LOGI(TAG, "4. 중복 x 상쇄 x 자산이동 Tran 확인: TRUE" + movingAssetTran.toString(), mIsDebug);

                    transaction = mRegHandler.combineTranWithMovingAsset(transaction);
                    movingAssetTran = mRegHandler.combineTranWithMovingAsset(movingAssetTran);

                } else {

                    LOGI(TAG, "4. 중복 x 상쇄 x 자산이동 Tran 확인: FALSE", mIsDebug);

                }
            }
        }
        //잔액 업데이트
        if(transaction != null) transaction = mRegHandler.updateBalance(transaction);

        //목록 넣기
        ArrayList<Transaction> resultTrans = getResultTransactionsWithBulk(transaction, offsetTran, movingAssetTran);

        if(resultTrans.size() == 0)
            return null;

        LOGI(TAG, "최종 리턴 사이즈:" + resultTrans.size(), mIsDebug);
        return resultTrans;
    }

    private ArrayList<Transaction> getResultTransactionsWithBulk(TransactionTableData transaction,
                                                                 TransactionTableData offsetTran,
                                                                 TransactionTableData movingAssetTran) {

        LOGI(TAG, "getResultTransactionsWithBulk", mIsDebug);

        ArrayList<Transaction> resultTrans = new ArrayList<>();

        try {
            ArrayList<TransactionTableData> transactions = new ArrayList<>();
            if(transaction != null)
            transactions.add(transaction);
            if(offsetTran != null)
                transactions.add(offsetTran);
            if(movingAssetTran != null)
                transactions.add(movingAssetTran);

            //리턴 맵핑

            for (TransactionTableData tran : transactions) {
                if(mTransactionMap!=null)
                    mTransactionMap.put(tran.identifier, tran);

                if(mCardMap!=null)
                    mCardMap.put(makeCardKey(tran.cardTableData), tran.cardTableData);//최종 잔액 업데이트

                Transaction resultTran = mMapper.transform(transaction);
                if(resultTran!=null) {
                    resultTrans.add(resultTran);
                }
            }
        }catch (Exception e) {
            e.printStackTrace();
        }

        return resultTrans;
    }


    private String getRepSenderWithBulk(String sender) {

        if(TextUtils.isEmpty(sender))
            return null;

        LOGI(TAG, "getRepSender", mIsDebug);
        return mSendersMap==null?
                getRepSender(sender)
                :
                mSendersMap.get(sender);

    }

    private String getRepSender(String sender) {

        LOGI(TAG, "getRepSender", mIsDebug);

        String transformSender = transformSender(sender);


        Cursor c = null;
        String query =
                SELECT + ParserReaderContract.SendersTable.COLUMN_REP_SENDER +
                FROM + ParserReaderContract.SendersTable.TABLE_NAME +
                WHERE + ParserReaderContract.SendersTable.COLUMN_SENDER + IN + "('"+sender+"','"+transformSender+"')" +
                LIMIT + "1";

        LOGI(TAG, "getRepSender query:" + query, mIsDebug);

        try {

            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {
                    return c.getString(c.getColumnIndex(ParserReaderContract.SendersTable.COLUMN_REP_SENDER));
                }
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            if (c != null)
                c.close();
        }

        return null;

    }

    /**
     * 정규표현식을 통한 Transaction 추출
     *
     * 1. getRegDatas map.get(sender) == null? where sender and smsType
     * 2. matching start
     * 3. matched -> TransactionTableData , cardTableData, 데이터 셋팅
     *
     * @param sms 수신된 문자 or 알림
     * @return 파싱된 Transaction
     */
    private TransactionTableData getBulkTransactionWithReg(SMS sms, String repSender){
        LOGI(TAG, "getBulkTransactionWithReg start" + sms.toString(), mIsDebug);

        TransactionTableData parsedTran;

        String newFullSms  = Utils.transformFullSMS(sms.fullSms);
        LOGI(TAG, "getBulkTransactionWithReg newFullSms" + newFullSms, mIsDebug);

        ArrayList<RegData> regDatas = getRegDatasWithBulk(repSender, sms.smsType);
        parsedTran = mRegHandler.getTransactionWithReg(regDatas, sms);
        if(parsedTran != null) {
            parsedTran.cardTableData = getCardWithBulk(parsedTran.cardTableData);
            if(parsedTran.userPriority < 10){
                updateUserPriorityWithBulk(repSender, sms.smsType);
            }

        }

        return  parsedTran;

    }

    /**
     * 카드 정보 구하기
     *
     * 카드 정보 없는 경우 인서트
     * 카드 정보 있는 경우 추출
     *
     * @param parsedCard 파싱된 카드 정보
     * @return 카드 정보
     */
    private CardTableData getCardWithBulk(CardTableData parsedCard) throws SQLException {

        LOGI(TAG, "getCard", mIsDebug);

        if(mCardMap == null) mCardMap = new HashMap<>();

        String key = makeCardKey(parsedCard);
        CardTableData cardTableData = mCardMap.get(key);
        if(cardTableData != null) {

            return cardTableData;

        } else {

            cardTableData = getCard(parsedCard);
            mCardMap.put(key, cardTableData);
        }


        return cardTableData;

    }



    private ArrayList<RegData> getRegDatasWithBulk(String repSender, int smsType) {

        LOGI(TAG, "getRegDatasWithBulk", mIsDebug);
        Cursor c = null;
        ArrayList<RegData> regDatas;

        smsType = getSMSType(smsType);

        String key = makeRegKey(repSender, smsType);
        if (mRegMap == null) mRegMap = new HashMap<>();

        regDatas = mRegMap.get(key);

        if(regDatas == null) {

            String query =
                    SELECT + "*" +
                    FROM + ParserReaderContract.RegExpressionTable.TABLE_NAME +
                    WHERE +ParserReaderContract.RegExpressionTable.COLUMN_SMS_TYPE + IN +" (" +smsType + ", 3 )";

            query += AND + ParserReaderContract.RegExpressionTable.COLUMN_SENDER +" = '" + repSender +"'";

            query += ORDER_BY + ParserReaderContract.RegExpressionTable.COLUMN_PRIORITY + DESC + "," +
                    ParserReaderContract.RegExpressionTable.COLUMN_USER_PRIORITY + DESC;
            try{
                c = runQuery(query);
                if (c != null) {
                    if (c.moveToFirst()) {
                        regDatas = new ArrayList<>();
                        while (!c.isAfterLast()) {

                            regDatas.add(ParserReaderContract.RegExpressionTable.populateModel(c, mContext));
                            c.moveToNext();
                        }
                    }
                }
            } catch (SQLiteException e){
                e.printStackTrace();
            } finally {
                if (c != null)
                    c.close();
            }

            if(mRegMap != null)
                mRegMap.put(key, regDatas);

        }
        return regDatas;

    }


    private boolean isExistTranWithBulk(int smsId, int smsType) {
        if (smsId == 0)
            return false;

        if(smsType == Constants.SMSType.SMS.ordinal()) {
            if (PrefUtils.getInstance(mContext).loadIntValue(PrefUtils.MAX_SMS_ID, 0) >= smsId) {
                LOGI(TAG,"maxSmsId:" + PrefUtils.getInstance(mContext).loadIntValue(PrefUtils.MAX_SMS_ID, 0), mIsDebug);
                LOGI(TAG,"smsId:" + smsId, mIsDebug);

                return true;
            }
        } else if(smsType == Constants.SMSType.MMS.ordinal()) {
            if (PrefUtils.getInstance(mContext).loadIntValue(PrefUtils.MAX_MMS_ID, 0) >= smsId) {

                LOGI(TAG,"maxMmsId:" + PrefUtils.getInstance(mContext).loadIntValue(PrefUtils.MAX_MMS_ID, 0), mIsDebug);
                LOGI(TAG,"smsId:" + smsId, mIsDebug);

                return true;
            }
        }

        return false;

    }

    /**
     * 중복 Transaction 추출 함수
     *
     * @param tran 파싱된 Transaction
     * @return 중복된 Transaction
     */
    private TransactionTableData getDuplicationTranWithBulk(TransactionTableData tran) throws SQLException {
        LOGI(TAG, "getDuplicationTran", mIsDebug);

        TransactionTableData duplicationTran = null;
        try {
            //key spendMoney, dwType
            String key = makeSpentMoneyDwTypeKey(tran.spentMoney, tran.dwType);
            LOGI(TAG, "getDuplicationTran key" + key, mIsDebug);
            if(mDuplMap == null)  mDuplMap = new HashMap<>();

            ArrayList<TransactionTableData> duplTrans = mDuplMap.get(key);
            duplicationTran = getDuplicationTranWithBulk(key, duplTrans, tran, true);

            if(duplicationTran == null) {
                int dwType = tran.dwType == Constants.DWType.DEPOSIT.ordinal()? Constants.DWType.WITHDRAW.ordinal() : Constants.DWType.DEPOSIT.ordinal();
                key =  makeSpentMoneyDwTypeKey(-1 * tran.spentMoney, dwType);
                duplTrans = mDuplMap.get(key);
                duplicationTran = getDuplicationTranWithBulk(key, duplTrans, tran, false);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }


        return duplicationTran;


    }

    private TransactionTableData getDuplicationTranWithBulk(String key, ArrayList<TransactionTableData> duplTrans, TransactionTableData tran, boolean isInsert){
        LOGI(TAG, "getDuplicationTranWithBulk", mIsDebug);
        TransactionTableData duplicationTran = null;

        try {
            if(duplTrans == null || duplTrans.isEmpty()) {
                if(isInsert) {
                    duplTrans = new ArrayList<>();
                    duplTrans.add(tran);
                    if(mDuplMap!=null)
                        mDuplMap.put(key, duplTrans);
                }
                return  null;
            } else {
                //존재
                Calendar spendCal = DateUtil.convertStringToCalendarFULL(tran.spentDate);
                Calendar fixedCal = DateUtil.convertStringToCalendarFULL(tran.spentDate);
                fixedCal.add(Calendar.HOUR_OF_DAY, -14);
                spendCal.add(Calendar.MINUTE, 5);

                long endSpendTime = spendCal.getTimeInMillis();
                long fixedSpendTime =  fixedCal.getTimeInMillis();

                if (SAMSUNG_PAY.equals(tran.cardTableData.cardName)) {
                    spendCal.add(Calendar.HOUR_OF_DAY, -13);
                } else {
                    spendCal.add(Calendar.MINUTE, -15);
                }

                long startSpendTime = spendCal.getTimeInMillis();

                for(int i = 0 ; i < duplTrans.size(); i++) {
                    TransactionTableData dupl = duplTrans.get(i);
                    long duplSpendTime = DateUtil.convertStringToCalendarFULL(dupl.spentDate).getTimeInMillis();

                    if(duplSpendTime >= fixedSpendTime) {

                        if((startSpendTime < duplSpendTime && endSpendTime >= duplSpendTime) &&
                                mRegHandler.isDuplicate(dupl.originInfos, tran.originInfos)) {
                            duplicationTran = dupl;
                            duplTrans.remove(i);
                            break;
                        }
                    }
                }
                Collections.sort(duplTrans, new Utils.TranComparator());
                if(mDuplMap!=null)
                    mDuplMap.put(key, duplTrans);
            }
        } catch (Exception e){
            e.printStackTrace();
        }


        return duplicationTran;

    }

    private TransactionTableData getOffsetTranWithBulk(TransactionTableData parsedTran) throws SQLException {
        LOGI(TAG, "getOffsetTran", mIsDebug);

        try {
            if (parsedTran.spentMoney < 0) {

                String key = makeSpentMoneyDwTypeKey(-1*parsedTran.spentMoney ,parsedTran.dwType);
                ArrayList<TransactionTableData> offsetTrans = mOffsetMap.get(key);

                if(offsetTrans == null || offsetTrans.isEmpty())
                    return null;

                TransactionTableData offsetTran = null;
                Calendar spendCal = DateUtil.convertStringToCalendarFULL(parsedTran.spentDate);
                spendCal.add(Calendar.DATE, 1);

                long endSpendTime = spendCal.getTimeInMillis();

                spendCal.add(Calendar.MONTH, -2);

                long startSpendTime = spendCal.getTimeInMillis();

                for(int i = 0 ; i < offsetTrans.size(); i++) {
                    TransactionTableData offset = offsetTrans.get(i);
                    long offsetSpendTime = DateUtil.convertStringToCalendarFULL(offset.spentDate).getTimeInMillis();
                    if(
                            (startSpendTime <= offsetSpendTime && endSpendTime > offsetSpendTime) &&
                                    (parsedTran.keyword.equals(offset.keyword) ||
                                    offset.cardTableData.cardName.contains(parsedTran.cardTableData.cardName) ||
                                    offset.cardTableData.cardName.contains(Utils.transformCardName(parsedTran.cardTableData.cardName))
                                    )
                            ) {
                        offsetTran = offset;
                        offsetTran.isOffset = Constants.YES;
                        offsetTrans.remove(i);
                        break;
                    }

                }
                Collections.sort(offsetTrans, new Utils.TranComparator());
                if(mOffsetMap!=null)
                    mOffsetMap.put(key, offsetTrans);
                return offsetTran;


            } else {
                if(mOffsetMap == null)mOffsetMap = new ArrayMap<>();
                String key = makeSpentMoneyDwTypeKey(parsedTran.spentMoney, parsedTran.dwType);
                ArrayList<TransactionTableData> offsetTrans = mOffsetMap.get(key);
                if(offsetTrans==null)offsetTrans = new ArrayList<>();
                offsetTrans.add(parsedTran);
                mOffsetMap.put(key, offsetTrans);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }


        return null;
    }

    /**
     * 자산이동 Transaction 추출 함수
     * @param tran 파싱된 Transaction
     * @return 자산이동 Transaction
     */
    private TransactionTableData getMovingAssetTranWithBulk(@NonNull TransactionTableData tran) throws SQLException {

        LOGI(TAG, "getMovingAssetTran", mIsDebug);

        try {
            if (tran.cardTableData.cardType == Constants.CardType.BANK_ACCOUNT.ordinal()) {
                String key = tran.spentMoney + "";
                if(mMovingAssetMap == null)  mMovingAssetMap = new HashMap<>();

                ArrayList<TransactionTableData> assetTrans = mMovingAssetMap.get(key);

                if(assetTrans == null || assetTrans.isEmpty()) {
                    assetTrans = new ArrayList<>();
                    assetTrans.add(tran);

                    mMovingAssetMap.put(key, assetTrans);
                    return  null;
                } else {

                    TransactionTableData movingAssetTran = null;
                    //존재
                    Calendar spendCal = DateUtil.convertStringToCalendarFULL(tran.spentDate);
                    spendCal.add(Calendar.MINUTE, 3);
                    long endSpendTime = spendCal.getTimeInMillis();

                    spendCal.add(Calendar.MINUTE, -6);
                    long startSpendTime = spendCal.getTimeInMillis();


                    for(int i = 0 ; i < assetTrans.size(); i++) {
                        TransactionTableData asset = assetTrans.get(i);
                        long assetSpendTime = DateUtil.convertStringToCalendarFULL(asset.spentDate).getTimeInMillis();

                        if((startSpendTime < assetSpendTime && endSpendTime >= assetSpendTime) &&
                                !(asset.cardTableData.cardName.equals(tran.cardTableData.cardName) &&
                                        asset.cardTableData.cardNum.equals(tran.cardTableData.cardNum))&&
                                asset.dwType != tran.dwType) {
                            movingAssetTran = asset;
                            assetTrans.remove(i);
                            break;
                        }

                    }
                    Collections.sort(assetTrans, new Utils.TranComparator());
                    mMovingAssetMap.put(key, assetTrans);
                    return movingAssetTran;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return null;
    }

    void initMap() {

        mRegMap =  new HashMap<>();
        mSendersMap =  new HashMap<>();
        mDuplMap =  new HashMap<>();
        mMovingAssetMap =  new HashMap<>();
        mOffsetMap =  new HashMap<>();
        mTransactionMap =  new HashMap<>();
        mCardMap =  new HashMap<>();


    }

    void setSendersWithBulk(){

        LOGI(TAG, "setSenders", mIsDebug);

        Cursor c = null;
        String query =
                SELECT + ParserReaderContract.SendersTable.COLUMN_SENDER +"," + ParserReaderContract.SendersTable.COLUMN_REP_SENDER +
                FROM + ParserReaderContract.SendersTable.TABLE_NAME;

        try {
            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        String key = c.getString(c.getColumnIndex(ParserReaderContract.SendersTable.COLUMN_SENDER));
                        String repSender = c.getString(c.getColumnIndex(ParserReaderContract.SendersTable.COLUMN_REP_SENDER));
                        if(mSendersMap != null && !repSender.equals(Constants.ALL)) {
                            mSendersMap.put(key,repSender);
                        }
                        c.moveToNext();
                    }
                }
            }
        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            if (c != null)
                c.close();
        }

    }

    void insertTransactionAndMaxSmsId(ArrayList<Transaction> transactions) {
        LOGI(TAG, "insertTransactionAndMaxSmsId", mIsDebug);

        int maxSMSId = 0;
        int maxMMSId = 0;

        try {

            Map<Integer, RegData> regDataMap = new HashMap<>();
            Map<Integer, CardTableData> cardMap = new HashMap<>();

            ArrayList<String> values = new ArrayList<>();
            for(int i = 0; i < transactions.size() ; i++ ) {

                Transaction tran = transactions.get(i);
                if(mTransactionMap!=null) {
                    TransactionTableData transactionTableData = mTransactionMap.get(tran.identifier);

                    if(transactionTableData!=null) {
                        transactionTableData.isSuccess = 1;
                        values.add(transactionTableData.insertValue());// bulk insert transaction

                        //update userPriority
                        String key = makeRegKey(transformSender(transactionTableData.sender), getSMSType(transactionTableData.smsType));
                        ArrayList<RegData>  regDatas = mRegMap.get(key);
                        if(regDatas!=null) {
                            for( RegData regData : regDatas ){
                                if(regData.regId == transactionTableData.regId) {
                                    regDataMap.put(regData.regId, regData);
                                    break;
                                }
                            }
                        }

                        //update card
                        if(mCardMap!=null) {
                            CardTableData cardTableData = mCardMap.get(makeCardKey(transactionTableData.cardTableData));
                            if(cardTableData!=null) {
                                if(transactionTableData.cardTableData.cardId == cardTableData.cardId) {
                                    cardMap.put(cardTableData.cardId, cardTableData);
                                }
                            }
                        }
                    }
                }


                if(tran.smsType == Constants.SMSType.SMS.ordinal()) {

                    if(maxSMSId < tran.smsId) {
                        maxSMSId = tran.smsId;
                    }


                } else if(tran.smsType == Constants.SMSType.MMS.ordinal()){
                    if(maxMMSId < tran.smsId) {
                        maxMMSId = tran.smsId;
                    }
                }

            }

            //update card
            for( Map.Entry<Integer, CardTableData> elem : cardMap.entrySet() ){
                ContentValues values1 = ParserReaderContract.CardTable.populateContent(elem.getValue());
                String[] selectionArgs = { elem.getKey()+"" };
                update(ParserReaderContract.CardTable.TABLE_NAME, values1, ParserReaderContract.CardTable._ID + "= ?", selectionArgs);
            }


            //update userPriority
            for( Map.Entry<Integer, RegData> elem : regDataMap.entrySet() ){
                ContentValues regValues = new ContentValues();
                regValues.put(ParserReaderContract.RegExpressionTable.COLUMN_USER_PRIORITY, elem.getValue().userPriority);
                update(ParserReaderContract.RegExpressionTable.TABLE_NAME, regValues, ParserReaderContract.RegExpressionTable.COLUMN_REG_ID+"="+elem.getKey(), null);
            }


            //bulk insert transaction
            if(!values.isEmpty()) {
                LOGI(TAG, INSERT_TRANSACTION+ TextUtils.join(",", values), mIsDebug);
                wdb.execSQL(INSERT_TRANSACTION+ TextUtils.join(",", values));
                values.clear();
            }

            LOGI(TAG, "maxSMSId" + maxSMSId, mIsDebug);
            LOGI(TAG, "maxMMSId" + maxMMSId, mIsDebug);

            if(maxSMSId != 0)
                PrefUtils.getInstance(mContext).saveIntValue(PrefUtils.MAX_SMS_ID, maxSMSId);

            if(maxMMSId != 0)
                PrefUtils.getInstance(mContext).saveIntValue(PrefUtils.MAX_MMS_ID, maxMMSId);

        } catch (Exception e) {
            e.printStackTrace();
        }


    }


    private void updateUserPriorityWithBulk(String repSender, int smsType) {

        if(mRegMap != null) {
            String key = makeRegKey(repSender, getSMSType(smsType));
            ArrayList<RegData> regDatas = mRegMap.get(key);
            if(regDatas!=null) {
                Collections.sort(regDatas, new Utils.RegComparator());
                mRegMap.put(key, regDatas);
            }
        }
    }



    void onBulkComplete() {

        mRegMap = null;
        mSendersMap = null;
        mDuplMap = null;
        mMovingAssetMap = null;
        mOffsetMap = null;
        mTransactionMap = null;
        mCardMap = null;

    }



    /**************************************      ONE         *************************************************/


    /**
     * is_success = 0 인 리스트 함께 보냄
     * 0. sender 필터
     * 1. 정규표현식을 통한 데이터 추출
     * 2. 중복 데이터 확인
     * 3. 중복 x 상쇄 Tran 확인
     * 4. 중복 x 상쇄 x 자산이동 Tran 확인
     * 5. 잔액 변경

     * @param sms 수신된 문자
     * @return 파싱된 Tran List
     */
    ArrayList<Transaction> parse(SMS sms) {

        TransactionTableData transaction;
        TransactionTableData offsetTran = null;
        TransactionTableData movingAssetTran = null;

        if(!isSpendSMS(sms.fullSms))
            return null;
        //0. sender 필터
        String repSender = getRepSender(sms.sender, sms.displaySender);
        if(repSender == null) {
            repSender = transformRepSender(sms.fullSms);
        }
        LOGI(TAG, "0. sender 대표 필터" + repSender, mIsDebug);
        if (repSender == null)
            return null;

        //1. 정규표현식을 통한 데이터 확인
        LOGI(TAG, "1. 정규표현식을 통한 데이터 확인", mIsDebug);
        transaction = getTransactionWithReg(sms, repSender);
        if (transaction == null) {
            LOGI(TAG, "1. 정규표현식을 통한 데이터 확인: FALSE", mIsDebug);
            LOGI(TAG, "NOT PARSED"+sms.toString(), mIsDebug);
            return null;
        }

        LOGI(TAG, "1. 정규표현식을 통한 데이터 확인: TRUE" + transaction.toString(), mIsDebug);
        //2. 중복 데이터 확인
        LOGI(TAG, "2. 중복 데이터 확인", mIsDebug);
        TransactionTableData duplicationTran = getDuplicationTran(transaction);

        //중복
        if (duplicationTran != null) {

            LOGI(TAG, "2. 중복 데이터 확인: TRUE" + duplicationTran.toString(), mIsDebug);
            //중복인 경우 우선순위에 따라 정보 변경
            transaction = mRegHandler.combineTran(duplicationTran, transaction);

        } else {
            //중복 아님
            LOGI(TAG, "2. 중복 데이터 확인: FALSE", mIsDebug);

            //상쇄 확인
            offsetTran = getOffsetTran(transaction);
            LOGI(TAG, "3. 중복 x 상쇄 Tran 확인", mIsDebug);

            //상쇄
            if (offsetTran != null) {

                LOGI(TAG, "3. 중복 x -> 상쇄 Tran 확인: TRUE" + offsetTran.toString(), mIsDebug);
                transaction = mRegHandler.combineTranWithOffset(offsetTran, transaction);


            } else {
                //상쇄아님
                LOGI(TAG, "3. 중복 x -> 상쇄 Tran 확인: FALSE", mIsDebug);

                //자산이동 확인
                LOGI(TAG, "4. 중복 x -> 상쇄 x -> 자산이동 Tran 확인", mIsDebug);

                movingAssetTran = getMovingAssetTran(transaction);

                //자산이동
                if (movingAssetTran != null) {
                    LOGI(TAG, "4. 중복 x 상쇄 x 자산이동 Tran 확인: TRUE" + movingAssetTran.toString(), mIsDebug);

                    transaction = mRegHandler.combineTranWithMovingAsset(transaction);
                    movingAssetTran = mRegHandler.combineTranWithMovingAsset(movingAssetTran);

                } else {

                    LOGI(TAG, "4. 중복 x 상쇄 x 자산이동 Tran 확인: FALSE", mIsDebug);

                }
            }
        }
        //잔액 업데이트
        if(transaction != null) transaction = mRegHandler.updateBalance(transaction);

        //목록 넣기
        ArrayList<Transaction> failTrans = getFailTransaction();
        ArrayList<Transaction> resultTrans = getResultTransactions(transaction, offsetTran, movingAssetTran);
        resultTrans.addAll(failTrans);
        if(resultTrans.size() == 0)
            return null;

        LOGI(TAG, "최종 리턴 사이즈:" + resultTrans.size(), mIsDebug);
        return resultTrans;

    }

    private ArrayList<Transaction> getResultTransactions(TransactionTableData transaction,
                                                         TransactionTableData offsetTran,
                                                         TransactionTableData movingAssetTran) {

        ArrayList<Transaction> resultTrans = new ArrayList<>();

        try {
            ArrayList<TransactionTableData> transactions = new ArrayList<>();
            if(transaction != null)
                transactions.add(transaction);
            if(offsetTran != null)
                transactions.add(offsetTran);
            if(movingAssetTran != null)
                transactions.add(movingAssetTran);

            //리턴 맵핑
            for (TransactionTableData tran : transactions) {

                Transaction resultTran = mMapper.transform(tran);
                if(resultTran!=null) {
                    mergeTransaction(tran);
                    LOGI(TAG, "result"+resultTran.toString(), mIsDebug);
                    resultTrans.add(resultTran);
                }

            }
        }catch (Exception e) {
            e.printStackTrace();
        }

        return resultTrans;
    }





    /**
     * 대표 sender 구하기
     *
     * @param sender 수신된 전화번호 또는 패키지
     * @return 대표 Sender
     */
    private String getRepSender(String sender, String displaySender) {

        LOGI(TAG, "getRepSender", mIsDebug);
        Cursor c = null;
        String query = "";
        if (TextUtils.isEmpty(displaySender) ||
                sender.equals(displaySender)) {

            String transformSender = transformSender(sender);
            query +=
                    SELECT + ParserReaderContract.SendersTable.COLUMN_REP_SENDER +
                    FROM + ParserReaderContract.SendersTable.TABLE_NAME +
                    WHERE + ParserReaderContract.SendersTable.COLUMN_SENDER + IN + "('"+sender+"','"+transformSender+"')" +
                    LIMIT + "1";
        } else {

            String transformSender = transformSender(sender);
            String transformDisplaySender = transformSender(displaySender);
            query +=
                    SELECT + ParserReaderContract.SendersTable.COLUMN_REP_SENDER +
                    FROM + ParserReaderContract.SendersTable.TABLE_NAME +
                    WHERE + ParserReaderContract.SendersTable.COLUMN_SENDER + IN + "('"+sender+"','"+transformSender+"','"+displaySender+"','"+transformDisplaySender+"')" +
                    LIMIT + "1";

        }


        LOGI(TAG, "getRepSender query:" + query, mIsDebug);

        try {

            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {
                    return c.getString(c.getColumnIndex(ParserReaderContract.SendersTable.COLUMN_REP_SENDER));
                }
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            if (c != null)
                c.close();
        }

        return null;

    }




    /**
     * 실패 내역 뽑아오기
     * @return fail tran list
     */
    private ArrayList<Transaction> getFailTransaction () {

        LOGI(TAG, "getFailTransaction", mIsDebug);

        ArrayList<Transaction> failTrans = new ArrayList<>();

        Cursor c = null;
        String query =
                SELECT + "*" +
                        FROM + getJoinTable() +
                        WHERE+  ParserReaderContract.TransactionsTable.ALIAS + ParserReaderContract.TransactionsTable.COLUMN_IS_SUCCESS + "=" + Constants.NO +
                        ORDER_BY + ParserReaderContract.TransactionsTable.ALIAS+ ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE + ASC;

        try {
            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {
                    while (!c.isAfterLast()) {

                        TransactionTableData failTran = ParserReaderContract.TransactionsTable.populateModel(c);
                        failTran.cardTableData = ParserReaderContract.CardTable.populateModel(c);
                        Transaction tran = mMapper.transform(failTran);
                        if(tran!=null)
                            failTrans.add(tran);
                        c.moveToNext();
                    }
                }
            }
        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            if (c != null)
                c.close();
        }

        return failTrans;

    }

    /**
     * 정규표현식을 통한 Transaction 추출
     *
     * 1. getRegDatas map.get(sender) == null? where sender and smsType
     * 2. matching start
     * 3. matched -> TransactionTableData , cardTableData, 데이터 셋팅
     *
     * @param sms 수신된 문자 or 알림
     * @return 파싱된 Transaction
     */
     private TransactionTableData getTransactionWithReg(SMS sms, String repSender) throws SQLException{

         LOGI(TAG, "getTransactionWithReg start" + sms.toString(), mIsDebug);
         TransactionTableData parsedTran;

         String newFullSms  = Utils.transformFullSMS(sms.fullSms);
         LOGI(TAG, "getTransactionWithReg newFullSms" + newFullSms, mIsDebug);


         ArrayList<RegData> regDatas = getRegDatas(repSender, sms.smsType);


         parsedTran = mRegHandler.getTransactionWithReg(regDatas, sms);
         if(parsedTran != null) {

             parsedTran.cardTableData = getCard(parsedTran.cardTableData);
             if(parsedTran.userPriority < 10){
                 updateUserPriority(parsedTran.regId, parsedTran.userPriority);
             }

         }

         return  parsedTran;
    }


    /**
     * 카드 정보 구하기
     *
     * 카드 정보 없는 경우 인서트
     * 카드 정보 있는 경우 추출
     *
     * @param parsedCard 파싱된 카드 정보
     * @return 카드 정보
     */
    private CardTableData getCard(CardTableData parsedCard) throws SQLException {

        LOGI(TAG, "getCard", mIsDebug);

        Cursor c = null;
        String query =
                SELECT + "*" +
                FROM + ParserReaderContract.CardTable.TABLE_NAME +
                WHERE + ParserReaderContract.CardTable.COLUMN_CARD_NAME + "='" + parsedCard.cardName + "'" +
                    AND + ParserReaderContract.CardTable.COLUMN_CARD_NUM + "='" + parsedCard.cardNum + "'" +
                    AND + ParserReaderContract.CardTable.COLUMN_CARD_TYPE + "=" + parsedCard.cardType +
                    AND + ParserReaderContract.CardTable.COLUMN_CARD_SUB_TYPE + "=" + parsedCard.cardSubType;
        try {

            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {
                    while (!c.isAfterLast()) {

                        parsedCard = ParserReaderContract.CardTable.populateModel(c);
                        c.moveToNext();

                    }
                }

            } else {
                parsedCard.cardId = (int)insert(ParserReaderContract.CardTable.TABLE_NAME, ParserReaderContract.CardTable.populateContent(parsedCard));
            }

        } catch (SQLiteException e){
            e.printStackTrace();
        }finally {
            if (c != null)
                c.close();
        }


        return parsedCard;

    }


    private void updateUserPriority(int id, int userPriority) {

        ContentValues values = new ContentValues();
        values.put(ParserReaderContract.RegExpressionTable.COLUMN_USER_PRIORITY, ++userPriority);
        update(ParserReaderContract.RegExpressionTable.TABLE_NAME, values, ParserReaderContract.RegExpressionTable.COLUMN_REG_ID+"="+id, null);
    }

    private ArrayList<RegData> getRegDatas(String sender, int smsType) {

        LOGI(TAG, "getRegDatas", mIsDebug);
        Cursor c = null;
        ArrayList<RegData> regDatas = null;

        smsType = getSMSType(smsType);
        String query =
                SELECT + "*" +
                        FROM + ParserReaderContract.RegExpressionTable.TABLE_NAME +
                        WHERE +ParserReaderContract.RegExpressionTable.COLUMN_SMS_TYPE + IN +" (" +smsType + ", 3 )";


        query += (ALL.equals(sender) ||(smsType==0&&mIsDebug))?
                ""
                :
                AND + ParserReaderContract.RegExpressionTable.COLUMN_SENDER +" = '" + sender + "'";

        query += ORDER_BY + ParserReaderContract.RegExpressionTable.COLUMN_PRIORITY + DESC + "," +
                ParserReaderContract.RegExpressionTable.COLUMN_USER_PRIORITY + DESC;
        try{
            LOGI(TAG, "getRegDatas query:" + query, mIsDebug);

            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {
                    regDatas = new ArrayList<>();
                    while (!c.isAfterLast()) {

                        regDatas.add(ParserReaderContract.RegExpressionTable.populateModel(c, mContext));
                        c.moveToNext();
                    }
                }
            }

        } catch (SQLiteException e){
            LOGE(TAG, e.toString());
        }finally {
            if (c != null)
                c.close();
        }

        return regDatas;

    }

    /**
     * 중복 Transaction 추출 함수
     *
     * @param tran 파싱된 Transaction
     * @return 중복된 Transaction
     */
    private TransactionTableData getDuplicationTran(TransactionTableData tran) throws SQLException {
        LOGI(TAG, "getDuplicationTran", mIsDebug);

        TransactionTableData duplicationTran;

        Cursor c = null;

        Calendar cal = DateUtil.convertStringToCalendarFULL(tran.spentDate);
        cal.add(Calendar.MINUTE, 5);
        String endDate = DateUtil.getStringDateAsYYYYMMddHHmm(cal);

        if (SAMSUNG_PAY.equals(tran.cardTableData.cardName)) {
            cal.add(Calendar.HOUR_OF_DAY, -13);
        } else {
            cal.add(Calendar.MINUTE, -15);
        }

        String startDate = DateUtil.getStringDateAsYYYYMMddHHmm(cal);

        String query =

                SELECT + "*" +
                FROM + getJoinTable()+
                WHERE +
                    "strftime(" + YYYY_MM_DD_H_M + "," + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE + ")>'" + startDate + "' " +
                    AND + " strftime(" + YYYY_MM_DD_H_M+ "," + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE+ ")<='" + endDate + "' " +
                    AND + "(" +
                    "(" +
                    ParserReaderContract.TransactionsTable.COLUMN_SPENT_MONEY + "=" + tran.spentMoney+
                    AND + ParserReaderContract.TransactionsTable.COLUMN_DW_TYPE + "=" + tran.dwType+
                    ")" +
                    OR +
                    "(" +
                    ParserReaderContract.TransactionsTable.COLUMN_SPENT_MONEY + "=" + -1*tran.spentMoney +
                    AND + ParserReaderContract.TransactionsTable.COLUMN_DW_TYPE + "!=" + tran.dwType +
                    ")" +
                    ")" +
                ORDER_BY + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE + DESC +
                LIMIT + " 1 ";

        try {
            c = runQuery(query);

            if(c != null){
                if (c.moveToFirst()) {
                    while (!c.isAfterLast()) {

                        duplicationTran = ParserReaderContract.TransactionsTable.populateModel(c);
                        duplicationTran.cardTableData = ParserReaderContract.CardTable.populateModel(c);
                        //처음 잡힌 정보에따른 중복 체크 함수
                        if(mRegHandler.isDuplicate(duplicationTran.originInfos, tran.originInfos)) {
                            return  duplicationTran;
                        }

                        c.moveToNext();
                    }
                }
            }

        } catch (SQLiteException e){
            LOGE(TAG, e.toString());
        } finally {
            if (c != null)
                c.close();
        }

        return  null;

    }

    /**
     * 승인 인식 후 취소 ( 합 == 0) 와 같이 상쇄 Transaction 추출 함수
     *
     * @param tran 파싱된 Transaction
     * @return 상쇄된 Transaction
     */
    private TransactionTableData getOffsetTran(TransactionTableData tran) throws SQLException {
        LOGI(TAG, "getOffsetTran", mIsDebug);

        if (tran.spentMoney < 0) {

            Cursor c = null;
            Calendar cal= DateUtil.convertStringToCalendarFULL(tran.spentDate);
            cal.add(Calendar.DATE, 1);
            String endDate = DateUtil.getStringDateAsYYYYMMddHHmmss(cal);
            cal.add(Calendar.MONTH, -2);
            String startDate = DateUtil.getStringDateAsYYYYMMddHHmmss(cal);

            String query =
                    SELECT + "*" +
                    FROM + getJoinTable() +
                    WHERE+  ParserReaderContract.TransactionsTable.ALIAS + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE+ " >='" + startDate+"' " +
                        AND + ParserReaderContract.TransactionsTable.ALIAS + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE + " < '" + endDate+ "'" +
                        AND + ParserReaderContract.TransactionsTable.ALIAS + ParserReaderContract.TransactionsTable.COLUMN_SPENT_MONEY + "="+ -1*tran.spentMoney +
                        AND + ParserReaderContract.TransactionsTable.ALIAS + ParserReaderContract.TransactionsTable.COLUMN_DW_TYPE + "=" + tran.dwType +
                        AND + ParserReaderContract.TransactionsTable.ALIAS + ParserReaderContract.TransactionsTable.COLUMN_IS_OFFSET+ " = " + NO +
                        AND +
                        "(" +
                        ParserReaderContract.TransactionsTable.ALIAS + ParserReaderContract.TransactionsTable.COLUMN_KEYWORD+ "='"+ tran.keyword.trim() + "'"+
                        OR +
                        ParserReaderContract.CardTable.ALIAS + ParserReaderContract.CardTable.COLUMN_CARD_NAME + " like '%" + tran.cardTableData.cardName + "%'" +
                        OR +
                        ParserReaderContract.CardTable.ALIAS + ParserReaderContract.CardTable.COLUMN_CARD_NAME + " like '%" + Utils.transformCardName(tran.cardTableData.cardName) + "%'" +
                        ")" +

                    ORDER_BY + ParserReaderContract.TransactionsTable.ALIAS+ ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE + DESC+
                    LIMIT + "1";

            LOGI(TAG, "offset query"+ query);


            try{
                c = runQuery(query);
                if(c != null){
                    if (c.moveToFirst()) {
                        TransactionTableData offsetTran = ParserReaderContract.TransactionsTable.populateModel(c);
                        offsetTran.isOffset = YES;
                        offsetTran.cardTableData = ParserReaderContract.CardTable.populateModel(c);
                        return offsetTran;
                    }
                }
            } catch (SQLiteException e){

                e.printStackTrace();

            } finally {
                if (c != null)
                    c.close();
            }

        }
        return null;
    }


    /**
     * 자산이동 Transaction 추출 함수
     * @param tran 파싱된 Transaction
     * @return 자산이동 Transaction
     */
    private TransactionTableData getMovingAssetTran(@NonNull TransactionTableData tran) throws SQLException {

        TransactionTableData movingAssetTran;
        LOGI(TAG, "getMovingAssetTran", mIsDebug);
        //앞+ 3분 뒤 3분 and dwTyp!= spent_money= cardName+cardNum cardType bank
        //spentMoney, cardType
        if (tran.cardTableData.cardType == Constants.CardType.BANK_ACCOUNT.ordinal()) {


            Calendar cal= DateUtil.convertStringToCalendarFULL(tran.spentDate);
            cal.add(Calendar.MINUTE, 3);
            String endDate= DateUtil.getStringDateAsYYYYMMddHHmm(cal);
            cal.add(Calendar.MINUTE, -6);
            String startDate= DateUtil.getStringDateAsYYYYMMddHHmm(cal);
            String query =
                    SELECT + "*" +
                    FROM + getJoinTable() +
                    WHERE +
                        "strftime(" + YYYY_MM_DD_H_M+ "," + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE+ ")>'" + startDate+ "' " +
                        AND + " strftime(" + YYYY_MM_DD_H_M + "," + ParserReaderContract.TransactionsTable.COLUMN_SPENT_DATE+ ")<='" + endDate+ "' " +
                        AND + ParserReaderContract.TransactionsTable.COLUMN_DW_TYPE+"!=" + tran.dwType +
                        AND + ParserReaderContract.TransactionsTable.COLUMN_SPENT_MONEY+"=" + tran.spentMoney +
                        AND + ParserReaderContract.TransactionsTable.COLUMN_CATEGORY_CODE + " NOT IN(98,88)" +
                        AND + "(" + ParserReaderContract.CardTable.COLUMN_CARD_NAME + "!='" + tran.cardTableData.cardName+ "'" +
                        OR + ParserReaderContract.CardTable.COLUMN_CARD_NUM + "!='" + tran.cardTableData.cardNum+ "'" +
                        ")" +
                        AND + ParserReaderContract.CardTable.COLUMN_CARD_TYPE + "="+Constants.CardType.BANK_ACCOUNT.ordinal() +
                    LIMIT + " 1";

            Cursor c = null;
            try {
                c = runQuery(query);
                if(c != null){
                    if (c.moveToFirst()) {
                        movingAssetTran = ParserReaderContract.TransactionsTable.populateModel(c);
                        movingAssetTran.cardTableData = ParserReaderContract.CardTable.populateModel(c);
                        return movingAssetTran;
                    }
                }
            } catch (SQLiteException e){
                LOGE(TAG, e.toString());
            } finally {
                if (c != null)
                    c.close();
            }
        }
        return null;
    }

    private void mergeTransaction(TransactionTableData tran) throws SQLException{

        LOGI(TAG, "mergeTransaction", mIsDebug);
        if (tran == null)
            return;

        ContentValues values = ParserReaderContract.TransactionsTable.populateContent(tran);
        String[] selectionArgs = { tran.identifier+"" };

        long suc = update(ParserReaderContract.TransactionsTable.TABLE_NAME, values, ParserReaderContract.TransactionsTable.COLUMN_IDENTIFIER + "= ?", selectionArgs);
        if (suc == 0) {
            insert(ParserReaderContract.TransactionsTable.TABLE_NAME, values);
        }

        values = ParserReaderContract.CardTable.populateContent(tran.cardTableData);
        selectionArgs = new String[]{ tran.cardTableData.cardId+"" };

        suc = update(ParserReaderContract.CardTable.TABLE_NAME, values, ParserReaderContract.CardTable._ID + "= ?", selectionArgs);
        if (suc == 0) {
            insert(ParserReaderContract.CardTable.TABLE_NAME, values);
        }

    }

    void updateIsSuccess(ArrayList<Transaction> transactions, int isSuccess) throws SQLException  {

        LOGI(TAG, "updateIsSuccess", mIsDebug);
        ArrayList<String> identifierList = new ArrayList<>();
        for (Transaction transaction : transactions) {

            identifierList.add("'" + transaction.identifier + "'");

        }

        ContentValues values = new ContentValues();
        values.put(ParserReaderContract.TransactionsTable.COLUMN_IS_SUCCESS, isSuccess);
        update(ParserReaderContract.TransactionsTable.TABLE_NAME, values, ParserReaderContract.TransactionsTable.COLUMN_IDENTIFIER + IN + "("+ TextUtils.join(",", identifierList)+")", null);
    }


    void deleteOldTransaction(String smsDate) {

        LOGI(TAG, "deleteOldTransaction", mIsDebug);
        Calendar cal = DateUtil.convertStringToCalendarFULL(smsDate);
        cal.add(Calendar.MONTH, -2);
        delete(ParserReaderContract.TransactionsTable.TABLE_NAME, ParserReaderContract.TransactionsTable.COLUMN_SMS_DATE + "<'"+ DateUtil.getStringDateAsYYYYMMddHHmmss(cal)+"'", null);

    }


    /**************************************      ETC         *************************************************/

    /**
     * transactions 테이블 초기화, senders delete where id<0
     */
    void initDb() throws SQLException {
        LOGI(TAG, "initDb", mIsDebug);
        PrefUtils.getInstance(mContext).clear();
        DatabaseHelper.getInstance(mContext).onCreate(wdb);
    }

    /**
     * 사용자 추가 번호 아이디는 최소아이디 값 -1 (음수 처리)
     * 1.get senders min id
     * 2. min id > 0 ? id = -1 : id = min_id -1
     * 3. insert
     * @param number 전화번호
     */
    void addTestOriginNumber(String number) throws SQLException {

        LOGI(TAG, "addTestOriginNumber", mIsDebug);
        int minId = getMinSendersId();
        int id = minId >= 0 ? -1 : minId - 1;

        Sender sender = new Sender();
        sender.senderId = id;
        sender.sender = transformSender(number);
        sender.repSender = ALL;
        sender.smsType = Constants.SMSType.SMS.ordinal();

        ContentValues values = ParserReaderContract.SendersTable.populateContent(sender);

        String[] selectionArgs = { sender.sender };

        long suc = update(ParserReaderContract.SendersTable.TABLE_NAME, values, ParserReaderContract.SendersTable.COLUMN_SENDER + "= ?", selectionArgs);
        if(suc == 0) {
            insert(ParserReaderContract.SendersTable.TABLE_NAME, values);
        }

    }


    private int getMinSendersId(){

        LOGI(TAG, "getMinSendersId", mIsDebug);

        Cursor c = null;
        String query =
                SELECT + "MIN("+ ParserReaderContract.SendersTable.COLUMN_SENDER_ID+")" +
                FROM + ParserReaderContract.SendersTable.TABLE_NAME;

        try {
            c = runQuery(query);
            if (c != null) {
                if (c.moveToFirst()) {

                    return c.getInt(0);

                }
            }
        } catch (SQLException e) {
            LOGE(TAG, e.toString());
        } finally {
            if (c != null)
                c.close();
        }

        return 0;

    }

    /**
     * regId 가 같으면 업데이트 다르면 insert
     * isDelete == 1 이면 삭제
     *
     * @param parsingRule 파싱룰 데이터
     */
    void syncParsingRule(ParsingRule parsingRule) {

        try {
            LOGI(TAG, "syncParsingRule", mIsDebug);
            //upsert reg
            mergeRegDatas(parsingRule.regDatas);
            //upsert senders
            mergeSenders(parsingRule.senders);


        } catch (SQLiteException e) {
            e.printStackTrace();

        }

    }

    private void mergeRegDatas(ArrayList<RegData> regDatas) throws SQLException {

        LOGI(TAG, "mergeRegDatas", mIsDebug);
        if(regDatas == null)
            return;

        for (RegData regData : regDatas) {

            ContentValues values = ParserReaderContract.RegExpressionTable.populateContent(regData);
            String[] selectionArgs = { regData.regId+"" };

            if (regData.isDelete == YES) {

                delete(ParserReaderContract.RegExpressionTable.TABLE_NAME, ParserReaderContract.RegExpressionTable.COLUMN_REG_ID + "= ?", selectionArgs);

            } else {
                long suc = update(ParserReaderContract.RegExpressionTable.TABLE_NAME, values, ParserReaderContract.RegExpressionTable.COLUMN_REG_ID + "= ?", selectionArgs);
                if(suc == 0) {
                    insert(ParserReaderContract.RegExpressionTable.TABLE_NAME, values);
                }
            }
        }
    }

    private void mergeSenders(ArrayList<Sender> senders) throws SQLException{

        LOGI(TAG, "mergeSenders", mIsDebug);
        if (senders == null) {
            return;
        }

        for (Sender sender : senders) {

            ContentValues values = ParserReaderContract.SendersTable.populateContent(sender);
            String[] selectionArgs = { sender.senderId+"" };

            if (sender.isDelete == YES) {

                delete(ParserReaderContract.SendersTable.TABLE_NAME, ParserReaderContract.SendersTable.COLUMN_SENDER_ID + "= ?", selectionArgs);

            } else {
                long suc = update(ParserReaderContract.SendersTable.TABLE_NAME, values, ParserReaderContract.SendersTable.COLUMN_SENDER_ID + "= ?", selectionArgs);
                if (suc == 0) {
                    insert(ParserReaderContract.SendersTable.TABLE_NAME, values);
                }
            }
        }
    }


}
