/**
 *
 * Copyright 2013-2015 Martin Goellnitz
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 */
package org.tangram.mutable;

import com.thoughtworks.xstream.XStream;
import static com.thoughtworks.xstream.XStream.PRIORITY_NORMAL;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.basic.BigDecimalConverter;
import com.thoughtworks.xstream.converters.basic.BigIntegerConverter;
import com.thoughtworks.xstream.converters.basic.BooleanConverter;
import com.thoughtworks.xstream.converters.basic.ByteConverter;
import com.thoughtworks.xstream.converters.basic.CharConverter;
import com.thoughtworks.xstream.converters.basic.DateConverter;
import com.thoughtworks.xstream.converters.basic.DoubleConverter;
import com.thoughtworks.xstream.converters.basic.FloatConverter;
import com.thoughtworks.xstream.converters.basic.IntConverter;
import com.thoughtworks.xstream.converters.basic.LongConverter;
import com.thoughtworks.xstream.converters.basic.NullConverter;
import com.thoughtworks.xstream.converters.basic.ShortConverter;
import com.thoughtworks.xstream.converters.basic.StringBufferConverter;
import com.thoughtworks.xstream.converters.basic.StringConverter;
import com.thoughtworks.xstream.converters.basic.URIConverter;
import com.thoughtworks.xstream.converters.basic.URLConverter;
import com.thoughtworks.xstream.converters.collections.ArrayConverter;
import com.thoughtworks.xstream.converters.collections.BitSetConverter;
import com.thoughtworks.xstream.converters.collections.CharArrayConverter;
import com.thoughtworks.xstream.converters.collections.CollectionConverter;
import com.thoughtworks.xstream.converters.collections.MapConverter;
import com.thoughtworks.xstream.converters.collections.PropertiesConverter;
import com.thoughtworks.xstream.converters.collections.SingletonCollectionConverter;
import com.thoughtworks.xstream.converters.collections.SingletonMapConverter;
import com.thoughtworks.xstream.converters.collections.TreeMapConverter;
import com.thoughtworks.xstream.converters.collections.TreeSetConverter;
import com.thoughtworks.xstream.converters.enums.EnumConverter;
import com.thoughtworks.xstream.converters.enums.EnumMapConverter;
import com.thoughtworks.xstream.converters.enums.EnumSetConverter;
import com.thoughtworks.xstream.converters.extended.EncodedByteArrayConverter;
import com.thoughtworks.xstream.converters.extended.FileConverter;
import com.thoughtworks.xstream.converters.extended.GregorianCalendarConverter;
import com.thoughtworks.xstream.converters.extended.JavaClassConverter;
import com.thoughtworks.xstream.converters.extended.JavaFieldConverter;
import com.thoughtworks.xstream.converters.extended.JavaMethodConverter;
import com.thoughtworks.xstream.converters.extended.LocaleConverter;
import com.thoughtworks.xstream.converters.reflection.ExternalizableConverter;
import com.thoughtworks.xstream.converters.reflection.PureJavaReflectionProvider;
import com.thoughtworks.xstream.converters.reflection.ReflectionConverter;
import com.thoughtworks.xstream.converters.reflection.ReflectionProvider;
import com.thoughtworks.xstream.converters.reflection.SerializableConverter;
import com.thoughtworks.xstream.core.ClassLoaderReference;
import com.thoughtworks.xstream.core.JVM;
import com.thoughtworks.xstream.core.util.SelfStreamingInstanceChecker;
import com.thoughtworks.xstream.io.HierarchicalStreamDriver;
import com.thoughtworks.xstream.mapper.Mapper;
import java.io.Serializable;


/**
 * XStream derived class dealing with the weaknesses resulting from the limits of the Google App Engine
 * class white list.
 *
 * Inspired by Chris http://stackoverflow.com/users/1043467/chris-khoo
 * and snae http://stackoverflow.com/users/930326/snae
 */
public class AppEngineXStream extends XStream {

    /**
     * Custom converter for certain serializable instances.
     */
    private class AppEngineSerializableConverter extends SerializableConverter {

        public AppEngineSerializableConverter(Mapper mapper, ReflectionProvider reflectionProvider, ClassLoaderReference classLoaderReference) {
            super(mapper, reflectionProvider, classLoaderReference);
        }


        @Override
        @SuppressWarnings("rawtypes")
        public boolean canConvert(Class type) {
            return JVM.canCreateDerivedObjectOutputStream()&&myIsSerializable(type);
        }


        @SuppressWarnings("deprecation")
        private boolean myIsSerializable(final Class<?> type) {
            // LOG.warn("myIsSerializable() "+type.getName());
            if (type!=null) {
                if (type.getName().startsWith("org.datanucleus.store.types")) {
                    return true;
                } // if
                if (Serializable.class.isAssignableFrom(type)
                        &&!type.isInterface()
                        &&(serializationMethodInvoker.supportsReadObject(type, true)||serializationMethodInvoker
                        .supportsWriteObject(type, true))) {
                    for (Object o : hierarchyFor(type)) {
                        final Class<?> clazz = (Class<?>) o;
                        if (!Serializable.class.isAssignableFrom(clazz)) {
                            return canAccess(type);
                        }
                    }
                    return true;
                } // if
            } // if
            return false;
        } // myIsSerializable()

    } // AppEngineSerializableConverter


    public AppEngineXStream(HierarchicalStreamDriver hierarchicalStreamDriver) {
        super(new PureJavaReflectionProvider(), hierarchicalStreamDriver);
    } // AppEngineXStream()


    /**
     * Small helper method to keep areas with suppressed warnings small.
     *
     * @param reflectionConverter
     */
    @SuppressWarnings("deprecation")
    private void addSelfStreamingInstanceChecker(ReflectionConverter reflectionConverter) {
        registerConverter(new SelfStreamingInstanceChecker(reflectionConverter, this), PRIORITY_NORMAL);
    } // addSelfStreamingInstanceChecker()


    @Override
    protected void setupConverters() {
        Mapper mapper = getMapper();
        ReflectionProvider reflectionProvider = getReflectionProvider();
        ClassLoader classLoader = getClassLoader();
        ClassLoaderReference reference = new ClassLoaderReference(classLoader);

        ReflectionConverter reflectionConverter = new ReflectionConverter(mapper, reflectionProvider);
        registerConverter(reflectionConverter, PRIORITY_VERY_LOW);

        registerConverter(new AppEngineSerializableConverter(mapper, reflectionProvider, reference), PRIORITY_LOW);

        registerConverter(new ExternalizableConverter(mapper, reference), PRIORITY_LOW);

        registerConverter(new NullConverter(), PRIORITY_VERY_HIGH);
        registerConverter(new IntConverter(), PRIORITY_NORMAL);
        registerConverter(new FloatConverter(), PRIORITY_NORMAL);
        registerConverter(new DoubleConverter(), PRIORITY_NORMAL);
        registerConverter(new LongConverter(), PRIORITY_NORMAL);
        registerConverter(new ShortConverter(), PRIORITY_NORMAL);
        registerConverter((Converter) new CharConverter(), PRIORITY_NORMAL);
        registerConverter(new BooleanConverter(), PRIORITY_NORMAL);
        registerConverter(new ByteConverter(), PRIORITY_NORMAL);

        registerConverter(new StringConverter(), PRIORITY_NORMAL);
        registerConverter(new StringBufferConverter(), PRIORITY_NORMAL);
        registerConverter(new DateConverter(), PRIORITY_NORMAL);
        registerConverter(new BitSetConverter(), PRIORITY_NORMAL);
        registerConverter(new URIConverter(), PRIORITY_NORMAL);
        registerConverter(new URLConverter(), PRIORITY_NORMAL);
        registerConverter(new BigIntegerConverter(), PRIORITY_NORMAL);
        registerConverter(new BigDecimalConverter(), PRIORITY_NORMAL);
        registerConverter(new ArrayConverter(mapper), PRIORITY_NORMAL);
        registerConverter(new CharArrayConverter(), PRIORITY_NORMAL);
        registerConverter(new CollectionConverter(mapper), PRIORITY_NORMAL);
        registerConverter(new MapConverter(mapper), PRIORITY_NORMAL);
        registerConverter(new TreeMapConverter(mapper), PRIORITY_NORMAL);
        registerConverter(new TreeSetConverter(mapper), PRIORITY_NORMAL);
        registerConverter(new SingletonCollectionConverter(mapper), PRIORITY_NORMAL);
        registerConverter(new SingletonMapConverter(mapper), PRIORITY_NORMAL);
        registerConverter(new PropertiesConverter(), PRIORITY_NORMAL);
        registerConverter((Converter) new EncodedByteArrayConverter(), PRIORITY_NORMAL);

        registerConverter(new FileConverter(), PRIORITY_NORMAL);
        registerConverter(new JavaClassConverter(reference), PRIORITY_NORMAL);
        registerConverter(new JavaMethodConverter(reference), PRIORITY_NORMAL);
        registerConverter(new JavaFieldConverter(reference), PRIORITY_NORMAL);
        registerConverter(new LocaleConverter(), PRIORITY_NORMAL);
        registerConverter(new GregorianCalendarConverter(), PRIORITY_NORMAL);

        // extension by
        registerConverter(new EnumConverter(), PRIORITY_NORMAL);
        registerConverter(new EnumSetConverter(mapper), PRIORITY_NORMAL);
        registerConverter(new EnumMapConverter(mapper), PRIORITY_NORMAL);

        addSelfStreamingInstanceChecker(reflectionConverter);
    } // setupConverters()

} // AppEngineXStream
