package idv.neo.utils.progressdownloader;

import android.util.Log;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.lang.ref.WeakReference;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import idv.neo.utils.download.progress.DownloadProgressResponseBody;
import idv.neo.utils.okhttp.OkHttpClientMaker;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.ResponseBody;

/**
 * Created by Neo on 2018/3/2.
 * http://www.voidcn.com/article/p-dfepcshi-bqt.html
 */

public class ProgressDownloader {
    public static final String TAG = "ProgressDownloader";
    private String mDownloadUrl;
    private OkHttpClient sClient;
    private File mDestination;
    private Call mCall;
    private DownloadProgressResponseBody.ProgressListener mProgressListener;
    private static List<WeakReference<DownloadProgressResponseBody.ProgressListener>> listeners = Collections.synchronizedList(new ArrayList<WeakReference<DownloadProgressResponseBody.ProgressListener>>());

    public ProgressDownloader(String url, File destination, boolean isdebug, int timeout,int retrycount, DownloadProgressResponseBody.ProgressListener progressListener) {
        this.mDownloadUrl = url;
        this.mDestination = destination;
        this.mProgressListener = progressListener;
        //在下载、暂停后的继续下载中可复用同一个client对象
        sClient = OkHttpClientMaker.getNewHttpClientBuilder(isdebug, timeout,retrycount).addInterceptor((Interceptor.Chain chain) -> {
            final Request request = chain.request();
            final Response originalResponse = chain.proceed(chain.request());
            return originalResponse.newBuilder()
                    .body(new DownloadProgressResponseBody(request.url(), originalResponse.body(), mProgressListener))
                    .build();
        }).build();
    }

    //每次下载需要新建新的Call对象
    private Call newCall(long startPoints) {
        final Request request = new Request.Builder()
                .url(mDownloadUrl)
                .header("RANGE", "bytes=" + startPoints + "-")//断点续传要用到的，指示下载的区间
                .build();
        return sClient.newCall(request);
    }


    // startsPoint指定开始下载的点
    public void download(final long startsPoint) {
        mCall = newCall(startsPoint);
        mCall.enqueue(new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                Log.e(TAG, "Call onFailure IOException : " + e);
                if (null != mProgressListener) {
                    mProgressListener.exceptionUpdate(call.request().url(), e);
                }
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                saveFile(response, startsPoint);
            }
        });
    }

    public void pause() {
        if (mCall != null) {
            mCall.cancel();
        }
    }

    private void saveFile(Response response, long startsPoint) {
        final ResponseBody body = response.body();
        final InputStream in = body.byteStream();
        FileChannel channelOut = null;
        // 随机访问文件，可以指定断点续传的起始位置
        RandomAccessFile randomAccessFile = null;
        try {
            randomAccessFile = new RandomAccessFile(mDestination, "rwd");
            //Chanel NIO中的用法，由于RandomAccessFile没有使用缓存策略，直接使用会使得下载速度变慢，亲测缓存下载3.3秒的文件，用普通的RandomAccessFile需要20多秒。
            channelOut = randomAccessFile.getChannel();
            // 内存映射，直接使用RandomAccessFile，是用其seek方法指定下载的起始位置，使用缓存下载，在这里指定下载位置。
            //FIXME https://fucknmb.com/2017/11/06/Android-FileChannel%E7%9A%84%E5%9D%91/
            // IllegalArgumentException: position=0 size=-1
            final MappedByteBuffer mappedBuffer = channelOut.map(FileChannel.MapMode.READ_WRITE, startsPoint, body.contentLength());
            final byte[] buffer = new byte[1024];
            int len;
            while ((len = in.read(buffer)) != -1) {
                mappedBuffer.put(buffer, 0, len);
            }
        } catch (IOException e) {
            Log.e(TAG, "save file Get Exception : " + e);
            pause();
//            if (null != mProgressListener) {
//                mProgressListener.update(response.request().url(), startsPoint, startsPoint, false, e);
//            }
            if (null != mProgressListener) {
                mProgressListener.exceptionUpdate(response.request().url(), e);
            }
        } finally {
            try {
                in.close();
                if (channelOut != null) {
                    channelOut.close();
                }
                if (randomAccessFile != null) {
                    randomAccessFile.close();
                }
            } catch (IOException e) {
                Log.d(TAG, "save finally IOException : " + e);
            }
        }
    }

    private static final DownloadProgressResponseBody.ProgressListener LISTENER = new DownloadProgressResponseBody.ProgressListener() {
        @Override
        public void onPreExecute(HttpUrl httpUrl, long contentLength) {
            if (listeners == null || listeners.size() == 0) return;
            for (int i = 0; i < listeners.size(); i++) {
                final WeakReference<DownloadProgressResponseBody.ProgressListener> listener = listeners.get(i);
                final DownloadProgressResponseBody.ProgressListener progressListener = listener.get();
                if (progressListener == null) {
                    listeners.remove(i);
                } else {
                    progressListener.onPreExecute(httpUrl, contentLength);
                }
            }
        }

        @Override
        public void update(HttpUrl httpUrl, long totalBytes, long contentLength, boolean done, Exception e) {
            if (listeners == null || listeners.size() == 0) return;
            for (int i = 0; i < listeners.size(); i++) {
                WeakReference<DownloadProgressResponseBody.ProgressListener> listener = listeners.get(i);
                DownloadProgressResponseBody.ProgressListener progressListener = listener.get();
                if (progressListener == null) {
                    listeners.remove(i);
                } else {
                    progressListener.update(httpUrl, totalBytes, contentLength, done, e);
                }
            }
        }

        @Override
        public void exceptionUpdate(HttpUrl httpUrl, Exception e) {

        }
    };

    public static void addProgressListener(DownloadProgressResponseBody.ProgressListener progressListener) {
        if (progressListener == null) {
            return;
        }

        if (findProgressListener(progressListener) == null) {
            listeners.add(new WeakReference<>(progressListener));
        }
    }

    public static void removeProgressListener(DownloadProgressResponseBody.ProgressListener progressListener) {
        if (progressListener == null) {
            return;
        }
        final WeakReference<DownloadProgressResponseBody.ProgressListener> listener = findProgressListener(progressListener);
        if (listener != null) {
            listeners.remove(listener);
        }
    }

    private static WeakReference<DownloadProgressResponseBody.ProgressListener> findProgressListener(DownloadProgressResponseBody.ProgressListener listener) {
        if (listener == null) return null;
        if (listeners == null || listeners.size() == 0) return null;

        for (int i = 0; i < listeners.size(); i++) {
            WeakReference<DownloadProgressResponseBody.ProgressListener> progressListener = listeners.get(i);
            if (progressListener.get() == listener) {
                return progressListener;
            }
        }
        return null;
    }
}
