/*
 * MIT License
 *
 * Copyright (c) ReformCloud-Team
 * Copyright (c) contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package systems.reformcloud.reformcloud2.shared.network.packet;

import org.jetbrains.annotations.NotNull;
import systems.reformcloud.reformcloud2.executor.api.network.packet.Packet;
import systems.reformcloud.reformcloud2.executor.api.network.packet.PacketProvider;
import systems.reformcloud.reformcloud2.executor.api.network.packet.exception.PacketAlreadyRegisteredException;

import java.util.Collection;
import java.util.EmptyStackException;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

public class DefaultPacketProvider implements PacketProvider {

    private final Map<Integer, Class<? extends Packet>> packetClasses = new ConcurrentHashMap<>();

    @Override
    public void registerPacket(@NotNull Class<? extends Packet> packetClass) throws PacketAlreadyRegisteredException {
        this.registerPacket(this.newInstanceFromClass(packetClass));
    }

    @Override
    public void registerPacket(@NotNull Packet packet) throws PacketAlreadyRegisteredException {
        Optional<Packet> packetOptional = this.getPacketById(packet.getId());
        if (packetOptional.isPresent()) {
            throw new PacketAlreadyRegisteredException(packet.getClass().getName(), packetOptional.get().getClass().getName(), packet.getId());
        }

        this.packetClasses.put(packet.getId(), packet.getClass());
    }

    @Override
    public void registerPackets(@NotNull Collection<Class<? extends Packet>> packetClasses) throws PacketAlreadyRegisteredException {
        for (Class<? extends Packet> packetClass : packetClasses) {
            this.registerPacket(packetClass);
        }
    }

    @Override
    public void registerPacket(@NotNull Collection<Packet> packets) throws PacketAlreadyRegisteredException {
        for (Packet packet : packets) {
            this.registerPacket(packet);
        }
    }

    @Override
    public void unregisterPacket(int id) {
        this.packetClasses.remove(id);
    }

    @NotNull
    @Override
    public Optional<Packet> getPacketById(int id) {
        Class<? extends Packet> packetClass = this.packetClasses.get(id);
        return packetClass != null ? Optional.of(this.newInstanceFromClass(packetClass)) : Optional.empty();
    }

    @Override
    public void clearRegisteredPackets() {
        this.packetClasses.clear();
    }

    private @NotNull Packet newInstanceFromClass(@NotNull Class<? extends Packet> packetClass) {
        try {
            return packetClass.getDeclaredConstructor().newInstance();
        } catch (NoSuchMethodException exception) {
            System.err.println("Unable to load packet " + packetClass.getName() + " -> NoArgsConstructor is missing");
            System.err.println("Please report this or fix your application before running the system again");
            throw new EmptyStackException();
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }
}
