package systems.dennis.shared.service;

import lombok.SneakyThrows;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.springframework.core.io.UrlResource;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.annotations.NeverNullResponse;
import systems.dennis.shared.annotations.security.ISecurityUtils;
import systems.dennis.shared.beans.AbstractDataFilterProvider;
import systems.dennis.shared.beans.AbstractEditDeleteHistoryBean;
import systems.dennis.shared.beans.IdValidator;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.items.magic.MagicRequest;
import systems.dennis.shared.entity.AbstractEntity;
import systems.dennis.shared.entity.KeyValue;
import systems.dennis.shared.exceptions.*;
import systems.dennis.shared.form.AbstractForm;
import systems.dennis.shared.model.IDPresenter;
import systems.dennis.shared.pojo_view.list.PojoListView;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.repository.AbstractRepository;
import systems.dennis.shared.utils.ErrorSupplier;
import systems.dennis.shared.utils.Supplier;
import systems.dennis.shared.utils.bean_copier.BeanCopier;
import systems.dennis.shared.utils.bean_copier.ObjectDefinition;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.util.*;

import static systems.dennis.shared.annotations.DeleteStrategy.DELETE_STRATEGY_PROPERTY;
import static systems.dennis.shared.controller.forms.Serviceable.findDeclaredClass;
import static systems.dennis.shared.controller.forms.Serviceable.log;
import static systems.dennis.shared.utils.bean_copier.BeanCopier.findField;

/**
 * Implementation should implement methods which are necessary for web service to work properly
 * In this implementation all methods, like add, edit, delete or list are in the same interface. Form more complex functionality, it can make sence to move all this logical functions to the separate service
 *
 * @param <DB_TYPE> A class of the entity to be saved. Should be persistant entity, in the best an implementation of {@link AbstractForm}
 */
public interface AbstractService<DB_TYPE extends IDPresenter<ID_TYPE>, ID_TYPE extends Serializable> extends DeleteObject<DB_TYPE, ID_TYPE> {
    /**
     * A simple fetch all implementation
     *
     * @return list of all records of T
     */
    List<DB_TYPE> find();

    /**
     * Updates an object in DB
     *
     * @param object - object to be updated, with ID
     * @return Updated T object
     * @throws ItemNotUserException                Exceptions on thrown when trying to update and object by user, not owner of the object
     * @throws ItemNotFoundException               When object cannot be found by id of the #object, this exception is thrown
     * @throws UnmodifiedItemSaveAttemptException  Exception says, that there are no changes in the object which should be updated
     * @throws ItemDoesNotContainsIdValueException Exception is throw when Objects has null ID
     */
    DB_TYPE edit(DB_TYPE object) throws ItemNotUserException, ItemNotFoundException, UnmodifiedItemSaveAttemptException, ItemDoesNotContainsIdValueException;

    KeyValue editField(ID_TYPE id, KeyValue keyValue) throws ItemNotUserException, ItemNotFoundException, UnmodifiedItemSaveAttemptException, ItemDoesNotContainsIdValueException, IllegalAccessException, InvocationTargetException;

    /**
     * Removes object from DB
     *
     * @param id - id of the object to be removed
     * @throws ItemNotUserException  Exceptions are thrown on trying to update and object by user, not owner of the object
     * @throws ItemNotFoundException When object cannot be found by id of the #object, this exception is thrown
     */
    default void delete(ID_TYPE id) throws ItemNotUserException, ItemNotFoundException {
        var object = findById(id).orElseThrow(() -> ItemNotFoundException.fromId(id));
        preDelete(object);
        object = delete(object, this);
        afterDelete(object);
        var deleteHistory = getContext().getBean(AbstractEditDeleteHistoryBean.class);
        if (deleteHistory.isEnabled()) {
            deleteHistory.delete(id, object);
        }
    }

    /**
     * Removes objects from DB
     *
     * @param ids - ids of objects to be removed
     * @throws ItemNotUserException  Exception is thrown when trying to update an object from a user, not owner of the object
     * @throws ItemNotFoundException When object cannot be found by id of the #object, this exception is thrown
     */
    void deleteItems(List<ID_TYPE> ids) throws ItemNotUserException, ItemNotFoundException;

    /**
     * Before object is saved in DB this method is called
     *
     * @param object   - an object variant to be updated
     * @param original - an original object from DB
     * @return - a final object to be updated
     * @throws ItemNotFoundException              When object cannot be found by id of the #object, this exception is thrown
     * @throws UnmodifiedItemSaveAttemptException Exception says, that there are no changes in the object which should be updated
     */
    default DB_TYPE preEdit(DB_TYPE object, DB_TYPE original) throws UnmodifiedItemSaveAttemptException, ItemNotFoundException {
        return object;
    }



    default boolean exists(AbstractDataFilter<DB_TYPE> filter) {
        return count(filter) > 0;
    }


    default void saveVersionIfRequired(DB_TYPE original, DB_TYPE object) {

        var historyVersion = getContext().getBean(AbstractEditDeleteHistoryBean.class);

        try {
            if (historyVersion.isEnabled()) {
                historyVersion.edit(original, object);
            }


        } catch (Exception e) {
            getLogger().error("Cannot save original version for object", e);
        }
    }

    /**
     * Before object is added this method is called
     *
     * @param object to be saved
     * @return a modified object before saving
     * @throws ItemForAddContainsIdException No id should be in edit object
     */
    default DB_TYPE preAdd(DB_TYPE object) throws ItemForAddContainsIdException {
        return object;
    }

    WebContext.LocalWebContext getContext();


    @SneakyThrows
    default Page<DB_TYPE> search(String field, String subtype, String value, int page, Integer size, Serializable[] additionalIds) {
        var model = getModel().getConstructor().newInstance();
        return getRepository().filteredData(getSearchRequestAbstractDataFilter(field, value, updateIds(additionalIds)), PageRequest.of(page, size, model.defaultSearchOrderField().getValue(), model.defaultSearchOrderField().getKey()));
    }

    default AbstractDataFilter<DB_TYPE> getSearchRequestAbstractDataFilter(String field, String value, ID_TYPE[] additionalIds) {
        AbstractDataFilter<DB_TYPE> spec = getFilterImpl().contains(field, value).setInsensitive(true);
        AbstractDataFilter<DB_TYPE> additionalAbstractDataFilter = getAdditionalSpecification();
        if (additionalAbstractDataFilter != null) {
            spec = spec.and(additionalAbstractDataFilter);
        }

        return spec;
    }

    default AbstractDataFilter<DB_TYPE> getFilterImpl(){
        return getContext().getBean(AbstractDataFilterProvider.class).get();
    }

    default ID_TYPE[] updateIds(Object[] ids) {
        return (ID_TYPE[]) ids;
    }

    default ID_TYPE updateId(Object id) {
        return (ID_TYPE) id;
    }

    /**
     * What to do with object after it is stored in DB
     *
     * @param object Saved object with ID
     * @return modified object
     */
    default DB_TYPE afterAdd(DB_TYPE object) {
        return object;
    }

    /**
     * What to do with object before it is deleted from DB
     *
     * @param object deleted object with ID
     * @return deleted object
     */
    default DB_TYPE preDelete(DB_TYPE object) {
        return object;
    }

    /**
     * What to do with object after it is deleted from DB
     *
     * @param object deleted object with ID
     * @return deleted object
     */
    default DB_TYPE afterDelete(DB_TYPE object) {
        return object;
    }


    Logger getLogger();

    /**
     * Verifies whether object exist or not ( normally, by Id)
     *
     * @param object Object to check
     * @return true if object exists, false otherwise
     */
    boolean exists(DB_TYPE object);

    DB_TYPE save(DB_TYPE form);

    @SneakyThrows
    default <F extends DB_TYPE> F findByIdOrThrow(ID_TYPE id) {
        DB_TYPE res = findById(id).orElseThrow(() -> ItemNotFoundException.fromId(id));
        return (F) res;
    }

    default <T> T findById(ID_TYPE id, Supplier<T> orElse) {
        var res = findById(id);
        if (res.isEmpty()) {
            return orElse.onNull(id);
        }
        return (T) res.get();
    }

    default AbstractDataFilter<DB_TYPE> getSelfCreatedItems(Serializable currentUser) {
        return getSelfCreatedItems(currentUser, true);
    }

    @NeverNullResponse(on = "ignoreOnAdmin = false")
    default AbstractDataFilter<DB_TYPE> getSelfCreatedItems(Serializable currentUser, boolean ignoreOnAdmin) {
        return getSelfCreatedItemsQuery(currentUser, ignoreOnAdmin);
    }


    default AbstractDataFilter<DB_TYPE> getSelfCreatedItemsQuery(Serializable currentUser) {
        return getSelfCreatedItemsQuery(currentUser, true);
    }

    @NeverNullResponse(on = "ignoreOnAdmin = false")
    default AbstractDataFilter<DB_TYPE> getSelfCreatedItemsQuery(Serializable currentUser, boolean ignoreOnAdmin) {
        if (ignoreOnAdmin && getContext().getBean(ISecurityUtils.class).isAdmin()) {
            return getFilterImpl().notNull(IDPresenter.ID_FIELD);
        }
        return getFilterImpl().eq("userDataId", currentUser);
    }


    default Optional<DB_TYPE> findByIdClone(ID_TYPE id) {
        var el = findById(id);

        if (el.isPresent()) {
            return Optional.of(getContext().getBean(BeanCopier.class).clone(el.get()));
        } else {
            return el;
        }

    }

    default boolean isIdSet(ID_TYPE id) {
        return getContext().getBean(IdValidator.class).isIdSet(id);
    }

    default Optional<DB_TYPE> findById(Serializable idSerializable) {
        var id = updateId(idSerializable);
        if (!isIdSet(id)) {
            throw new StandardException(String.valueOf(id), "id.was.not_set.ty_find_by_id");
        }
        AbstractDataFilter<DB_TYPE> AbstractDataFilter = getAdditionalSpecification();

        Optional<DB_TYPE> element;
        if (AbstractDataFilter != null) {
            element = getRepository().filteredOne(AbstractDataFilter.and(getFilterImpl().id(id)));
        } else {
            element = getRepository().filteredOne(getFilterImpl().id(id));
        }

        getContext().getBean(AbstractEditDeleteHistoryBean.class).throwIfDeleted(id, getModel());
        return element;
    }

    default <T> T findById(ID_TYPE id, Exception els) {
        return findById(id, new ErrorSupplier<>(els));
    }

    default <T extends AbstractEntity> Class<T> getModel() {
        var res = findDeclaredClass(getClass(), getClass().getAnnotation(DataRetrieverDescription.class));

        return (Class<T>) res.model();
    }

    default <T extends AbstractForm> Class<T> getForm() {
        var res = findDeclaredClass(getClass(), getClass().getAnnotation(DataRetrieverDescription.class));

        return (Class<T>) res.form();
    }

    /**
     * Returns all records from db having id more than from (or ignoring if null) and limited of limit
     * Default method to get all values should be used with care or better not to be used !
     *
     * @param limit - max
     * @param page  - a page to select
     * @return List of res
     */
    default Page<DB_TYPE> find(Integer limit, Integer page) {
        return find(null, limit, page);
    }


    default Page<DB_TYPE> find(AbstractDataFilter<DB_TYPE> searchAbstractDataFilter, Integer limit, Integer page) {

        PageRequest request = null;


        if (limit == null || limit == -1) {
            limit = 200;
        }

        request = PageRequest.of(Objects.requireNonNullElse(page, 0), limit);


        if (searchAbstractDataFilter != null) {
            return getRepository().filteredData(searchAbstractDataFilter, request);
        } else {
            return getRepository().findAll(request);
        }
    }

    <F extends AbstractRepository<DB_TYPE, ID_TYPE>> F getRepository();

    default Page<DB_TYPE> search(MagicRequest request) {
        Page<DB_TYPE> data;
        if (request.getQuery().isEmpty() && request.getCases().isEmpty() && getAdditionalSpecification() == null) {
            data = getRepository().findAll(PageRequest.of(request.getPage(), request.getLimit(), createFromRequest(request)));

            return data;
        } else {
            AbstractDataFilter<DB_TYPE> finalAbstractDataFilter = prepareAbstractDataFilter(request);
            data = getRepository().filteredData(finalAbstractDataFilter,
                    PageRequest.of(request.getPage(), request.getLimit(), createFromRequest(request)));
            return data;
        }

    }


    default AbstractDataFilter<DB_TYPE> prepareAbstractDataFilter(MagicRequest request) {
        AbstractDataFilter<DB_TYPE> modifyQuery = getAdditionalSpecification();
        for (var query : request.getQuery()) {
            modifyQuery.and(modifyQuery(query.toQuery(getContext())));
        }
        for (var query : request.getCases()) {
            modifyQuery.and(modifyQuery(query));
        }


        return modifyQuery;

    }

    default Sort createFromRequest(MagicRequest request) {
        if (request.getSort() == null || request.getSort().isEmpty()) {
            return Sort.unsorted();
        }

        List<Sort> sorts = new ArrayList<>();


        request.getSort().stream().filter(Objects::nonNull).forEach(x -> sorts.add(
                Sort.by(x.getDesc() ? Sort.Direction.DESC : Sort.Direction.ASC, getFieldOrder(x.getField()))));
        if (sorts.size() > 1) {

            var firstSort = sorts.get(0);
            for (int i = 1; i < sorts.size(); i++) {

                firstSort = firstSort.and(sorts.get(i));
            }

            return firstSort;
        } else {
            return sorts.get(0);
        }
    }

    default String getFieldOrder(String field) {
        try {
            var sortField = findField(field, getModel());

            if (AbstractForm.class.isAssignableFrom(sortField.getType())) {
                return sortField.getName() + "." + ((AbstractEntity) sortField.getType().getConstructor().newInstance()).defaultSearchOrderField().getKey();
            }
            return field;
        } catch (Exception e) {
            return field;
        }

    }

    default long count() {
        AbstractDataFilter<DB_TYPE> spec = getAdditionalSpecification();
        return getRepository().filteredCount(spec);
    }

    default boolean isEmpty(AbstractDataFilter AbstractDataFilter) {
        return count(AbstractDataFilter) == 0;
    }


    default boolean isEmpty() {
        return count() == 0;
    }

    default long count(AbstractDataFilter<DB_TYPE> spec) {
        AbstractDataFilter<DB_TYPE> customAbstractDataFilter = getAdditionalSpecification();

        if (customAbstractDataFilter != null) {
            return getRepository().filteredCount(customAbstractDataFilter.and(spec));
        }

        return getRepository().filteredCount(spec);
    }

    @SneakyThrows
    default UrlResource download(MagicRequest request) {
        Page<DB_TYPE> data;
        var path = getContext().getEnv("global.download.path", "./");
        AbstractDataFilter<DB_TYPE> customAbstractDataFilter = getAdditionalSpecification();

        if (request.getQuery().isEmpty()) {
            if (customAbstractDataFilter != null) {
                data = getRepository().filteredData(customAbstractDataFilter, PageRequest.of(0, 100000, createFromRequest(request)));
            } else {
                data = getRepository().findAll(PageRequest.of(0, 100000, createFromRequest(request)));
            }
        } else {

            AbstractDataFilter<DB_TYPE> filter = getFilterImpl().empty();

            for (var query : request.getQuery()) {
                filter.and(query.toQuery(getContext()));
            }

            if (customAbstractDataFilter != null) {
                data = getRepository().filteredData(filter.and(customAbstractDataFilter),
                        PageRequest.of(0, 100000, createFromRequest(request)));
            } else {
                data = getRepository().filteredData(filter, (PageRequest.of(0, 100000, createFromRequest(request))));
            }
        }

        List<Map<String, Object>> res = new ArrayList<>();
        var bc = getContext().getBean(BeanCopier.class);

        for (DB_TYPE entity : data.getContent()) {
            var form = bc.copy(entity, getForm());
            Map<String, Object> values = BeanCopier.values(form, entity, getContext());
            res.add(values);
        }

        if (res.size() == 0) {
            res.add(Collections.singletonMap("res", " ---- NO DATA ---"));
        }


        Workbook workbook = null;

        String fileName = path + UUID.randomUUID().toString() + ".xls";

        if (fileName.endsWith("xlsx")) {
            workbook = new XSSFWorkbook();
        } else {
            workbook = new HSSFWorkbook();
        }

        Sheet sheet = workbook.createSheet("data");

        var headRow = 1;
        var cln = 0;
        Row row = sheet.createRow(headRow);
        for (String el : res.get(0).keySet()) {
            if ("action".equalsIgnoreCase(el)) {
                continue;
            }

            Cell cell0 = row.createCell(cln);
            cell0.setCellValue(el);
            cln++;

        }

        var dataRow = 2;
        var datacln = 0;

        for (Map<String, Object> element : res) {
            Row datarow = sheet.createRow(dataRow);


            for (Object el : element.values()) {
                Cell cell = datarow.createCell(datacln);
                if (el instanceof ObjectDefinition) {
                    el = ((ObjectDefinition) el).getValue();
                }

                if (el == null) {
                    el = "";
                }

                if (el.toString().length() > 5000) {
                    el = String.valueOf(el).substring(0, 5000);
                }

                cell.setCellValue(String.valueOf(el));
                datacln++;
            }
            datacln = 0;
            dataRow++;

        }

        FileOutputStream fos = new FileOutputStream(fileName);
        workbook.write(fos);
        fos.close();
        System.out.println(fileName + " written successfully");

        return new UrlResource(new File(fileName).toURI());

    }


    default String getItemFavoriteType() {
        if (getForm().getAnnotation(PojoListView.class) == null) {
            return getForm().getSimpleName();
        }

        return getForm().getAnnotation(PojoListView.class).favoriteType();
    }

    default UrlResource getInterNalResource(String resource) {

        var path = getContext().getEnv("global.download.path", "./");
        String fileName = path + resource;
        try {
            return new UrlResource(new File(fileName).toURI());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Since version 3.0.5 will never produce null
     *
     * @return not deleted query on notEmptyId query
     */
    default @NeverNullResponse AbstractDataFilter<DB_TYPE> getAdditionalSpecification() {
        return getNotDeletedQuery();
    }

    default void preSearch(String what, String type) {
    }

    default boolean isPublicSearchEnabled() {
        return false;
    }

    default void checkMy(DB_TYPE object) {
        getContext().getBean(ISecurityUtils.class).isMy(object);
    }

    default @NeverNullResponse AbstractDataFilter<DB_TYPE> modifyQuery(AbstractDataFilter<DB_TYPE> query) {
        return query;
    }

    default Boolean isObjectDeleted(DB_TYPE object) {
        int strategy = getDeleteStrategy(getClass());
        if (strategy == DELETE_STRATEGY_PROPERTY) {
            if (Objects.isNull(object.getHidden())) {
                return false;
            }
            return object.getHidden();
        }
        try {
            findById(object.getId());
            return false;
        } catch (ItemWasDeletedException e) {
            return true;
        }
    }

    default boolean getByIdAndUserDataId(ID_TYPE id, Serializable currentUser) {
        try {
            if (getContext().getBean(ISecurityUtils.class).isAdmin()) {
                return true;
            }
            AbstractDataFilter<DB_TYPE> AbstractDataFilter = getFilterImpl().id(id).and(getFilterImpl().ofUser(getModel(), currentUser));
            return count(AbstractDataFilter) > 0;
        } catch (Exception e) {
            return false;
        }
    }
}
