package systems.dennis.shared.config;

import jakarta.servlet.http.HttpServletRequest;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import systems.dennis.shared.annotations.security.ISecurityUtils;
import systems.dennis.shared.beans.AbstractDataFilterProvider;
import systems.dennis.shared.beans.AppSettingsResolver;
import systems.dennis.shared.beans.IdToAuthorizationIdBean;
import systems.dennis.shared.beans.LocaleBean;
import systems.dennis.shared.entity.TokenData;
import systems.dennis.shared.exceptions.SettingKeyNotFoundException;
import systems.dennis.shared.model.IDPresenter;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.utils.bean_copier.BeanCopier;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

@AllArgsConstructor
@Data
@Service
@Slf4j
public class WebContext {
    public static final Map<String, Object> data = new HashMap<>();
    @Autowired
    private HttpServletRequest request;

    @Autowired
    private MessageResourceSource messages;

    @Autowired
    private ApplicationContext applicationContext;

    @Autowired
    private Environment environment;

    @Autowired
    private LocaleBean localeBean;


    @Autowired
    private final IdToAuthorizationIdBean id2id;
    private static AppSettingsResolver resolver;
    private static AppSettingsResolver EMPTY_RESOLVER = new AppSettingsResolver() {
        @Override
        public boolean hasSetting(String setting) {
            return false;
        }

        @Override
        public String getEnv(String key) {
            return null;
        }

        @Override
        public String getEnv(String key, Object def) {
            return null;
        }
    };


    /**
     * @deprecated as uncontrollable data storing can provide to the leak
     * @param what to save key
     * @param cache object to save
     * @return cached object
     */
    @Deprecated

    public Object toCache(String what, Object cache) {
        return data.put(what, cache);
    }

    @Deprecated

    /**
     * @deprecated as uncontrollable data storing can provide to the leak
     * @param what to save key
     * @param def what to return by default
     *
     * @return cached object or default value
     *
     */
    public <T> T fromCache(String what, OnNull<T> def) {
        var res = data.get(what);
        if (res == null) return def.def();
        return (T) res;
    }


    static AppSettingsResolver getResolver(WebContext context){
        if (resolver == null || Objects.equals(resolver , EMPTY_RESOLVER )){
            return null;
        }
        resolver = context.getBean(AppSettingsResolver.class, EMPTY_RESOLVER);
        return  resolver;
    }

    @SneakyThrows
    public <T> T getEnv(String key) {
        var resolver = getBean(AppSettingsResolver.class, null);
        if (resolver == null){
            return (T) environment.getProperty(key);
        }

        return (T) resolver.getEnv(key);
    }

    public <T> T getEnv(String key, T def) {
        try {
            T env = getEnv(key);
            if (Objects.isNull(env)) {
                return def;
            }
            return env;

        } catch(SettingKeyNotFoundException e) {
            return def;
        }
    }

    public TokenData getToken() {
        return getBean(ISecurityUtils.class).getToken();
    }



    private String getMessageTranslation(String header, String lang) {
        messages.resolveCode(header, new Locale(localeBean.transform(lang, this)));
        final String message = messages.getMessage(
                header.toLowerCase(),
                new Object[]{},
                "?_" + header + "_?",
                new Locale(localeBean.transform(lang, this))); //--> en_EN -> en_US

        if (message == null || (message.startsWith("?_") && message.endsWith("_?"))) {
            log.warn(header + " not found !");
        }
        return message;

    }

    private String getMessageTranslation(String header, String lang, Object... params) {
        return getMessageTranslation(header, new Locale(localeBean.transform(lang, this)), params);
    }

    private String getMessageTranslation(String header, Locale locale, Object... params) {
        final String message = messages.getMessage(
                header.toLowerCase(),
                params,
                "?_" + header + "_?",
                locale);

        if (message == null || (message.startsWith("?_") && message.endsWith("_?"))) {
            log.warn(header + " not found !");
        }
        return message;

    }

    public <T extends Serializable> T currentUser() {
        return (T) getBean(ISecurityUtils.class).getUserDataId();
    }

    private ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    public <T> T getBean(Class<T> value) {
        return getApplicationContext().getBean(value);
    }
    public <T> T getBean(Class<T> value,T def) {
        try {
            return getApplicationContext().getBean(value);
        } catch (BeansException e){
            return def;
        }
    }


    public static class LocalWebContext {
        private final WebContext webContext;
        private String scope;

        public HttpServletRequest getRequest() {
            return webContext.getRequest();
        }

        public String httpParam(String param) {
            return webContext.getRequest().getParameter(param);
        }

        public WebContext getWebContext() {
            return webContext;
        }

        public String getMessageTranslation(String header, String lang) {
            return webContext.getMessageTranslation(
                    scope + "."
                            + header.toLowerCase(), lang);
        }

        public AbstractDataFilter<?> getDataFilterProvider(){
            return  this.getBean(AbstractDataFilterProvider.class).get();
        }

        public String getMessageTranslation(String header, String lang, Class parentScope) {
            return webContext.getMessageTranslation(
                    parentScope.getName() + "."
                            + header.toLowerCase(), lang);
        }

        public String getMessageTranslation(String header, String lang, Object... params) {
            return webContext.getMessageTranslation(
                    scope + "."
                            + header.toLowerCase(), lang, params);
        }

        private LocalWebContext(String scope, WebContext context) {
            this.scope = scope;
            this.webContext = context;

        }

        public static LocalWebContext of(String scope, WebContext context) {
            return new LocalWebContext(scope, context);

        }

        @SneakyThrows
        public <T>T getEnv(String key) {
            return webContext.getEnv(key);
        }

        public <T> T getEnv(String key, T def) {
            return webContext.getEnv(key, def);
        }

        public <T extends Serializable>T getCurrentUser() {
            return (T) getWebContext().id2id.idToAuthorizationId(getBean(ISecurityUtils.class).getUserDataId());
        }


        public <T> T transform(Object item, Class<? extends T> testPlanViewClass) {
            return getBean(BeanCopier.class).copy(item, testPlanViewClass);
        }

        public String getScoped(String header) {
            return scope + "." + header;
        }

        public <T> T getBean(Class<T> bean) {
            return webContext.getApplicationContext().getBean(bean);
        }

    }

}
