package systems.dennis.shared.beans;

import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.CommandLineRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.bind.annotation.RequestMapping;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.annotations.WebFormsSupport;
import systems.dennis.shared.annotations.entity.CreatedBy;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.SearchEntityApi;
import systems.dennis.shared.controller.SearcherInfo;
import systems.dennis.shared.exceptions.ApplicationStartFailedException;
import systems.dennis.shared.utils.bean_copier.BeanCopier;

import java.util.Objects;

@Configuration
@Slf4j
public class OnApplicationStart {


    @Autowired(required = false)
    @Bean
    public CommandLineRunner init(WebContext context, ApplicationContext applicationContext, OnAppStart onAppStart) {
        return args -> {
            //since 3.1.2 No more forced elements to start only what is defined by application
            //since 3.3.0 here is added scanning services for SearchEntityApi

            try {
                var controllers = applicationContext.getBeansWithAnnotation(WebFormsSupport.class);

                for (var controller : controllers.values()) {
                    var targetClass =  AopUtils.getTargetClass(controller);
                    WebFormsSupport support = targetClass.getAnnotation(WebFormsSupport.class);
                    if (!support.allowEntitySearch()) {continue;}

                    var name = getSearchName(targetClass);
                    var fieldName =  getFieldName(targetClass);
                    if (fieldName != null && name != null) {
                        SearchEntityApi.registerSearch(name, new SearcherInfo(fieldName, support.value()));
                    }
                    //since version 3.3.1 @CreatedBy annotation is now replacing UserAssignableEntity
                    //to avoid multiple fetching annotation data /searching for annotation a map is now created
                    //also check that double @CreatedBy by model is not defined
                    checkForCreatedAnnotations(support, targetClass, context);

                }
            } catch (Exception e) {
                log.info("error during automatic SearchEntity filling: ", e);
            }

            if (onAppStart != null) {
                onAppStart.onAppRun(context);
            }
        };
    }

    private void checkForCreatedAnnotations(WebFormsSupport support, Class<?> targetClass, WebContext context) {
        var model = support.value().getAnnotation(DataRetrieverDescription.class).model();
        var createdFields = BeanCopier.findAnnotatedFields(model, CreatedBy.class, BeanCopier.TYPE_ALL_FIELDS);
        if (createdFields.size() > 1){
            throw new ApplicationStartFailedException(model, "global.few.created_annotations");
        }

        if (createdFields.size() == 1){
            systems.dennis.shared.utils.ApplicationContext.CREATED_FIELDS_MAP.put(model, createdFields.get(0));
        }
    }


    public static String getFieldName(Class<?> controller) {
        WebFormsSupport support = controller.getAnnotation(WebFormsSupport.class);
        if (!Objects.equals(WebFormsSupport.DEFAULT_NAME, support.fieldName() )){
            return support.fieldName();
        }

        var model = support.value().getAnnotation(DataRetrieverDescription.class).model();

        for (var field :model.getDeclaredFields()){
            if (!java.lang.reflect.Modifier.isStatic(field.getModifiers())){
                if (field.getType() == String.class){
                    return  field.getName();
                }
            }

        }



        return null;
    }

    public static String getSearchName(Class<?> controller) {

        WebFormsSupport support = controller.getAnnotation(WebFormsSupport.class);
        var name = support.searchName();
        if (Objects.equals(name, WebFormsSupport.DEFAULT_NAME)) {
            name = retrieveNameFromPath(controller);
        }

        return name;
    }

    public static String retrieveNameFromPath(Class<?> controller) {
        var mapping = controller.getAnnotation(RequestMapping.class).value();

        if (mapping.length == 0) {
            return null;
        }

        var mappingControllerPath = mapping[0];

        if (mappingControllerPath.endsWith("/")) {
            mappingControllerPath = mappingControllerPath.substring(0, mappingControllerPath.length() -1);
        }
        var name = mappingControllerPath.split("/");
        return name[name.length - 1];


    }


}
