package systems.dennis.shared.service;

import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.transaction.Transactional;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.Session;
import org.slf4j.Logger;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.annotations.security.ISecurityUtils;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.forms.QueryObject;
import systems.dennis.shared.entity.KeyValue;
import systems.dennis.shared.exceptions.ItemDoesNotContainsIdValueException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.exceptions.ItemNotUserException;
import systems.dennis.shared.exceptions.UnmodifiedItemSaveAttemptException;
import systems.dennis.shared.model.IDPresenter;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.repository.AbstractRepository;
import systems.dennis.shared.utils.ApplicationContext;
import systems.dennis.shared.utils.bean_copier.BeanCopier;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

@Slf4j
public abstract class AbstractPaginationService<DB_TYPE extends IDPresenter<ID_TYPE>, ID_TYPE extends Serializable>
        extends ApplicationContext
        implements AbstractService<DB_TYPE, ID_TYPE> {

    @PersistenceContext
    private EntityManager entityManager;

    public AbstractPaginationService(WebContext holder) {
        super(holder);
    }


    @Override
    public Logger getLogger() {
        return log;
    }

    @Override
    public boolean exists(DB_TYPE object) {
        return getRepository().existsById(object.getId());
    }

    @SneakyThrows
    @Override
    public DB_TYPE save(DB_TYPE model) {

        if (model.getId() != null && !model.isIdSet()) {
            model.setId(null);
        }

        if (model.getId() != null) {
            Session session = entityManager.unwrap(Session.class);
            session.evict(model);
            return edit(getBean(BeanCopier.class).clone(model));
        }

        preAdd(model);
        assignUser(model);

        var res = getRepository().save(model);

        afterAdd(res);

        return res;
    }

    public void preFetchOriginal(DB_TYPE form) {

    }

    public AbstractDataFilter<DB_TYPE> getAdditionalCases(QueryObject<String> parameters) {
        return getFilterImpl().empty();
    }

    @Override
    public List<DB_TYPE> find() {
        return find(false);
    }

    public List<DB_TYPE> find(Boolean ignoreAdditionalSpecification) {
        List<DB_TYPE> result = new ArrayList<>();
        if (ignoreAdditionalSpecification) {
            getRepository().findAll().forEach(result::add);
        } else {
            getRepository().filteredData(getAdditionalSpecification()).forEach(result::add);
        }
        return result;
    }

    @Override
    public DB_TYPE edit(DB_TYPE model) throws ItemNotUserException, ItemNotFoundException, UnmodifiedItemSaveAttemptException, ItemDoesNotContainsIdValueException {
        ID_TYPE editedId = model.getId();
        DB_TYPE cloneOfOriginal = findByIdClone(editedId).orElseThrow(() -> ItemNotFoundException.fromId(editedId));
        //Here need to be cloned to avoid Hibernate to take original object from cache

        if (Objects.equals(model, cloneOfOriginal)) {
            throw new UnmodifiedItemSaveAttemptException("Object not changed");
        }

        var copier = getContext().getBean(BeanCopier.class);
        copier.copyTransientFields(model, cloneOfOriginal);

        assignUser(model, cloneOfOriginal);

        model = preEdit(model, cloneOfOriginal); // change to retrieve data

        var res = getRepository().save(model);
        saveVersionIfRequired(cloneOfOriginal, model);

        afterEdit(res, cloneOfOriginal); // afterEdit

        return res;
    }

    public void afterEdit(DB_TYPE object, DB_TYPE original) {

    }


    @Override
    public KeyValue editField(ID_TYPE id, KeyValue keyValue)
            throws ItemNotUserException, ItemNotFoundException, UnmodifiedItemSaveAttemptException, ItemDoesNotContainsIdValueException, IllegalAccessException, InvocationTargetException {

        DB_TYPE original = getRepository().findById(id).orElseThrow(() -> ItemNotFoundException.fromId(id));
        var cloneOfOriginal = getContext().getBean(BeanCopier.class).clone(original);
        var field = BeanCopier.findField(keyValue.getKey(), cloneOfOriginal.getClass());

        BeanCopier.setFieldValue(cloneOfOriginal, field, keyValue.getValue());
        cloneOfOriginal = edit(cloneOfOriginal);

        var valueFromModel = BeanCopier.readValue(cloneOfOriginal, keyValue.getKey());

        return new KeyValue(keyValue.getKey(), valueFromModel);
    }


    @Override
    @Transactional
    public void deleteItems(List<ID_TYPE> ids) throws ItemNotUserException, ItemNotFoundException {
        ids.forEach(this::delete);
    }



    @Override
    public <F extends AbstractRepository<DB_TYPE, ID_TYPE>> F getRepository() {
        return (F) getContext().getBean(getClass().getAnnotation(DataRetrieverDescription.class).repo());
    }

    @SuppressWarnings("Unchecked")
    public <T extends ISecurityUtils> T getUtils() {
        return (T) getContext().getBean(ISecurityUtils.class);
    }

    public boolean isIdSet(Long id) {
        return id != null && id > 0;
    }
}
