package systems.dennis.shared.servers.service;

import org.springframework.stereotype.Service;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.servers.exception.ServerException;

import systems.dennis.shared.servers.form.ServerConfigForm;
import systems.dennis.shared.servers.model.ServerConfig;
import systems.dennis.shared.servers.providers.ServerTypeProvider;
import systems.dennis.shared.repository.DefaultSearchSpecification;
import systems.dennis.shared.repository.QueryCase;

import systems.dennis.shared.servers.repository.ServerConfigRepo;
import systems.dennis.shared.service.PaginationService;

import java.util.Objects;

@Service
@DataRetrieverDescription(repo = ServerConfigRepo.class, form = ServerConfigForm.class, model = ServerConfig.class)
public class ServerConfigService extends PaginationService<ServerConfig> {
    public ServerConfigService(WebContext holder) {
        super(holder);
    }

    public void checkActiveTypeExists(ServerConfigForm element) {

        if (element.getActive() == null || !element.getActive()) {
            return;
        }

        if (Objects.equals(element.getType(), ServerTypeProvider.OTHER)) {
            return;
        }

        ServerConfigRepo repo = getRepository();

        if (element.getId() == null) {
            if (repo.existsByTypeAndActiveIsTrue(element.getType())) {
                throw new ServerException("Already_exists");
            }
        } else {
            if (repo.existsByTypeAndActiveIsTrueAndIdNot(element.getType(), element.getId())) {
                throw new ServerException("Already_exists");
            }
        }

    }

    /**
     * Find first server withing the config type
     *
     * @param type   - a config type to look for types are : {@link ServerTypeProvider}
     * @param silent returns null of true, if false on null throws ItemNotFound exception
     * @return according to the #silent parameter returns server
     */

    public ServerConfig findByType(Long type, boolean silent) {
        DefaultSearchSpecification<ServerConfig> specification = QueryCase.equalsOf("type", type)
                .specification().addCase(QueryCase.equalsOf("active", true));

        var res = getBean(ServerConfigRepo.class).findFirst(specification).orElse(null);

        if (silent && res == null) {
            return res;
        }

        if (res == null) throw new ItemNotFoundException(type);

        return res;
    }
}
