package systems.dennis.shared.servers.service;

import org.springframework.stereotype.Service;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.postgres.model.BaseEntity;
import systems.dennis.shared.postgres.service.PaginationService;

import systems.dennis.shared.servers.exception.ServerException;
import systems.dennis.shared.servers.form.ServerConfigForm;
import systems.dennis.shared.servers.model.ServerConfig;
import systems.dennis.shared.servers.providers.ServerTypeProvider;
import systems.dennis.shared.servers.repository.ServerConfigRepo;

import java.util.Objects;

@Service
@DataRetrieverDescription(repo = ServerConfigRepo.class, form = ServerConfigForm.class, model = ServerConfig.class)
public class ServerConfigService extends PaginationService<ServerConfig> {
    public ServerConfigService(WebContext holder) {
        super(holder);
    }

    public void checkActiveTypeExists(ServerConfigForm element) {

        if (element.getActive() == null || !element.getActive()) {
            return;
        }

        if (Objects.equals(element.getType(), ServerTypeProvider.OTHER)) {
            return;
        }

        if (element.getId() == null) {
            if (count(getFilterImpl().eq("type", element.getType())
                    .and(getFilterImpl().eq("active", true)))> 0){
                throw new ServerException("server_already_exists");
            }
        } else {
            if (count(getFilterImpl().eq("type", element.getType()).and(getFilterImpl().eq("active", true))
                    .and(getFilterImpl().notEq(BaseEntity.ID_FIELD, element.getId())))> 0) {
                throw new ServerException("server_already_exists");
            }
        }

    }

    /**
     * Find first server withing the config type
     *
     * @param type   - a config type to look for types are : {@link ServerTypeProvider}
     * @param silent returns null of true, if false on null throws ItemNotFound exception
     * @return according to the #silent parameter returns server
     */

    public ServerConfig findByType(Long type, boolean silent) {
        var filter = getFilterImpl().eq("type", type).and(getFilterImpl().eq("active", true));


        ServerConfig res =getRepository().filteredFirst(filter).orElse(null);

        if (silent && res == null) {
            return res;
        }

        if (res == null) throw ItemNotFoundException.fromId(type);

        return res;
    }
}
